#include "mbed.h"
#include "si5351a.h"
#include "PinDetect.h"
#include "DebounceIn.h"

#define MIN_FREQ 20000000
#define MAX_FREQ 120000000
#define STEP_FREQ 100000
#define RESET_FREQ 35000000
   
// Interrupt on digital pushbutton
PinDetect buttonReset(p13);
PinDetect buttonData(p11);
PinDetect buttonIref(p12);
PinDetect buttonStep(p23);

    
// periodic interrupt routines
Ticker data_out;            
Ticker clock_out;
    
// data, clk, iref switch output
DigitalOut data(p15);
DigitalOut clk(p14);    
DigitalOut iref_sw(p16);

//Encoder
DigitalIn A(p22);
DigitalIn B(p21);


// Leds
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
 

// setup terminal link
Serial pc(USBTX,USBRX); 
    
// Comparateur Threshold
AnalogOut Vth(p18);
    
// communication with Si5351A I2C_SDA=p9 & I2C_SCL=p10
I2C i2c(p9, p10);       
    
// Base clock = 25MHz
SI5351A ckref(i2c, 25000000UL,SI5351_CRYSTAL_LOAD_8PF,SI5351_CLK_DRIVE_STRENGTH_8MA);
    
// define local file system
LocalFileSystem local("local"); 

//Interruption function
void toggle(void);
void initialisation(void);
void stepChange(void);
void reset(void);


//read the data from the .txt file
uint16_t read_data (int);

//convert decemal to binary
uint16_t * set_dec2bin (uint16_t);

//send the data to programm the divider
void send_data (void);

//send the clock to programm the divider
void send_clk (void);

// turn all the leds  off
void allLedsOff(void);

// turn all the leds  on
void allLedsOn(void);

//Variabeles
int divider, mode, resetFreq(RESET_FREQ), stepFreq(STEP_FREQ);
uint16_t *div_bin; // pointer
int count_data,count_clk,counter(0),stepCount(0),thresholdVoltage(0);
int aLastState(0);

int main (){
    
    //Config the interrupt button with the debouncing
    buttonReset.mode(PullUp);
    wait(0.01);
    buttonReset.attach_asserted(&reset);
    
    buttonStep.mode(PullUp);
    wait(0.01);
    buttonStep.attach_asserted(&stepChange);
    
    buttonData.mode(PullUp);
    wait(0.01);
    buttonData.attach_asserted(&toggle);
    
    buttonIref.mode(PullUp);
    wait(0.01);
    buttonIref.attach_asserted(&initialisation);
    
    buttonReset.setSampleFrequency();
    buttonIref.setSampleFrequency();
    buttonData.setSampleFrequency();
    buttonStep.setSampleFrequency();
    
    //Init the variables
    ckref.set_frequency(SI5351_CLK0, RESET_FREQ);   // CLK0=48MHz
    data       = 0;
    clk        = 0;
    aLastState = A;
    
    //Getting the data from the file
    divider          = read_data(0);
    div_bin          = set_dec2bin(divider);
    mode             = read_data(1);
    resetFreq        = read_data(2)*1000000;
    stepFreq         = read_data(3);
    counter          = resetFreq;
    thresholdVoltage = read_data(4);
    Vth              = ((double)thresholdVoltage/1000) * 0.3f;
    // Loop
    while(1){
        if (A != aLastState){
            if(B != A)
                counter = counter + stepFreq;
            else
                counter = counter - stepFreq;
        }
        aLastState = A;
        if(stepCount == 0){
            allLedsOff();
            led1 = 1;
        }
        else if (stepCount == 1){
            allLedsOff();
            led2 = 1;
        }
        else if (stepCount == 2){
            allLedsOff();
            led3 = 1;
        }
        else if (stepCount == 3){
            allLedsOff();
            led4 = 1;
        }        
        ckref.set_frequency(SI5351_CLK0, counter);
        wait(0.2);
        printf("frequence : %d threshold voltage: %1.2f volts\n\r", counter, Vth.read() * 3.3f);        
    }     
}

void allLedsOff(){
    led1      = 0;
    led2      = 0;
    led3      = 0;
    led4      = 0;
}

void allLedsOn(){
    led1      = 1;
    led2      = 1;
    led3      = 1;
    led4      = 1;
}

//Interrupt fonction
void reset(){
    counter  = resetFreq;
    allLedsOn();
    wait(0.5);
    allLedsOff();
    wait(0.2);
    allLedsOn();
    wait(0.5);
    allLedsOff();    
}

void initialisation(){
    iref_sw   = 1;
    allLedsOn();
    wait(5);
    iref_sw   = 0;
    allLedsOff();
}
    
void toggle() {
    data_out.attach(&send_data, 0.01); 
    wait(0.005);
    clock_out.attach(&send_clk, 0.005);
    count_data  = 0;
    count_clk   = 0;
    allLedsOff();
    led1        = 1;
    wait(0.2);
    led1        = 0;
    led2        = 1;
    wait(0.2);
    led2        = 0;
    led3        = 1;
    wait(0.2);
    led3        = 0;
    led4        = 1;
    wait(0.2);
    led4        = 0;
}   

void stepChange(){
    if(stepCount == 3)
        stepCount = 0;
    else
        stepCount++;
    if(stepCount == 0){
        allLedsOff();
        stepFreq = stepFreq*1000;
        led1 = 1;
    }
    else if (stepCount == 1){
        allLedsOff();
        stepFreq = stepFreq*10000;
        led2 = 1;
    }
    else if (stepCount == 2){
        allLedsOff();
        stepFreq = stepFreq*100000;
        led3 = 1;
    }
    else{
        allLedsOff();
        stepFreq = stepFreq*1000000;
        led4 = 1;
    }      
}


void send_data (void){    
    if(count_data<12){
        if(mode==0){
            data= *(div_bin+count_data);
            pc.printf("%d",*(div_bin+count_data));
            count_data=count_data+1;
        }
        if(mode==1) {
            data= *(div_bin+11-count_data);
            pc.printf("%d",*(div_bin+11-count_data));
            count_data=count_data+1;
        }
    }
    else
        data=0;
}
    
void send_clk (void){
    if(count_clk<24) {
        clk=!clk;
        count_clk=count_clk+1;
    }
    else {
        clk=0;;
    }        
}  

uint16_t read_data (int  mode) 
{              
    int value;
    int i,j;
    char text[10][10]; // when the file exist and have data
    char line[10];     // it reads line by line

    for(i=0; i<10; i++)
        for(j=0; j<10; j++)
            text[i][j] = '\0';  // initializing the vector with null
    for(i=0; i<10; i++)
        line[i] = '\0';
               
    FILE* fp = fopen ("/local/div_prog.txt","r"); // open file for reading               
        if (fp!=NULL){
            fseek (fp, 0, SEEK_END);
            int  size = ftell(fp);
            if (size==0){
                pc.printf("File is empty\n\r");
            }
            else{
                rewind (fp); // restore pointer to the begin of the file
                i=0;
                while ( fgets ( line, sizeof line, fp ) != NULL ){ /* read a line */
                    strcpy(text[i], line);
                    printf("array ----> %s", text[i]);
                    i++;
                }
                fclose(fp); // close file
                if(mode==0){// Get the divider value
                    value=atoi(text[mode]);
                        pc.printf("Divider value = %d \n\r",value); // display read data value
                }
                if(mode==1){// Get the Mode (LSB/MSB)
                    value=atoi(text[mode]);
                    if (value==0)
                        pc.printf("LSB Mode \n"); //
                    if (value==1)
                        pc.printf("MSB Mode \n"); //        
                }
                if(mode==2){// Get the Reset frequency in MHz
                    value=atoi(text[mode]);
                    pc.printf("Reset Frequency = %d \n\r", value);    
                }
                if(mode==3){// Get the frequency step
                    value=atoi(text[mode]);
                    pc.printf("Frequency Step = %d \n\r", value);    
                }
                if(mode==4){
                    value=atoi(text[mode]);
                    pc.printf("Threshold voltage = %d \n\r",value);   
                }                             
            }
        }
                
        else{
            int num;    
            pc.printf("File does no exist \n\r");
            pc.printf("Enter the divider value \n: \r");
            pc.printf("Byte to send LSB(0) or MSB(1)\n: \r");
            for(i=0; i<2; i++){   //just read to save to lines
                pc.scanf("%s\r", text[i]);
                while (!(sscanf(text[i], "%d\r", &num))){
                    printf("Invalid input '%s'\n\r", text[i]);
                    pc.scanf("%s\r", text[i]);
                }     
            }                
                   
        }
        return value;
}
    
uint16_t * set_dec2bin (uint16_t read_div) {
    int i;
    static uint16_t  r[12];
    for( i = 0; i < 12; i++ ){
        r[i]= read_div >> i & 1;
        pc.printf("%d",r[i]); // display read data value
   }
     pc.printf("\n\r"); 
     return r;
}
