#ifndef MBED_PULSEMANAGER_H
#define MBED_PULSEMANAGER_H

#include "mbed.h"

class PulseManager   {
    public:
        /** 
        * Create a PulseManager object connected to the specified pin
        * @param triggerPin output pin to use for generating a trigger signal
        * @param echoPin input pin to use for measuring an echo signal
        */
        PulseManager(PinName triggerPin, PinName echoPin);
        
        /**
        * Destructor to free allocated memory
        */
        ~PulseManager();
        
        /** 
        * Set the value of the trigger pin
        * @param val Value to set, 0 for LOW, otherwise HIGH
        */
        void write(int val);
        
        /** 
        * Send a pulse of a given value for a specified time
        * @param val Value to set, 0 for LOW, otherwise HIGH
        * @param time Length of pulse in microseconds
        */
        void write_us(int val, int time);
        
        /** 
        * Return the length of the next HIGH pulse in microsconds
        */
        int read_high_us();
        
        /** 
        * Return the length of the next HIGH pulse in microseconds or -1 if longer than timeout
        * @param timeout Time before pulse reading aborts and returns -1, in microseconds
        */
        int read_high_us(int timeout);
        
        /** 
        * Return the length of the next LOW pulse in microsconds
        */
        int read_low_us();
        
        /** 
        * Return the length of the next LOW pulse in microseconds or -1 if longer than timeout
        * @param timeout Time before pulse reading aborts and returns -1, in microseconds
        */
        int read_low_us(int timeout);
        
        /** 
        * Return the length of the next pulse in microsconds
        */
        int read_us();
        
        /** 
        * Return the length of the next pulse in microseconds or -1 if longer than timeout
        * @param timeout Time before pulse reading aborts and returns -1, in microseconds
        */
        int read_us(int timeout);
        
    private:
    
        /** Start value for echo pin to use for detecting a change */
        int startval;
        /** Timer for measuring pulse width */
        Timer pulsetime;
        /** Timer for detecting a timeout */
        Timer runtime;
        /** Pin handle for trigger pin */
        DigitalOut trigger;
        /** Pin handle for echo pin */
        DigitalIn  echo;
};

#endif