#include "mbed.h"
#include "GPS.h"
#include "gprs.h"
#include "BufferedSoftSerial.h"
#include "gprs.h"
Serial pc(USBTX, USBRX);
GPS uGPS(p13, p14);
Serial uBP_RN41(p9,p10);
BufferedSoftSerial uBP(p19,p18);
GPRS uGSM(p28,p27,9600,"66426640");

//Power Management Variables
DigitalOut pBP(p5); //For toggling the BP power
DigitalOut BPbutton(p25);//For instigating BP measurement
DigitalOut pRN41(p6);//For toggling the RN41's power.
DigitalOut pGSM(p7);//For toggling the GSM's power.
DigitalOut GSM_pwk(p21); //For using the GSM power key (doesn't turn off Power LED, waste of power).
DigitalOut pGPS(p8); //For toggling the GPS's power.
DigitalOut Buzzer(p23);
DigitalOut LED(p24);

//GPS Variables
double lat, hlat=25.289536; //default longitude and latitude to Hamad Medical Corporation
double lon, hlon=51.522364; 
GPS_Time t; //Structure has: hour, minute, second, day, month, year in it

//BP Variables
char BP[15]={0}; //Buffer for Blood Pressure Reading
int SYS=120,DIA=80;
int hSYS=0, hDIA=0; //History of Systolic and Diastolic

//RN41-Nonin Variables
char FirstBit=129,FalsePulse=5,SpO2=99,LastBit=100;
int HR=70;

//Power Management Functions
void ToggleBP(bool state);
void ToggleRN41(bool state);
void ToggleGSM(bool state);
void ToggleGPS(bool state);

//GPS Functions
void UpdateGPS();
bool CheckTime();

//BP Functions
void InstigateReading();
void UpdateBP();

//RN41-Nonin Functions
void initializeRN41();
void UpdateNonin(); //Function to update the measurements

//QEWS variables & function
int HRScore,SPScore,BPScore,QEWS_Score;
void QEWS();

//GSM Variables
char message[160]; //Length of SMS message is 160.
int msglength=0;
void FormatData();
void SendMessageGSM();

//MAIN
int main(){
// Define Baudrates
    pBP=0; //For toggling the BP power
    BPbutton=1;//For instigating BP measurement
    pRN41=0;//For toggling the RN41's power.
    pGSM=0;//For toggling the GSM's power.
    GSM_pwk=0; //For using the GSM power key (doesn't turn off Power LED, waste of power).
    pGPS=0; //For toggling the GPS's power.
    uBP_RN41.baud(9600);
    while(1){
       wait(3);
       UpdateGPS();
       while(!CheckTime());
       UpdateNonin();
       UpdateBP();
       QEWS();
       FormatData();
       if(QEWS_Score>=1)
       SendMessageGSM();
    }
    
}

// Power Functions
void ToggleBP(bool state){
    if(state){
        pRN41=0; wait(2);
        pBP=1;  wait(2);
    }
    if(!state){
        pRN41=0; wait(2);
        pBP=0; wait(2);
    }
}

void ToggleRN41(bool state){
    if(state){
        pBP=0; wait(2); 
        pRN41=1; wait(2);
    }
    if(!state){
        pRN41=0; wait(2);
        pBP=0; wait(2);
    }
}

void ToggleGSM(bool state){
    if(state){
        pGSM=1;
        wait(2);
            GSM_pwk=0; wait(1);
            GSM_pwk=1; 
        wait(10);
    }
    if(!state){
        GSM_pwk=0; wait(1);
        GSM_pwk=1; 
        wait(5);
        pGSM=0;
    }
}

void ToggleGPS(bool state){
    if(state){
        pGPS=1;
        wait(5); //Waiting for 2 seconds. But is it enough to let the GPS module acquire GPS?
    }
    if(!state){
        pGPS=0;
        wait(2);
    }
}

//GPS Functions
void UpdateGPS(){
    ToggleGPS(1);
    //Keep updating over the course of 20 seconds, to get a sure location.
    
    for(int i=0; i<20; i++){    
        lat=uGPS.latitude();
        lon=uGPS.longitude();
        uGPS.timeNow(&t);
        pc.printf("before history lat:%f, lon:%f \n\r",lat,lon);
        pc.printf("Time: %d, %d, %d \n\r", t.hour, t.minute,t.second);
        wait(10);
    }
    pGPS=0;
    if( (lat <=0) && (lon <= 0)){ // if there is an error in the location (i.e. not in Qatar) then it will use the last known location
            lat=hlat; // take the last known location.
            lon=hlon;
        }
        pc.printf("after history lat:%f, lon:%f \n\r",lat,lon);
    
   hlat=lat;
   hlon=lon;
   ToggleGPS(0);
}

//RN41 Functions
void UpdateNonin(){
    ToggleRN41(1);
    wait(10);
    if(uBP_RN41.readable()){
        FirstBit =uBP_RN41.getc(); wait(0.5);
        FalsePulse=uBP_RN41.getc(); wait(0.5);
        HR= FalsePulse | ((FirstBit&=0x03)<<7); wait(0.5);
        SpO2=uBP_RN41.getc(); wait(0.5);
        LastBit=uBP_RN41.getc(); wait(0.5);
        pc.printf("SPO2: %d %d %d %d %d\n\r", FirstBit,FalsePulse,HR,SpO2,LastBit);
    }
    ToggleRN41(0);
}

//BP Functions
void InstigateReading(){
    wait(5);
    BPbutton=0; wait(1);
    BPbutton=1; wait(2);
    wait(70);
    BPbutton=0;wait(1);
    BPbutton=1;
}

void UpdateBP(){
    
    ToggleBP(1);
    uBP.baud(9600);
    SYS=0; DIA=0;
    for(int i=0;i<3;i++){//Beep 3 times to warn the patient that the device will start taking measurement
        Buzzer=1;LED=1;wait(0.5);
        Buzzer=0;LED=0;wait(0.5);
    }
    LED=1;
    InstigateReading();
    while(1){
        if(uBP.readable()){
            for(int i=0; i<15; i++)
                BP[i]=uBP.getc(); //Acquire 15 bytes
        }
        if(BP[14]==0x0A){ //Transmission is okay
            //      Hundreds            Tens                Units
            SYS=((BP[1]-0x30)*100)+ ((BP[2]-0x030)*10) + ((BP[3]-0x30)*1);
            DIA=((BP[6]-0x30)*100)+ ((BP[7]-0x030)*10) + ((BP[8]-0x30)*1);
            pc.printf("BP Before History: %d %d\n\r",SYS,DIA);
            hSYS=SYS; hDIA=DIA; //Recording values, for history.
            pc.printf("BP After History: %d %d\n\r",SYS,DIA);
            break;
        }
    }
    //If values are unchanged, use the last stored in history
    if(SYS==0 && DIA==0){
        SYS=hSYS; DIA=hDIA;
    }
    LED=0;
    ToggleBP(0);
}

void QEWS(){
//Calculating Oxygen Saturation Score
                                (SYS >= 110 && SYS <= 179) ? BPScore=0: BPScore=BPScore;
    ((SYS >= 90 && SYS <=109)||(SYS >= 180 && SYS <= 199)) ? BPScore=1: BPScore=BPScore;
                     ((SYS >= 71 && SYS <=89)||SYS >= 200) ? BPScore=2: BPScore=BPScore;
                                                (SYS < 70) ? BPScore=3: BPScore=BPScore;
//Calculating Oxygen Saturation Score
                (SpO2>=95) ? SPScore=0: SPScore=SPScore;
    (SpO2>=93 && SpO2<=95) ? SPScore=1: SPScore=SPScore;
    (SpO2>=90 && SpO2<=92) ? SPScore=2: SPScore=SPScore;
                (SpO2<=90) ? SPScore=3: SPScore=SPScore;
//Calculating Heart Rate Score
                          ((HR >= 51) && (HR <= 100)) ? HRScore=0: HRScore=HRScore;
    ((HR >= 41 && HR <=50)||(HR >= 101 && HR <= 110)) ? HRScore=1: HRScore=HRScore;
    ((HR >= 31 && HR <=40)||(HR >= 111 && HR <= 129)) ? HRScore=2: HRScore=HRScore;
                            ((HR < 31 )|| (HR > 129)) ? HRScore=3: HRScore=HRScore;
    QEWS_Score=HRScore+SPScore+BPScore;
}

//GSM Functions
void FormatData(){
/* This message has a total of 60 characters */
    msglength=sprintf(message,"SYS: %d\n\rDIA: %d\n\rHR: %d\n\rSpO2: %d\n\rLAT: %f\n\rLON: %f\n\rQEWS: %d\n\rTIME: %d:%d",SYS,DIA,HR,SpO2,lat,lon,QEWS_Score,t.hour,t.minute);
}

void SendMessageGSM(){
   ToggleGSM(1);
   wait(10);
   uGSM.sendSMS("66426640",message);
   wait(15);
   ToggleGSM(0);
}

//CONFIGURATION FUNCTIONS
void initializeRN41(){
    ToggleRN41(1); //Turn RN41 on.
    uBP_RN41.printf("$$$"); wait(0.5); // Entering Command Mode
    uBP_RN41.printf("SR,001C0500D414\r"); wait(0.5); // Setting MAC Address
    uBP_RN41.printf("SP,681010\r"); wait(0.5); //Setting Pass-key
    uBP_RN41.printf("SU,96\r"); wait(0.5); //Setting Baudrate to 9600
    uBP_RN41.printf("SM,6\r"); wait(0.5); //Setting Mode to Pairing Mode (6)
    uBP_RN41.printf("R,1\r"); wait(0.5); //Rebooting
}

bool CheckTime(){
    UpdateGPS();
    if(t.hour==0 && t.minute==0)
        return 1;
    if(t.hour==2 && t.minute==0)
        return 1;
    if(t.hour==4 && t.minute==0)
        return 1;
    if(t.hour==6 && t.minute==0)
        return 1;
    if(t.hour==7 && t.minute==4)
        return 1;
    if(t.hour==8 && t.minute==8)
        return 1;
    if(t.hour==9 && t.minute==12)
        return 1;
    if(t.hour==10 && t.minute==16)
        return 1;
    if(t.hour==11 && t.minute==20)
        return 1;
    if(t.hour==12 && t.minute==24)
        return 1;
    if(t.hour==13 && t.minute==28)
        return 1;
    if(t.hour==14 && t.minute==32)
        return 1;
    if(t.hour==15 && t.minute==36)
        return 1;
    if(t.hour==16 && t.minute==40)
        return 1;
    if(t.hour==17 && t.minute==44)
        return 1;
    if(t.hour==18 && t.minute==48)
        return 1;
    if(t.hour==19 && t.minute==52)
        return 1;
    if(t.hour==20 && t.minute==56)
        return 1;
    if(t.hour==22 && t.minute==0)
        return 1;
    else
        return 0;
}