#ifndef MINERENGINE_H
#define MINERENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Claw.h"
#include "Winch.h"
#include "Monster.h"
#include "Gold.h"

// gap from edge of screen
#define GAP 2

/** MinerEngine Class
@brief Class for the structure of the game
@brief It includes the control input, menu and other display parts(welcome,select,gameover,etc)
@author Yufan Zhong, University of Leeds and SWJTU
@date May 2020
*/ 

class MinerEngine
{

public:
    /** Constructor */
    MinerEngine();
    
    /** Destructor */
    ~MinerEngine();
    
    /** Initialise the components of the MinerEngine
    *   @param  winch_width - the width of the winch
    *   @param  winch_height - the height of the winch
    *   @param  gold_num - the gold number
    *   @param  monster_speed - the speed of monster
    */
    void init(int winch_width,int winch_height,int gold_num,int monster_speed);
    
    /** Initialise the unchanged parameter
    */
    void init_unchanged_parameter();
    
    /** Switch the states
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void state_switch(Gamepad &pad, N5110 &lcd);
    
     /** Get the select
    *   @param  pad - Gamepad library
    */
    int get_select(Gamepad &pad);
    
     /** Print the instruction
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void instruction(Gamepad &pad, N5110 &lcd);
    
     /** Print the menu
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void menu(Gamepad &pad, N5110 &lcd);
    
     /** Select the difficulties
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void options(Gamepad &pad, N5110 &lcd);
    
     /** Run the game
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void game_run(Gamepad &pad, N5110 &lcd);
    
     /** Draw all the components
    *   @param  lcd - N5110 library
    */
    void draw(N5110 &lcd);
    
     /** Read the gamepad input
    *   @param  pad - Gamepad library
    */
    void read_input(Gamepad &pad);
    
     /** Update the data
    *   @param  pad - Gamepad library
    */
    void update(Gamepad &pad);
    
     /** Print welcome component and wait for start
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void welcome(Gamepad &pad, N5110 &lcd);
    
     /** Print game over on the screen
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void game_over(Gamepad &pad, N5110 &lcd);
    
     /** Play the music
    *   @param  pad - Gamepad library
    */
    void play_music(Gamepad &pad);
    
     /** Roll up the picture
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void screen_rollup(Gamepad &pad, N5110 &lcd);
    
     /** Print loading on the screen
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void loading(Gamepad &pad, N5110 &lcd);
    
     /** Play the tunnel animation on the screen
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void tunnel(Gamepad &pad, N5110 &lcd);
    
     /** Draw the tunnel
    *   @param  lcd - N5110 library
    */
    void draw_tunnel(N5110 &lcd);
    
     /** Draw the walking man
    *   @param  lcd - N5110 library
    */
    void draw_man(N5110 &lcd);
    
     /** Print message in the tunnel
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void tunnel_print(Gamepad &pad, N5110 &lcd);
    
    
private:

     /** Check if the gold has been lifted
    *   @param  pad - Gamepad library
    */
    void check_gold_collision(Gamepad &pad);
    
     /** Check if the monster collides with claw
    *   @param  pad - Gamepad library
    */
    void check_monster_collision(Gamepad &pad);
    
     /** Check if the claw gets the gold or the bottom
    *   @param  pad - Gamepad library
    */
    void check_claw_collision(Gamepad &pad);
    
     /** Check if gold reloading is needed
    */
    void check_gold_reload();
    
     /** Print the score on the screen 
    *   @param  lcd - N5110 library
    */
    void print_scores(N5110 &lcd);
  
    Winch _winch;
    Claw _claw;
    Gold _gold;
    Monster _monster;
    
    int _winch_width;
    int _winch_height;
    int _gold_num;
    float _monster_speed;
    int _s;
    int _state;
    int _catching;
    int _claw_get;
    int _claw_speed;
    int _monster_collision;
    int _now_score;
    int _highest_score;
    int _gold_reached_num[9];
    int _h;
    int _i;
    int _n;
    int _z;
    int _j;
    
    Direction _d;
    float _mag;
    
};

#endif