/**
@file main.cpp
@brief Program implementation
*/

#include "main.h"

int main()
{
    lcd.init();                     //initialize the lcd
    initArray();                    //fill array with sine sample
    button.rise(&button_isr);       //will read the rise edge when button is pressed
    button.mode(PullDown);          //external button is set as PullDown

    while(1)    {

        intro();                            //state 0 - introduction interface
        firstView();                        //state 1 - main menu interface
        gameView();                         //state 2 - game interface
        optionView();                       //state 3 - option interface
        instructionView();                  //state 4 - how to play interface
        brightnessView();                   //state 5 - set up brightness interface
        finishView(g_score);                //state 6 - final score interface
    }
}

void intro()                                                        //state 0 - introduction interface
{
    if(state==0)    {

        for(int k=90; k>5; k--)   {
            lcd.printString("KAMIKAZE",k,1);                        //display game title in left moving motion
            lcd.printString("RACE",k+12,2);
            lcd.refresh();
            wait(0.1);
            lcd.clear();

            if(g_button_flag==1)    {
                g_button_flag=0;
                state=1;
                break;
            }
        }
        state=1;
    }
}

void firstView()                                                    //state 1 - main menu interface
{
    if(state==1)    {
        pointer_x=20;
        pointer_y=31;

        while(loop==0) {
            sleepTimer.start(); // start timer

            if(sleepTimer.read()<15)    {                           //display all the main menu interface
                lcd.printString("KAMIKAZE",5,1);
                lcd.printString("RACE",17,2);
                lcd.drawLine(58,46,58,2,2);
                lcd.drawLine(67,46,67,2,2);
                lcd.drawLine(76,46,76,2,2);
                lcd.drawCar(60,10,1);
                lcd.drawCar(60,25,1);
                lcd.drawCar(69,23,1);
                lcd.printString("Start",14,4);
                lcd.printString("Option",14,5);
                firstViewPointer();
                lcd.drawCar(5,pointer_y,1);
                lcd.refresh();
                wait(0.1);
                lcd.clear();

                if(g_button_flag==1)    {
                    g_button_flag=0;
                    loop=1;

                    if(pointer_y==31)   {
                        state=2;    //go to game state
                    } else {
                        state=3;    //go to option state
                    }
                }
            } else {                                                //go to sleep mode after 15 second has passed
                sleepTimer.stop();  // stop the timer
                sleepTimer.reset(); // reset to 0
                SleepView();
            }
        }
        loop=0;
        sleepTimer.stop();  // stop the timer
        sleepTimer.reset(); // reset to 0
    }
}

void gameView()                                                     //state 2 - game interface
{
    if(state==2)    {

        initialGameInterface();                                     //function to initialize the game with countdown timer.
        initGameVar();                                              //function to initiate all variable used in the game

        while(g_crash==0)    {                                      //0 - crash not occur

            for(int i=1; i<=g_level*20; i++)    {
                carValue[i]=9*((rand()%6)+1)-5;                                     //generate random position for all vehicle on the road
            }

            for(int y_move=-5; y_move<100000; y_move++)  {

                if(game==PLAY) {                                                 //when game is played
                    crash(carValue,y_move,g_score,g_level,car_x,car_y,delay);    //function to check whether crash occur
                    playermovement(g_level);                                     //function to control the player's car movement
                    lcd.refresh();
                    checkLastCar(g_level);                                      //function to check when will the last last car go out from the screen to finish level

                    if((y_move-(lastCar))>40)   {
                        finishLevel(g_level);                                   //finish level when last car has passed through
                        break;
                    }
                    if(g_crash==1)    {
                        crashsound();                               //play sound when crush happened
                        wait(2);                                    //delay to give time for new game

                        if(g_liveleft<=1)   {                       //check if live is zero after crash occured
                            lcd.clear();                            // if zero, game stop
                            break;
                        } else    {                                                 //if still has life
                            g_crash=0;                                              //continue the game
                            g_liveleft=g_liveleft-1;                                //reduce the life
                            g_level=g_level-1;                                      //start at same level when crash occur
                            car_x=22;                                               //return the car back to initial position
                            car_y=38;
                            break;
                        }
                    }
                    lcd.clear();
                    g_score=g_score+1;                                  //increase the score by 1 after every loop
                } else {                                                //when game is paused
                    y_move=y_move-1;                                    //fix y_move to make the vehicle static
                    game_interface(g_score,g_level,g_liveleft);
                    lcd.printString("Paused",15,2);                     //display pause on the screen
                    lcd.refresh();
                }
                if (g_button_flag==1) {                                 //press button to paused/play the game.
                    g_button_flag=0;
                    pausesound();
                    wait(0.1);
                    if(game == PLAY) {
                        game = PAUSE;
                    } else {
                        game=PLAY;
                    }
                }
            }
            g_level=g_level+1;
        }                                                               //when the game finished, move to finish View(display high score)
        state=6;
        finishView(g_score);
        state=0;                                                        //goes back to main menu
    }
}

void optionView()                                                       //state 3 - option interface
{
    if(state==3)    {
        pointer_x=5;
        pointer_y=7;

        while(loop==0)    {
            sleepTimer.start(); // start timer

            if(sleepTimer.read()<15)    {
                lcd.printString("Instruction",13,1);
                lcd.printString("Brightness",13,2);
                lcd.printString("Achievement",13,3);
                lcd.printString("Back",13,4);
                optionPointer();
                lcd.drawCar(4,pointer_y,1);
                lcd.refresh();
                wait(0.1);
                lcd.clear();

                if(g_button_flag==1)    {
                    g_button_flag=0;
                    loop=1;

                    if(pointer_y==7)   {
                        state=4;//go to instruction
                    } else if(pointer_y==15)   {
                        state=5;//go to brightness
                    } else if(pointer_y==31)   {
                        state=1;//go back
                    } else { //pointer====23
                        state=6;//go to achievement
                    }
                }
            } else {
                sleepTimer.stop();  // stop the timer
                sleepTimer.reset(); // reset to 0
                SleepView();
            }
        }
        loop=0;
        sleepTimer.stop();  // stop the timer
        sleepTimer.reset(); // reset to 0
    }
}

void instructionView()                                                  //state 4 - how to play interface
{
    if(state==4)    {

        while(loop==0)    {
            sleepTimer.start(); // start timer

            if(sleepTimer.read()<15)    {
                lcd.printString("Make your way",0,0);                   //explain how to play the game
                lcd.printString("through the",0,1);
                lcd.printString("traffic by ",0,2);
                lcd.printString("avoiding all",0,3);
                lcd.printString("vehicles on",0,4);
                lcd.printString("the highway.",0,5);
                lcd.refresh();
                wait(0.1);
                lcd.clear();

                if(g_button_flag==1)    {
                    g_button_flag=0;
                    loop=1;
                    state=9;
                }
            } else {
                sleepTimer.stop();  // stop the timer
                sleepTimer.reset(); // reset to 0
                SleepView();
            }
        }
        loop=0;
        sleepTimer.stop();  // stop the timer
        sleepTimer.reset(); // reset to 0
    }
    if(state==9)    {

        while(loop==0)    {
            sleepTimer.start(); // start timer

            if(sleepTimer.read()<15)    {
                lcd.printString("You have ",0,0);
                lcd.printString("three lives",0,1);
                lcd.printString("at the ",0,2);
                lcd.printString("start of",0,3);
                lcd.printString("the game.",0,4);
                lcd.printString("Good Luck!!!",0,5);
                lcd.refresh();
                wait(0.1);
                lcd.clear();

                if(g_button_flag==1)    {
                    g_button_flag=0;
                    loop=1;
                    state=3;
                }
            } else {
                sleepTimer.stop();  // stop the timer
                sleepTimer.reset(); // reset to 0
                SleepView();
            }
        }
        loop=0;
        sleepTimer.stop();  // stop the timer
        sleepTimer.reset(); // reset to 0
    }
}

void brightnessView()                                                //state 5 - set up brightness interface
{
    if(state==5)    {
        brightness_adjuster=5;

        while(loop==0)    {
            sleepTimer.start(); // start timer

            if(sleepTimer.read()<15)    {
                brightnessController();
                lcd.printString("Adjust",25,1);
                lcd.printString("Brightness",13,2);
                lcd.printString("+",78,4);
                lcd.printString("-",2,4);
                lcd.setBrightness(brightness_adjuster/10);
                lcd.drawRect(1,27,brightness_adjuster*8,4,1);
                lcd.drawRect(1,27,81,4,0);
                lcd.refresh();
                wait(0.1);
                lcd.clear();

                if(g_button_flag==1)    {
                    g_button_flag=0;
                    loop=1;
                    state=3;
                }
            } else {
                sleepTimer.stop();  // stop the timer
                sleepTimer.reset(); // reset to 0
                SleepView();
            }
        }
        loop=0;
        sleepTimer.stop();  // stop the timer
        sleepTimer.reset(); // reset to 0
    }
}

void finishView(int g_score)                                        //state 6 - final score interface
{
    if(state==6)    {

        while(loop==0)    {
            sleepTimer.start(); // start timer

            if(sleepTimer.read()<15)    {
                char buffer[14];
                int length = sprintf(buffer,"%2d",g_score/20);      //display final score
                lcd.printString("Your",30,1);
                lcd.printString("Final Score",10,2);
                lcd.printString("Is",34,3);
                lcd.printString(buffer,34,4);
                lcd.refresh();
                wait(0.1);
                lcd.clear();

                if(g_button_flag==1)    {
                    g_button_flag=0;
                    loop=1;
                    state=3;
                }
            } else {
                sleepTimer.stop();  // stop the timer
                sleepTimer.reset(); // reset to 0
                SleepView();
            }
        }
        loop=0;
    }
}

void SleepView()            //this function applies the sleep mode concept
{
    //improve power effciency for the game system
    while(loop==0)  {       //expand the battery lifetime.
        lcd.turnOff();      //also turn off the lcd scrren when not in used
        sleep();

        if(g_button_flag==1)  {
            g_button_flag=0;        //if button pressed, turn on the screen back.
            lcd.init();
            loop=1;
        }
    }
    loop=0;
}



void crash(int carValue[],int opponent_y,int g_score,int g_level,int k,int l,int delay)     //function to check if crash occur
{
    float carPixel[84][48];     //create array for vehicle on the road
    float enemyPixel[84][48];

    lcd.clear();
    lcd.drawCar(k,l,1);
    lcd.refresh();

    for(int i=0; i<=54; i++) {
        for(int j=0; j<=48; j++) {
            carPixel[i][j]=lcd.getPixel(i,j);       //check player's car pixel
        }
    }
    lcd.clear();

    for(int i=1; i<=(g_level*20); i++)    {

        if (g_level>=9) {                                               //for this part, it draw different type of vehicle for given road lane.
            if (carValue[i]<10) {
                lcd.drawTruck(carValue[i],opponent_y-delay,1);          //truck is drawn on most left lane
            } else if (carValue[i]<20) {
                lcd.drawVan(carValue[i],opponent_y-delay,1);            //van is drawn second left lane
            } else if (carValue[i]<40) {
                lcd.drawCar(carValue[i],opponent_y-delay,1);            //car is drawn on lane three and four
            }  else {
                lcd.drawCarReverse(carValue[i],opponent_y-delay,1);     //car is drawn  in opposite direction for opposite lane
            }
            if(i%3==0) {
                delay=delay+20;
            }
        } else if(g_level>=5) {                                         //the step is repeat for different level
            if (carValue[i]<10) {                                       //higher the level, more car will be on the road to make it more difficult
                lcd.drawTruck(carValue[i],opponent_y-delay,1);
            } else if (carValue[i]<20) {
                lcd.drawVan(carValue[i],opponent_y-delay,1);
            } else if (carValue[i]<40) {
                lcd.drawCar(carValue[i],opponent_y-delay,1);
            } else {
                lcd.drawCarReverse(carValue[i],opponent_y-delay,1);
            }
            if(i%2==0) {
                delay=delay+20;
            }
        } else {
            if (carValue[i]<10) {
                lcd.drawTruck(carValue[i],opponent_y-delay,1);
            } else if (carValue[i]<20) {
                lcd.drawVan(carValue[i],opponent_y-delay,1);
            } else if (carValue[i]<40) {
                lcd.drawCar(carValue[i],opponent_y-delay,1);
            } else {
                lcd.drawCarReverse(carValue[i],opponent_y-delay,1);
            }
            delay=delay+20;
        }
    }
    lcd.refresh();
    for(int i=0; i<=54; i++) {
        for(int j=0; j<=48; j++)  {
            enemyPixel[i][j]=lcd.getPixel(i,j);                     //now, check the pixel for traffic's car
        }
    }
    for(int i=0; i<=54; i++) {
        for(int j=0; j<=48; j++)  {
            if(enemyPixel[i][j]!=0 && carPixel[i][j]!=0)    {       //now, check if player's car pixel overlap with traffic's car pixel
                lcd.printString("!!CRASH!!",2,2);                   //if yes, print crash
                g_crash=1;
            }
        }
    }
    game_interface(g_score,g_level,g_liveleft);                     //put all the other displaying stuff to finish off the interface
    lcd.drawCar(k,l,1);
    lcd.refresh();
    double ki=-((g_level));
    duration=(0.02*exp(ki))+0.005;                                  //use exponential decay equation to be used for speed control at increasing level

    if(opponent_y%5==0)  {
        gameplaysound(duration);
    } else {
        wait(duration);
    }
}

void game_interface(int g_score,int g_level,int g_liveleft)         //function to provide game interface during gameplay
{
    score(61,2);
    level(61,18);
    char buffer[14];
    int length = sprintf(buffer,"%2d",g_score/20);
    lcd.printString(buffer,62,1);
    int length1 = sprintf(buffer,"%2d",g_level);
    lcd.printString(buffer,62,3);
    live(g_liveleft);
    roadline(g_score);
    lcd.drawRect(59,33,g_level*2,4,1);
    lcd.drawRect(59,33,22,4,0);
}

void playermovement(int g_level)            //function to control the player's car movement
{
    if (g_level>=3)  {                      //at higher level, speed is faster, so give small scale movement to give better handling
        playerSpeed=1;
    } else if(g_level<3) {                  //at lower level, speed is slower, so high scale movemement is allowed.
        playerSpeed=2;
    }

    if(xPot>=0.7f)    {                     //joystick movement control the screen is shown here
        if(car_x>-3 && car_x<49) {
            car_x=car_x+playerSpeed;
        }
    }
    if(xPot<=0.3f)    {
        if(car_x>5 && car_x<57) {
            car_x=car_x-playerSpeed;
        }
    }
    if(yPot>0.7f)    {
        if(car_y>5 && car_y<48) {
            car_y=car_y-playerSpeed;
        }
    }
    if(yPot<0.3f)    {

        if(car_y>-3 && car_y<38) {
            car_y=car_y+playerSpeed;
        }
    }
}

void checkLastCar(int g_level)          //check for last car will be on the screen
{
    //needed in order to stop the for loop and finish the level
    if(g_level>=9)   {
        lastCar=g_level*33*4;
    } else if(g_level>=5) {
        lastCar=g_level*50*4;
    } else {
        lastCar=g_level*100*4;
    }
}

void finishLevel(int g_level)           //display the high score screen when the game is finish
{
    char buffer[48];
    int length = sprintf(buffer,"%2d",g_level);
    lcd.printString(buffer,24,3);
    lcd.printString("Finish",10,1);
    lcd.printString("Level",14,2);
    lcd.refresh();
    wait(2);
    lcd.clear();
    delay=0;
    car_x=22;
    car_y=38;
}

void initialGameInterface()                                 //provide initial countdown before the game start
{
    for(int countdown=5; countdown>=1; countdown--)  {      //countdown from 5-1
        score(61,2);
        level(61,18);
        lcd.printString("0",68,1);
        lcd.printString("1",68,3);
        lcd.drawHeart(59,40,3,1);

        lcd.drawLine(0,47,0,0,1);
        lcd.drawLine(83,47,83,0,1);
        lcd.drawLine(83,0,0,0,1);
        lcd.drawLine(83,47,0,47,1);

        lcd.drawRect(59,33,22,4,1);
        lcd.printString("Game",15,1);
        lcd.printString("Start",14,2);
        lcd.printString("In",12,3);
        char buffer[14];
        int length = sprintf(buffer,"%2d",countdown);
        lcd.printString(buffer,30,3);

        lcd.refresh();
        wait(1);
        lcd.clear();
    }
}

void initGameVar()             //initiate every variable use in the game before the game start
{
    g_crash=0;
    g_liveleft=3;
    g_level=1;
    g_score=0;
    car_x=22;
    car_y=38;
}

void firstViewPointer()         //control the pointer on main menu
{
    if(yPot>0.7f)    {
        pointer_y=31;
    }
    if(yPot<0.3f)    {
        pointer_y=39;
    }
}

void optionPointer()            //control the pointer on option menu
{
    if(yPot>0.7f)    {
        if(pointer_y>11 && pointer_y<32) {
            pointer_y=pointer_y-8;
        }
    }
    if(yPot<0.3f)    {
        if(pointer_y>5 && pointer_y<28) {
            pointer_y=pointer_y+8;
        }
    }
}

void brightnessController()         //control the brightness by using joystick
{
    if(xPot>0.7f)    {
        if(brightness_adjuster>=0 && brightness_adjuster<=9) {
            brightness_adjuster=brightness_adjuster+1;
        }
    }
    if(xPot<0.3f)    {
        if(brightness_adjuster>=1 && brightness_adjuster<=10) {
            brightness_adjuster=brightness_adjuster-1;
        }
    }
}

void initArray()
{
    // create LUT - loop through array and calculate sine wave samples
    for (int i = 0; i < n ; i++) {
        y[i] = 60 + 60*sin(i*2*PI/n);
    }

}

void tone(float frequency,float duration)
{
    green=1;        //turn on LED

    if (frequency > 0) {  // a frequency of 0 indicates no note played so only play a note if frequency is not 0

        float dt = 1.0f/(frequency*n) - (1.34e-6 + 1e-6);  // calculate time step - take into account DAC time and wait() offset

        noteTimer.start(); // start timer

        while(noteTimer.read() < duration) { // keep looping while timer less than duration

            for (int i = 0; i < n ; i++) {  // loop through samples and output analog waveform
                buzzer = y[i];
                wait(dt); // leave appropriate delay for frequency
            }
        }

        noteTimer.stop();  // stop the timer
        noteTimer.reset(); // reset to 0

    } else { // if no note played, have a simple delay
        wait(duration);
    }

    green = 0;   // turn off LEDs
}

void gameplaysound(float duration)
{
    float noteArray[] = {
        // provide sound during the time game is played
        NOTE_G3,
    };

    int notes = sizeof(noteArray)/sizeof(float);
    for(int i=0; i < notes; i++) {
        tone(noteArray[i],duration);

    }
}

void crashsound()
{
    float noteArray[] = {
        // provide sound during the time game is played
        NOTE_G5, NOTE_F6,0,NOTE_F6,NOTE_F6,NOTE_E6,0,NOTE_D6,NOTE_C6,NOTE_G5,0,NOTE_E5,NOTE_C5
    };
    float noteDuration[] = {
        4,4,4,4,4,4,4,4,4,4,4,4,4
    };
    int notes = sizeof(noteArray)/sizeof(float);
    for(int i=0; i < notes; i++) {
        tone(noteArray[i],60.0f/(BPM*noteDuration[i]));

    }
}

void pausesound()
{
    float noteArray[] = {
        // provide sound during the time game is played
        NOTE_E6,NOTE_C6,NOTE_E6,NOTE_C6,
    };
    float noteDuration[] = {
        8,8,8,8
    };
    int notes = sizeof(noteArray)/sizeof(float);
    for(int i=0; i < notes; i++) {
        tone(noteArray[i],60.0f/(BPM*noteDuration[i]));

    }
}

void finishsound()
{
    float noteArray[] = {
        // provide sound during the time game is played
        NOTE_E6,NOTE_C6,NOTE_E6,NOTE_C6,
    };
    float noteDuration[] = {
        8,8,8,8
    };
    int notes = sizeof(noteArray)/sizeof(float);
    for(int i=0; i < notes; i++) {
        tone(noteArray[i],60.0f/(BPM*noteDuration[i]));

    }
}

void live(int liveleft)         //draw how many life left on the screen
{
    lcd.drawHeart(59,40,liveleft,1);
    lcd.refresh();
}

void roadline(int moving)           //draw the road line +border line on the screen
{
    //road line
    lcd.drawLine(11,46,11,2,2);
    lcd.drawLine(20,46,20,2,2);
    lcd.drawLine(29,46,29,2,2);
    lcd.drawLine(38,45,38,2,1);
    lcd.drawLine(47,46,47,2,2);
    //border line
    lcd.drawLine(0,47,0,0,1);
    lcd.drawLine(83,47,83,0,1);
    lcd.drawLine(83,0,0,0,1);
    lcd.drawLine(83,47,0,47,1);

    if(moving%2!=0) {
        lcd.drawLine(2,46,2,2,2);
        lcd.drawLine(56,46,56,2,2);
        lcd.refresh();
    } else    {
        lcd.drawLine(2,45,2,1,2);
        lcd.drawLine(56,45,56,1,2);
        lcd.refresh();
    }
}

void score(int m,int my)            //display the score text on the screen
{
    //show score title
    lcd.setPixel(m,my);
    lcd.setPixel(m,my+1);
    lcd.setPixel(m,my+2);
    lcd.setPixel(m,my+4);
    lcd.setPixel(m+1,my);
    lcd.setPixel(m+1,my+2);
    lcd.setPixel(m+1,my+4);
    lcd.setPixel(m+2,my);
    lcd.setPixel(m+2,my+2);
    lcd.setPixel(m+2,my+3);
    lcd.setPixel(m+2,my+4);
    lcd.setPixel(m+4,my);
    lcd.setPixel(m+4,my+1);
    lcd.setPixel(m+4,my+2);
    lcd.setPixel(m+4,my+3);
    lcd.setPixel(m+4,my+4);
    lcd.setPixel(m+5,my);
    lcd.setPixel(m+5,my+4);
    lcd.setPixel(m+6,my);
    lcd.setPixel(m+6,my+4);
    lcd.setPixel(m+8,my);
    lcd.setPixel(m+8,my+1);
    lcd.setPixel(m+8,my+2);
    lcd.setPixel(m+8,my+3);
    lcd.setPixel(m+8,my+4);
    lcd.setPixel(m+9,my);
    lcd.setPixel(m+9,my+4);
    lcd.setPixel(m+10,my);
    lcd.setPixel(m+10,my+1);
    lcd.setPixel(m+10,my+2);
    lcd.setPixel(m+10,my+3);
    lcd.setPixel(m+10,my+4);
    lcd.setPixel(m+12,my);
    lcd.setPixel(m+12,my+1);
    lcd.setPixel(m+12,my+2);
    lcd.setPixel(m+12,my+3);
    lcd.setPixel(m+12,my+4);
    lcd.setPixel(m+13,my);
    lcd.setPixel(m+13,my+2);
    lcd.setPixel(m+14,my);
    lcd.setPixel(m+14,my+1);
    lcd.setPixel(m+14,my+3);
    lcd.setPixel(m+14,my+4);
    lcd.setPixel(m+16,my);
    lcd.setPixel(m+16,my+1);
    lcd.setPixel(m+16,my+2);
    lcd.setPixel(m+16,my+3);
    lcd.setPixel(m+16,my+4);
    lcd.setPixel(m+17,my);
    lcd.setPixel(m+17,my+2);
    lcd.setPixel(m+17,my+4);
    lcd.setPixel(m+18,my);
    lcd.setPixel(m+18,my+4);
}

void level(int x1,int y1)           //display the level text on the screen
{
    int x0=x1;
    int y0=y1;
    lcd.drawLine(x1,y1+4,x0,y0,1);
    lcd.drawLine(x1+2,y1+4,x0,y0+4,1);
    lcd.drawLine(x1+4,y1+4,x0+4,y0,1);
    lcd.drawLine(x1+6,y1,x0+4,y0,1);
    lcd.drawLine(x1+6,y1+4,x0+4,y0+4,1);
    lcd.setPixel(x1+5,y1+2);
    lcd.drawLine(x1+8,y1+3,x0+8,y0,1);
    lcd.drawLine(x1+10,y1+3,x0+10,y0,1);
    lcd.setPixel(x1+9,y1+4);
    lcd.drawLine(x1+12,y1+4,x0+12,y0,1);
    lcd.drawLine(x1+14,y1,x0+12,y0,1);
    lcd.drawLine(x1+14,y1+4,x0+12,y0+4,1);
    lcd.setPixel(x1+13,y1+2);
    lcd.drawLine(x1+16,y1+4,x0+16,y0,1);
    lcd.drawLine(x1+18,y1+4,x0+16,y0+4,1);
}

void button_isr()               //use for interrupt
{
    g_button_flag=1;
}