/* mbed USBHost Gamepad driver sample
 * Copyright (c) 2014 Yuuichi Akagawa
 *
 * modified from mbed USBHostMouse
 *
 * mbed USBHost Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "USBHostGamepad.h"
//#if USBHOST_GAMEPAD

USBHostGamepad::USBHostGamepad() {
    host = USBHost::getHostInst();
    init();
}

void USBHostGamepad::init() {
    dev = NULL;
    int_in = NULL;
    onUpdate = NULL;
    report_id = 0;
    dev_connected = false;
    gamepad_device_found = false;
    gamepad_intf = -1;

    btnX = 0;
    btnY = 0;
    btnABCD = 0;
    btnSpecial = 0;
}

bool USBHostGamepad::connected() {
    return dev_connected;
}

bool USBHostGamepad::connect() {
    int len_listen;

    if (dev_connected) {
        return true;
    }

    for (uint8_t i = 0; i < MAX_DEVICE_CONNECTED; i++) {
        if ((dev = host->getDevice(i)) != NULL) {

            if(host->enumerate(dev, this))
                break;

            if (gamepad_device_found) {

                int_in = dev->getEndpoint(gamepad_intf, INTERRUPT_ENDPOINT, IN);
                if (!int_in)
                    break;

                USB_INFO("New Gamepad/Joystick device: VID:%04x PID:%04x [dev: %p - intf: %d]", dev->getVid(), dev->getPid(), dev, gamepad_intf);
#if DEBUG > 3
                //Parse HID Report Descriptor
                parseHidDescr();
#endif
                dev->setName("Gamepad", gamepad_intf);
                host->registerDriver(dev, gamepad_intf, this, &USBHostGamepad::init);

                int_in->attach(this, &USBHostGamepad::rxHandler);
                len_listen = int_in->getSize();
                if (len_listen > sizeof(report)) {
                    len_listen = sizeof(report);
                }
                host->interruptRead(dev, int_in, report, len_listen, false);

                dev_connected = true;
                return true;
            }
        }
    }
    init();
    return false;
}

void USBHostGamepad::rxHandler() {
    int len_listen = int_in->getSize();
#if DEBUG > 3
    USB_DBG("USBHostGamepad::rxHandler() len_listen=%d\r\n", len_listen);
      for (int i = 0; i < len_listen; i++)
          printf("%02X ", report[i]);
      printf("\r\n\r\n");
#endif
    if (onUpdate) {
        (*onUpdate)(report[0], report[1], report[5], report[6]);
    }

    // update gamepad state
    btnX       = report[0];
    btnY       = report[1];
    btnABCD    = report[5];
    btnSpecial = report[6];

    if (len_listen > sizeof(report)) {
        len_listen = sizeof(report);
    }

    if (dev)
        host->interruptRead(dev, int_in, report, len_listen, false);
}

/*virtual*/ void USBHostGamepad::setVidPid(uint16_t vid, uint16_t pid)
{
    // we don't check VID/PID for gamepad driver
}

/*virtual*/ bool USBHostGamepad::parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol) //Must return true if the interface should be parsed
{
    if ((gamepad_intf == -1) &&
        (intf_class == HID_CLASS) &&
        (intf_subclass == 0x00) &&
        (intf_protocol == 0x00)) {
        gamepad_intf = intf_nb;
        return true;
    }
    return false;
}

/*virtual*/ bool USBHostGamepad::useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir) //Must return true if the endpoint will be used
{
    if (intf_nb == gamepad_intf) {
        if (type == INTERRUPT_ENDPOINT && dir == IN) {
            gamepad_device_found = true;
            return true;
        }
    }
    return false;
}

USB_TYPE USBHostGamepad::getReportDescriptor(USBDeviceConnected * dev, uint8_t * buf, uint16_t max_len_buf, uint16_t * len_rep_descr)
{
    USB_TYPE t = host->controlRead(  dev,
                         USB_DEVICE_TO_HOST | USB_RECIPIENT_INTERFACE | USB_REQUEST_TYPE_STANDARD,
                         HID_GET_DESCRIPTOR,
                         0x2200,
                         0, buf, max_len_buf);
    if (len_rep_descr)
        *len_rep_descr = max_len_buf;

    return t;
}

/*
 * HID Report Descriptor parser
 */ 
bool USBHostGamepad::parseHidDescr()
{
    bool ret = true;
    uint8_t *buf;
    uint16_t desclen = host->getLengthReportDescr();

    //allocate report descriptor's buffer
    buf = (uint8_t *)malloc(desclen);
    if(buf == NULL){
        return false;
    }
    getReportDescriptor(dev, buf, desclen);
  #if (DEBUG > 3)
      USB_DBG("HID REPORT DESCRIPTOR:\r\n");
      for (int i = 0; i < desclen; i++)
          printf("%02X ", buf[i]);
      printf("\r\n\r\n");
  #endif
  if(  buf[0] == 0x05 //Usage page
    && buf[1] == 0x01 //Generic Desktop
    && buf[2] == 0x09 //Usage
    ){
        if( buf[3] == 0x04 ){
            USB_DBG("GAMEPAD");
        }else if( buf[3] == 0x05){
            USB_DBG("JOYSTICK");
        }else{
            ret = false;
        }
    }
    free(buf);
    return ret;
}
//#endif
