/* fluent-logger-mbed 
 * Copyright (c) 2014 Yuuichi Akagawa
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "FluentLogger.h"
#ifdef USE_NTP
#include <time.h>
#endif

FluentLogger::FluentLogger(const char *host, const int port, uint32_t bufsize) :
_sock(), _host(host), _port(port), _timeout(1000)
{
    _mp = new uMP(bufsize);
    _sock = NULL;
}

int FluentLogger::open()
{
    if ( _sock != NULL ) {
        return 0;
    }
    _sock = new TCPSocketConnection();
    int ret = _sock->connect(_host, _port);
    if (ret < 0)
    {
        _sock->close();
        return -1;
    }
    return 0;
}

int FluentLogger::close()
{
    if (_sock->is_connected()) {
        _sock->close();
    }
    delete _sock;
    _sock = NULL;
    return 0;
}

int FluentLogger::log(const char *tag, const char *msg)
{
    if (_sock == NULL || !_sock->is_connected()) {
        if (open() < 0) {
            return -1;
        }
    }
    _mp->init();

    // tag, timestamp, message
    if (!_mp->start_array(3)) {
        return -1;
    }
    if (!_mp->set_str(tag, strlen(tag))) {
        return -1;
    }
#ifdef USE_NTP  
    if (!_mp->set_u32(time(NULL))) {
        return -1;
    }
#else
    if (!_mp->set_u32(0)) {
        return -1;
    }
#endif
    if (!_mp->set_str(msg, strlen(msg))) {
        return -1;
    }
    return(send());
}

int FluentLogger::log(const char *tag, uMP &mpmsg)
{
    if (_sock == NULL || !_sock->is_connected()) {
        if (open() < 0) {
            return -1;
        }
    }
    _mp->init();

    // tag, timestamp, message
    if (!_mp->start_array(3)) {
        return -1;
    }
    if (!_mp->set_str(tag, strlen(tag))) {
        return -1;
    }
#ifdef USE_NTP  
    if (!_mp->set_u32(time(NULL))) {
        return -1;
    }
#else
    if (!_mp->set_u32(0)) {
        return -1;
    }
#endif
    if (!_mp->set_raw((const char*)mpmsg.get_buffer(), mpmsg.get_size())) {
        return -1;
    }
    return(send());
}

int FluentLogger::send()
{
    _sock->set_blocking(false, _timeout);
    int ret = _sock->send_all((char*)_mp->get_buffer(), (int)_mp->get_size());
    if ( ret < 0 ) {//fail
        close();
        ret = -2;
    }
    return(ret);
}
