//*****************************************************************************
// Example writing to SD card, sford
//
// 21 June 2011: Enhanced by D. Wendelboe to test SD file and directories.
// Illustrates the following:
//      * rename a file (via copy & delete old).
//      * copy an existing file to a new file.
//      * listing of SD directory names.
//
// TODO: Add method to list filenames in a directory.
//       Add a method to return file length.
//*****************************************************************************

#include "mbed.h"
#include "SDFileSystem.h"

#define PC_BAUD   38400

SDFileSystem sd(p5, p6, p7, p8, "sd"); 
Serial pc(USBTX,USBRX);

int file_rename(const char *oldfname, const char *newfname);
int file_copy(const char *src, const char *dst);

int main() 
{
    int i, status;
    pc.baud(PC_BAUD);
    
    printf("\nExercise SD Card functions\n"); 
      
    printf("Create 3 directories on SD Card.\n");  
    mkdir("/sd/mydir1", 0777);
    mkdir("/sd/mydir2", 0777);
    mkdir("/sd/mydir3", 0777);
    
//--------------------------------------------------------------    
    printf("Create \"/sd/mydir1/sdtest.txt\" and write some data in it\n");
    
    FILE *fp = fopen("/sd/mydir1/sdtest.txt", "w");
    if(fp == NULL) {
        error("Could not open file for write\n");
    }
    fprintf(fp, "Hello SD Card World!\n");
    for (i=0; i<10; i++) {
        fprintf(fp, "Line #%i\n", i);
    }
    fclose(fp); 
                   
//--------------------------------------------------------------
    printf("Open directory on /sd\n");    
    DIR *d = opendir("/sd");              
    struct dirent *p;
    while((p = readdir(d)) != NULL) {         
      printf("%s\n", p->d_name);            
    }
    closedir(d);
    printf("Directory closed\n");
    
 //--------------------------------------------------------------  
    printf("Remove /sd/mydir3\n"); 
    remove("/sd/mydir3");      
              
//--------------------------------------------------------------
    printf("List directory again. mydir3 should be gone.\n");    
    d = opendir("/sd");              
    while((p = readdir(d)) != NULL) {         
      printf("%s\n", p->d_name);            
    }
    closedir(d);
    
 //-------------------------------------------------------------- 
    printf("Rename \"/sd/mydir1/sdtest.txt\" to \"/sd/mydir1/new_name.txt\"\n"); 
    status = file_rename("/sd/mydir1/sdtest.txt", "/sd/mydir1/new_name.txt");

    if (status == -1)  printf("Error: file not renamed. (status=%d)\n", status);
    else printf("Success: file renamed OK (status=%d)\n", status);
    
//--------------------------------------------------------------
    printf("Copy \"/sd/mydir1/doug.txt\" to \"/sd/mydir1/new_copy.txt\"\n");
    status = file_copy("/sd/mydir1/new_name.txt", "/sd/mydir1/new_copy.txt");
    
    if (status == -1)  printf("Error: file not copied. (status=%d)\n", status);
    else printf("Success: file copied OK (status=%d)\n", status);
    
//--------------------------------------------------------------
    printf("Done.\n");
}   

//***********************************************************
// file_rename: renames a file (via copy & delete).
//    Moves data instead of adjusting the file name in the
//    file directory. Checks to insure the file was renamed.
//    Returns -1 = error; 0 = success
//***********************************************************
int file_rename(const char *oldfname, const char *newfname) {
    int retval = 0;
    int ch;

    FILE *fpold = fopen(oldfname, "r");   // src file
    FILE *fpnew = fopen(newfname, "w");   // dest file
    
    while (1) {                   // Copy src to dest  
        ch = fgetc(fpold);        // until src EOF read. 
        if (ch == EOF) break;
        fputc(ch, fpnew);  
    }
    
    fclose(fpnew);
    fclose(fpold);

    fpnew = fopen(newfname, "r"); // Reopen dest to insure
    if(fpnew == NULL) {           // that it was created.
        retval = (-1);            // Return Error.
    } 
    else {
        fclose(fpnew);	
        remove(oldfname);         // Remove original file.
		retval = (0);             // Return Success.
    }
    return (retval);
}

//***********************************************************
// file_copy: Copies a file
//            Checks to insure destination file was created.
//            Returns -1 = error; 0 = success
//***********************************************************
int file_copy (const char *src, const char *dst) {
    int retval = 0;
    int ch;

    FILE *fpsrc = fopen(src, "r");   // src file
    FILE *fpdst = fopen(dst, "w");   // dest file
    
    while (1) {                  // Copy src to dest
        ch = fgetc(fpsrc);       // until src EOF read.
        if (ch == EOF) break;
        fputc(ch, fpdst);  
    }
	fclose(fpsrc);  
    fclose(fpdst);
  
    fpdst = fopen(dst, "r");     // Reopen dest to insure
    if(fpdst == NULL) {          // that it was created.
        retval = (-1);           // Return error.
    } 
    else {
        fclose(fpdst); 
        retval = (0);            // Return success.
    }
    return (retval);
}

