#include "mbed.h"
#include "BLE.h"
#include "color_pixels.h"
#include <math.h>

#include "PulseSensor.h"

#define NEED_DEBUG
#ifdef NEED_DEBUG
#define DEBUG(...) { printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif 


const static char  DEVICE_NAME[] = "mbed HRM1017";
static volatile bool  triggerSensorPolling = false;


BLEDevice  ble;


/* Heart Rate Service */ 
static uint8_t hrmCounter = 100;
static uint8_t bpm[2] = {0x00, hrmCounter};
GattCharacteristic  hrmChar(GattCharacteristic::UUID_HEART_RATE_MEASUREMENT_CHAR
                           ,bpm, sizeof(bpm) ,sizeof(bpm)
                           ,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
static uint8_t location = 0x05; /* Ear Lobe */
GattCharacteristic hrmLocation(GattCharacteristic::UUID_BODY_SENSOR_LOCATION_CHAR
                           ,(uint8_t *)&location ,sizeof(location) ,sizeof(location)
                           ,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *hrmChars[] = {&hrmChar, &hrmLocation,};
GattService hrmService(GattService::UUID_HEART_RATE_SERVICE
                       ,hrmChars ,sizeof(hrmChars)/sizeof(GattCharacteristic *));



/* Battery Level Service */
static uint8_t batt = 100;
GattCharacteristic battLevel(GattCharacteristic::UUID_BATTERY_LEVEL_CHAR
                            ,(uint8_t *)&batt ,sizeof(batt) ,sizeof(batt)
                            ,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);             
GattCharacteristic *battChars[] = {&battLevel,};
GattService battService(GattService::UUID_BATTERY_SERVICE
                       ,battChars ,sizeof(battChars)/sizeof(GattCharacteristic *));



/* Device Information service */
static uint8_t deviceName[] = {'H', 'R', 'M', '1', '0', '1', '7'};
GattCharacteristic deviceManufacturer(GattCharacteristic::UUID_MANUFACTURER_NAME_STRING_CHAR
                                     ,(uint8_t *)deviceName ,sizeof(deviceName) ,sizeof(deviceName)
                                     ,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *devInfoChars[] = {&deviceManufacturer,};
GattService deviceInformationService(GattService::UUID_DEVICE_INFORMATION_SERVICE
                                     ,devInfoChars ,sizeof(devInfoChars)/sizeof(GattCharacteristic *));


static uint16_t uuid16_list[] = {
                          GattService::UUID_HEART_RATE_SERVICE
                         ,GattService::UUID_BATTERY_SERVICE
                         ,GattService::UUID_DEVICE_INFORMATION_SERVICE
                          };


void updateServiceValues(void);
static Gap::ConnectionParams_t connectionParams;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    DEBUG("Disconnected handle %u, reason %u\r\n", params->handle, params->reason);
    DEBUG("Restarting the advertising process\r\n");
    ble.gap().startAdvertising();
}

void onConnectionCallback(const Gap::ConnectionCallbackParams_t *params)   //Mod
{
    DEBUG("connected. Got handle %u\r\n", params->handle);

    connectionParams.slaveLatency = 1;
    if (ble.gap().updateConnectionParams(params->handle, &connectionParams) != BLE_ERROR_NONE) {
        DEBUG("failed to update connection paramter\r\n");
    }
}

void periodicCallback(void)
{
    triggerSensorPolling = true;
}

ColorPixels pixels(6,8);
int ledfade = 0;
void pixelPattern(int n, uint8_t r, uint8_t g, uint8_t b);


int main(void)
{   
    Ticker ticker;
    ticker.attach(periodicCallback, 1);
        
    DEBUG("Initialising the nRF51822\r\n");
    ble.init();
    DEBUG("Init done\r\n");
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gap().onConnection(onConnectionCallback);

    ble.gap().getPreferredConnectionParams(&connectionParams);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t*)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_HEART_RATE_SENSOR);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.gap().startAdvertising();
    DEBUG("Start Advertising\r\n");

    ble.gattServer().addService(hrmService);
    ble.gattServer().addService(battService);
    ble.gattServer().addService(deviceInformationService);
    DEBUG("Add Service\r\n");
    
    initPulseSensor();

    Ticker hrm_ticker;
    hrm_ticker.attach(calcHeartRate, 0.02);
    
    int counter = 0;
    uint8_t cl = 193;

    while (true) {
        if (triggerSensorPolling) {
            triggerSensorPolling = false;
            updateServiceValues();
        } else {
            ble.waitForEvent();
        }
        if (counter % 3 == 0) {
            if (ledfade) {
                --ledfade;
                pixelPattern(ledfade, cl, cl, cl);
                counter = 0;
            }
        }
        ++counter;
    }
    
}



void updateServiceValues(void)
{
    /* Decrement the battery level. */
    batt <= 50 ? batt = 100 : batt--;
    ble.gattServer().write(battLevel.getValueAttribute().getHandle(), (uint8_t *)&batt, sizeof(batt));

    /* Randomize the heart rate. */
    // hrmCounter = (rand() % 150) + 30;
    if (isQS()) {
        hrmCounter = getBPM();
        DEBUG("BPM: %d\r\n", hrmCounter);
        ledfade = 5;
    }
        
    bpm[1] = hrmCounter;
    ble.gattServer().write(hrmChar.getValueAttribute().getHandle(), bpm, sizeof(bpm));
}


void pixelPattern(int n, uint8_t r, uint8_t g, uint8_t b) {
    
    if (n < 0 || n > 4) return;
    
    switch (n) {
    case 4:
        pixels.set_color(7, r, g, b);
        pixels.set_color(6, r, g, b);
        pixels.set_color(5, r, g, b);
        pixels.set_color(4, r, g, b);
        pixels.set_color(3, r, g, b);
        pixels.set_color(2, r, g, b);
        pixels.set_color(1, r, g, b);
        pixels.set_color(0, r, g, b);
        pixels.update();
        break;
    case 3:
        pixels.set_color(7, 0, 0, 0);
        pixels.set_color(6, r/2, g/2, b/2);
        pixels.set_color(5, r*3/4, g*3/4, b*3/4);
        pixels.set_color(4, r, g, b);
        pixels.set_color(3, r, g, b);
        pixels.set_color(2, r*3/4, g*3/4, b*3/4);
        pixels.set_color(1, r/2, g/2, b/2);
        pixels.set_color(0, 0, 0, 0);
        pixels.update();
        break;
    case 2:
        pixels.set_color(7, 0, 0, 0);
        pixels.set_color(6, 0, 0, 0);
        pixels.set_color(5, r/2, g/2, b/2);
        pixels.set_color(4, r*3/4, g*3/4, b*3/4);
        pixels.set_color(3, r*3/4, g*3/4, b*3/4);
        pixels.set_color(2, r/2, g/2, b/2);
        pixels.set_color(1, 0, 0, 0);
        pixels.set_color(0, 0, 0, 0);
        pixels.update();
        break;
    case 1:
        pixels.set_color(7, 0, 0, 0);
        pixels.set_color(6, 0, 0, 0);
        pixels.set_color(5, 0, 0, 0);
        pixels.set_color(4, r/2, g/2, b/4);
        pixels.set_color(3, r/2, g/2, b/4);
        pixels.set_color(2, 0, 0, 0);
        pixels.set_color(1, 0, 0, 0);
        pixels.set_color(0, 0, 0, 0);
        pixels.update();
        break;
    case 0:
    default:
        pixels.set_color(7, 0, 0, 0);
        pixels.set_color(6, 0, 0, 0);
        pixels.set_color(5, 0, 0, 0);
        pixels.set_color(4, 0, 0, 0);
        pixels.set_color(3, 0, 0, 0);
        pixels.set_color(2, 0, 0, 0);
        pixels.set_color(1, 0, 0, 0);
        pixels.set_color(0, 0, 0, 0);
        pixels.update();
    }
}