/**
 * @file   TrackVector2D.h
 * @author Ernesto Palacios
 *
 * @brief
 * Develop Under  GPL v3.0 License
 * http://www.gnu.org/licenses/gpl-3.0.html .
 *
 * Created on 22 de octubre de 2011, 11:26 PM
 */

#ifndef TRACKVECTOR2D_H
#define TRACKVECTOR2D_H

#include "mbed.h"
#include "Matrix.h"
#include "MatrixMath.h"

#include <complex>

/**@brief
 * This class keeps track of a "vector" on a 2D plane.
 * Vector is defined as any point on an XY plane, with a determine direction.
 * Could be a mobile robot for instance. Distance is unit-less depends on the use.
 * Angles are internally measured in radiands but the API uses degrees.
 *
 * Static Functions are provided for Angle and Coordenates transformations.
 */
class TrackVector2D {

public:

    /**@brief default Constructor.
     * Initializes Matrices to default Values
     */
    TrackVector2D();


    /**@brief Copies all elements.
     * @param orig Original "Vector"
     */
    TrackVector2D(const TrackVector2D& orig);


    /**@brief
     * Move the "vector" to position x, y in the plane.
     * Calculates the angle the vector should rotate, and the distance
     * it should travel. Distance depeneds on metric units of coordenates.
     * @param x  Coordenate in "X" axis.
     * @param y  Coordenate in "Y" axis.
     * @return   1x2 Matrix, [1][1] Distance to travel -no units.
     *                       [1][2] Rotation Angle in degrees
     */
    const Matrix moveto( float x, float y );


    /**@brief
     * Changes the direction of the "vector" to point to a new direction.
     * Keeps track of ratation. Doesn't return anything because does not travels
     * any distance, and rotation is already known.
     * @param degrees Angle in degrees to Rotate
     */
    void rotate( float degrees );


    /**@brief Returns 3x3 Matrix Rotation
     *
     * @return 3x3 Matrix Rotation only
     */
    const Matrix GetRotM();


    /**@brief Retirns 4x4 Full Transformation Matrix
     *
     * @return 4x4 Full Transfrmation Matrix.
     */
    const Matrix GetTrasfM();


    /**@brief Returns 1x2 Matrix last [ X ][ Y ] coordenates
     * @return 1x2 with last only Cordenates.
     */
    const Matrix GetLastMove();


    /**@brief Retruns nx2 Matrix with all n-Moves.
     * @return Large nx2 Matrix with all moves done by "vector"
     */
    const Matrix GetMove( int index );

    /**@brief Cuerrent X position of Robot
     * @return X position in Coordenate Sistem.
     */
    float getX();


    /**@brief Current Y position of Robot
     * @return Y position in Coordenate Sistem
     */
    float getY();


    /**@brief Current direction with respect to "X" axis.
     * @return Angle in radians
     */
    float getRad();


    /**@brief Current direction with respect to "X" axis.
     * @return Angle in Degrees
     */
    float getDeg();


    /**@brief Transform from RADIANS to DEGREES.
     * @param angle in radians
     * @return angle in degrees
     */
    static float rad2deg( float angle );


    /**@brief Transform from DEGREES to RADIANS.
     * @param angle in degrees
     * @return angle in radians
     */
    static float deg2rad( float angle );


    /**@brief Transforms from Rectangular to Polar coordenates.
     * @param 1x2 Matrix [x][y]
     * @return 1x2 Matrix [Magnitude][Direction]  in Degrees!
     */
    static const Matrix Rect2Polar( const Matrix& Coord );


    /**@brief Transforms from Polar to Rectangular coordenates.
     * @param 1x2 Matrix [Magnitude][Direction] in Degrees!
     * @return 1x2 Matrix [x][y]
     */
    static const Matrix Polar2Rect( const Matrix& Coord );


private:

    Matrix _TrnsfM;
    Matrix _InvM;
    Matrix _RotM;
    vector< complex<float> > _Moves; 

    int   _nMoves;
    float _angRad;
    float _angDeg;
    float _distance;
};

#endif  /* TRACKVECTOR2D_H */
