#ifndef L293D_H
#define L293d_H

#include "mbed.h"

/**@brief  Class to handle L293D in two modes, Pwm or Digital.
 * Pwm Mode requires constructing an object with two PwmOut'puts to handle Speed.
 * Enable, or Digital Mode requires One DigitalOut'put to Enable or Disable both sides
 * of the L293D driver.
 *
 * This of course requires an 'special' wiring of the driver, using one transistor for each side.
 * The transistor is effectively a NOT gate, assuring that the direction of the current of the 
 * driver output will be controlled using just one pin.
 *                                                  _____________
 *                               Enable/ Pwm --->[1] 
 *                   _5V__                         |      L
 *                     |         _______________ [2]
 *                    _|_       /                  |      2
 *                   |  |       |       ---------[3]
 *                   |1k|       |       |          |      9
 *                   |__|       |       |   (GND)[4]
 *                    /_________/     ( ~ )        |      3
 * _left            |/                  |   (GND)[5]
 * -.---|470ohm|----|\  (Q 2N3904)      |          |
 *  |               | \                 | -------[6]      D
 *  |                  |_(GND)                     |    
 *  |                                              |
 *  \___________________________________________ [7]
 *                                                 |
 *                                       5~15V-->[8]_____________
 *
 * Hope you get my ASCII Art.  same goes for the other side, thus allowing a complete control of the driver 
 * with a few pins.
 *
 * Also for Sake of Simplicity SetSpeedLeft/Right will control Motor Speed and Direction, from -1.0 to 1.0 
 * Reverse if < 0.
 */
class L293D{

public:

    /**@brief
     * Two PWM for speed Cpntrol & two digital outs needed for setting direction 
     * on each side of the L293D.
     * @param left:  Output to the left side of the driver
     * @param right: Output to the right side of the driver
     * @param pwmLeft: Left side PWM.
     * @param pwmRight: Right side PWM.
     */
    L293D( PwmOut pwmLeft, PwmOut right, DigitalOut left, DigitalOut right );
    
    
    /**@brief
     * Set the left PWM to current speed and Direction.
     * Values from -1.0 to 1.0, to control direction and speed,
     * if Enable mode, only use -1.0 & 1.0
     */
    void setSpeedLeft( float speed );
    
    
    /**@brief
     * Set the right PWM to current speed and direction.
     * Values from -1.0 to 1.0, to control direction and speed,
     * if Enable mode, only use -1.0 & 1.0
     * Only to use when PWM mode is declared
     */
    void setSpeedRight( float speed );
    
    
    /**@brief
     * Read the left PWM current speed & direction
     * Ranges from -1.0 to 1.0, as being set.
     * @return Current PWM duty cycle.
     */
    float getSpeedLeft();
    
    
    /**@brief
     * Read the left PWM current speed & direction.
     * Ranges from -1.0 to 1.0 as being set.
     * @return Current PWM duty cycle.
     */
    float getSpeedRight();
    
    
private:

    DigitalOut _left;
    DigitalOut _right;
    PwmOut _pwmLeft;
    PwmOut _pwmRight;
    
    float _speedLeft;
    float _speedRight;
    
};

#endif