/* mbed simplified access to Microchip MCP42xxx/MCP41xxx Digital Potentiometer devices (SPI)
 * Copyright (c) 2013 ygarcia, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "MCP4xxxx_SPI.h"

namespace MCP4xxxx_SPI {

    unsigned char CMCP4xxxx_SPI::SPIModuleRefCounter = 0;

    CMCP4xxxx_SPI::CMCP4xxxx_SPI(const PinName p_mosi, const PinName p_miso, const PinName p_sclk, const PinName p_cs, const PinName p_reset, const PinName p_shdn, const unsigned int p_frequency) : _internalId("") {
        DEBUG_ENTER("CMCP4xxxx_SPI")

        CMCP4xxxx_SPI::SPIModuleRefCounter += 1;
        if (CMCP4xxxx_SPI::SPIModuleRefCounter > 1) {
            //FIXME Check that SPI settings are identical. Otherwise it should failed
            return;
        }

        _spiInstance = new SPI(p_mosi, p_miso, p_sclk);
        _spiInstance->frequency(p_frequency); // Set the frequency of the SPI interface
        _spiInstance->format(16, 0); // See http://mbed.org/users/mbed_official/code/mbed/docs/0954ebd79f59//classmbed_1_1SPI.html
        DEBUG("CMCP4xxxx_SPI: refCounter=%d", CMCP4xxxx_SPI::SPIModuleRefCounter)

        if (p_cs != NC) {
            DEBUG("CMCP4xxxx_SPI: /CS managed");
            _cs = new DigitalOut(p_cs);
            _cs->write(1); // Disable chip
        } else {
            DEBUG("CMCP4xxxx_SPI: /CS not managed");
            _cs = NULL; // Not used
        }
    
        if (p_reset != NC) {
            DEBUG("CMCP4xxxx_SPI: /RESET managed");
            _reset = new DigitalOut(p_reset);
            _reset->write(1); // Disable reset
        } else {
            DEBUG("CMCP4xxxx_SPI: /RESET not managed");
            _reset = NULL; // Not used
        }
    
        if (p_shdn != NC) {
            DEBUG("CMCP4xxxx_SPI: /SHDN managed");
            _shdn = new DigitalOut(p_shdn);
            _shdn->write(1); // Disable shutdown
        } else {
            DEBUG("CMCP4xxxx_SPI: /SHDN not managed");
            _shdn = NULL; // Not used
        }
    
        DEBUG_LEAVE("CMCP4xxxx_SPI")
    }

    CMCP4xxxx_SPI::~CMCP4xxxx_SPI() {
        DEBUG_ENTER("~CMCP4xxxx_SPI")
    
        // Release I2C instance
        DEBUG_ENTER("~CMCP4xxxx_SPI: refCounter=%d", CMCP4xxxx_SPI::SPIModuleRefCounter)
        CMCP4xxxx_SPI::SPIModuleRefCounter -= 1;
        if (CMCP4xxxx_SPI::SPIModuleRefCounter == 0) {
            delete _spiInstance;
            _spiInstance = NULL;
        }
        // Release _reset if required
        if (_cs != NULL) {
            _cs->write(1);
            delete _cs;
        }
        // Release _reset if required
        if (_reset != NULL) {
            _reset->write(0);
            delete _reset;
        }
        // Release _shdn if required
        if (_shdn != NULL) {
            _shdn->write(0);
            delete _shdn;
        }
   
        DEBUG_LEAVE("~CMCP4xxxx_SPI")
    }
    
    unsigned short CMCP4xxxx_SPI::Write(const Commands p_command, const unsigned char p_value) {
        DEBUG_ENTER("CMCP4xxxx_SPI::Write: 0x%02x - 0x%02x", (unsigned char)p_command, p_value)
        
        // Sanity check
        if ((p_command != WriteToPot1) && (p_command != WriteToPot2) && (p_command != WriteToBoth)) {
            // Wrong parameters
            return (unsigned short) -1;
        }
        
        unsigned short command = 0;
        switch (p_command) {
            case WriteToPot1: 
                command = (0x11 << 8 | p_value);
                break;
            case WriteToPot2:
                command = (0x12 << 8 | p_value);
                break;
            default:
                command = (0x13 << 8 | p_value);
        } // End of 'switch' statement
        
        DEBUG("CMCP4xxxx_SPI: Send command: 0x%04x", command)
        if (_cs != NULL) {
            _cs->write(0);
        }
        unsigned short result = _spiInstance->write(command);
        if (_cs != NULL) {
            _cs->write(1);
        }
        
        DEBUG_LEAVE("CMCP4xxxx_SPI::Write: %d", result)
        return result;
    }

    unsigned short CMCP4xxxx_SPI::Write(const Commands p_command) {
        DEBUG_ENTER("CMCP4xxxx_SPI::Write: 0x%02x", (unsigned char)p_command)
        
        // Sanity check
        if ((p_command != ShutdownPot1) && (p_command != ShutdownPot2) && (p_command != ShutdownBoth)) {
            // Wrong parameters
            return (unsigned short) -1;
        }
        
       unsigned short command = 0;
        switch (p_command) {
            case ShutdownPot1:
                command = (0x21 << 8);
                break;
            case ShutdownPot2:
                command = (0x21 << 8);
                break;
            default: //<! Shutdown both digital potentiometers
                command = (0x23 << 8);                
        } // End of 'switch' statement
        
        DEBUG("CMCP4xxxx_SPI: Send command: 0x%04x", command)
        if (_cs != NULL) {
            _cs->write(0);
        }
        unsigned short result = _spiInstance->write(command);
        if (_cs != NULL) {
            _cs->write(1);
        }
        
        DEBUG_LEAVE("CMCP4xxxx_SPI::Write: %d", result)
        return result;
    }
    unsigned short CMCP4xxxx_SPI::Write() {
        return _spiInstance->write(0);
    }

    bool CMCP4xxxx_SPI::Reset() {
        // Sanity check
        if (_reset == NULL) {
            return false;
        }
        
        _reset->write(0); // Set level low to activate reset 
        wait_us(1); // Wait for 1us
        _reset->write(1); // Set level low to de-activate reset 
       
        return true;
    }
    
    bool CMCP4xxxx_SPI::Shutdown(const bool p_set) {
        // Sanity check
        if (_shdn == NULL) {
            return false;
        }
        
        _shdn->write(p_set == false ? 0 : 1);
        
        return true;
    }

} // End of namespace MCP4xxxx_SPI
