#include <string>
#include <iostream>
#include <iomanip>

#include "MCP4xxxx_SPI.h" // Use SPI module #1 and /CS mapped on pin 8
#include "MCP320x_SPI.h" // Use SPI module #1 and /CS mapped on pin 9

struct UserChoice {
    char choice;
    unsigned char potId;
    unsigned char adcId;
    CMCP4xxxx_SPI::Mcp4xxxFamilly familly;
    bool isShutdown;
};

/*
 * Declare functions
 */
void AvailableIndicator(); // LED1 flashing for program while program is alive
void DisplayMenuAndGetChoice(); // Display and get the user choice

/*
 * Declare statics
 */
DigitalOut g_availableLed(LED1); // To verify if program in running
Ticker g_available; // LED1 will flash with a period of 2s
CMCP4xxxx_SPI g_digitalPot(p5, p6, p7);
CMCP320x_SPI *g_adc = NULL;
DigitalOut g_cs3201(p9);  // /CS mapped on pin 8 for MCP3201
DigitalOut g_cs3208(p10); // /CS mapped on pin 10 for MCP3208
DigitalOut g_cs42100(p8); // /CS mapped on pin 8 for MCP421pp
DigitalOut g_cs41050(p11); // /CS mapped on pin 11 for MCP41050

DigitalOut g_cs4152(p13); // /CS mapped on pin 13 for MCP4152
DigitalOut g_cs4251(p14); // /CS mapped on pin 14 for MCP4251
DigitalOut g_shdn4251(p15); // /SHDN mapped on pin 15 for MCP4251
DigitalOut g_wp4251(p16); // /WP mapped on pin 16 for MCP4251

DigitalOut *g_csCurrentAdc = NULL;    
    
static UserChoice g_userChoice; // Used to store user choice from displayed menu

int main() {
    g_userChoice.familly = CMCP4xxxx_SPI::_41xxx; // Default value used by the constructor
    g_userChoice.isShutdown = false;

    // Deactivate all SPI devices
    g_cs3201 = 1;
    g_cs3208 = 1;
    g_cs42100 = 1;
    g_cs41050 = 1;
    g_cs4152= 1; 
    g_cs4251 = 1; 
    g_shdn4251 = 1; // Active at low level 
    g_wp4251 = 1; 

    unsigned char potLevel = 0x80; // Initial digital potentiometer value
    
    // Launch available indicator
    g_available.attach(&AvailableIndicator, 2.0);
    
    while (true) {
        // Retrieve user choices 
        DisplayMenuAndGetChoice();
        // Set the pot. value
        //      1. Enable de right digipot
        switch (g_userChoice.potId) {
            case 'a':
                g_userChoice.familly = CMCP4xxxx_SPI::_42xxx;
                g_digitalPot.SetFamilly(g_userChoice.familly);
                g_cs42100 = 0;
                break;
            case 'b':
               g_userChoice.familly = CMCP4xxxx_SPI::_41xxx;
               g_digitalPot.SetFamilly(g_userChoice.familly);
               g_cs41050 = 0;
                break;
            case 'c':
                g_userChoice.familly = CMCP4xxxx_SPI::_42xx;
                g_digitalPot.SetFamilly(g_userChoice.familly);
                g_digitalPot.ReadRegister(CMCP4xxxx_SPI::Status);
                g_cs4251 = 0; 
                break;
            default:
                g_userChoice.familly = CMCP4xxxx_SPI::_41xx;
                g_digitalPot.SetFamilly(g_userChoice.familly);
                g_digitalPot.ReadRegister(CMCP4xxxx_SPI::Status);
                g_cs4152 = 0;
                break;
        } // End of 'switch' statement
        //      2. Apply user action
        switch (g_userChoice.choice) {
            case 'a':
                potLevel += 1;
                g_digitalPot.Write(CMCP4xxxx_SPI::WriteToDigiPot1, potLevel);
                break;
            case 'b':
                potLevel -= 1;
                g_digitalPot.Write(CMCP4xxxx_SPI::WriteToDigiPot1, potLevel);
                break;
            case 'c':
                g_userChoice.isShutdown = !g_userChoice.isShutdown;
                g_digitalPot.Shutdown(CMCP4xxxx_SPI::ShutdownDigiPot1, g_userChoice.isShutdown);
                break;
            case 'd':
                potLevel = 0x80;
                break;
           default:
                std::cout << "Invalid user choice\r" << std::endl;
                break;
        } // End of 'switch' statement
        //      3. Disable de right digipot
        switch (g_userChoice.potId) {
            case 'a':
                g_cs42100 = 1;
                break;
            case 'b':
                g_cs41050 = 1;
                break;
            case 'c':
                g_cs4251 = 1; 
                break;
            default:
                g_cs4152 = 1;
                break;
        } // End of 'switch' statement
        
        // Set adc to use
        switch (g_userChoice.adcId) {
            case 'a': // MCP3201
                g_adc = new CMCP320x_SPI(p5, p6, p7);
                g_csCurrentAdc = &g_cs3201;
                break;
            case 'b': // MCP3208
                g_adc = new CMCP320x_SPI(p5, p6, p7, NC, CMCP320x_SPI::_3208);
                g_csCurrentAdc = &g_cs3208;
                break;
        } // End of 'switch' statement
        g_csCurrentAdc->write(0);
        float sample = g_adc->Read(CMCP320x_SPI::CH2);
        g_csCurrentAdc->write(1);
        std::cout << "Voltage at PW0/41050: " << setprecision(5) << sample << "\r" << std::endl;
        
        g_csCurrentAdc->write(0);
        sample = g_adc->Read(CMCP320x_SPI::CH3);
        g_csCurrentAdc->write(1);
        std::cout << "Voltage at PW0/4152: " << setprecision(5) << sample << "\r" << std::endl;
        
        g_csCurrentAdc->write(0);
        sample = g_adc->Read(CMCP320x_SPI::CH4);
        g_csCurrentAdc->write(1);
        std::cout << "Voltage at PW0/4251#1: " << setprecision(5) << sample << "\r" << std::endl;
        
        g_csCurrentAdc->write(0);
        sample = g_adc->Read(CMCP320x_SPI::CH5);
        g_csCurrentAdc->write(1);
        std::cout << "Voltage at PW0/4251#2: " << setprecision(5) << sample << "\r" << std::endl;
        // Remove instance
        delete g_adc;
        g_csCurrentAdc = NULL;
    } // End of 'while' statement
} // End of program - nerver reached

void AvailableIndicator() {
    g_availableLed = !g_availableLed;
} // End of AvailableIndicator

void DisplayMenuAndGetChoice() {
    // Display the title
    std::cout << "\r" << std::endl << "MCP320x_SPI v0.3\r" << std::endl;

    // Display the pot selection menu
    std::cout << "\tUse pot 42100:\t\t\ta\r" << std::endl;
    std::cout << "\tUse pot 41050:\t\t\tb\r" << std::endl;
    std::cout << "\tUse pot  4251:\t\t\tc\r" << std::endl;
    std::cout << "Enter your choice: " << std::flush;
    g_userChoice.potId = getchar();
    std::cout << "\r" << std::endl << std::flush;

    // Display the adc selection menu
    std::cout << "\tUse adc 3201:\t\t\ta\r" << std::endl;
    std::cout << "\tUse adc 3208:\t\t\tb\r" << std::endl;
    std::cout << "Enter your choice: " << std::flush;
    g_userChoice.adcId = getchar();
    std::cout << "\r" << std::endl << std::flush;
    
    // Display the menu
    std::cout << "\tIncrease level of pot:\t\t\ta\r" << std::endl;
    std::cout << "\tDecrease level of pot:\t\t\tb\r" << std::endl;
    std::cout << "\tShutdown pot         :\t\t\tc\r" << std::endl;
    std::cout << "\tReset pot            :\t\t\td\r" << std::endl;
    std::cout << "Enter your choice: " << std::flush;
    g_userChoice.choice = getchar();
    std::cout << "\r" << std::endl << std::flush;
}
