/* mbed simplified interface for GPIO expender (e.g. Microchip MCP23x17)
 * Copyright (c) 2014-2015 ygarcia, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#pragma once

#include <list>

#include <mbed.h>

#define GPA0        0
#define GPA1        1
#define GPA2        2
#define GPA3        3
#define GPA4        4
#define GPA5        5
#define GPA6        6
#define GPA7        7
#define GPB0        8
#define GPB1        9
#define GPB2        10
#define GPB3        11
#define GPB4        12
#define GPB5        13
#define GPB6        14
#define GPB7        15
#define GPIO_MAX    GPB7
#define GPIO_MED    GPB0
#define GPIO_SIZE   8

/** This class describes a simplified interface for GPIO expender (e.g. Microchip MCP23x17)
 */
class AbstractGpioExpender {
public:
    /** Interrupt modes
     */
    enum InterruptModes {
        OnChange = 0x00,
        OnRising = 0x01,
        OnFalling = 0x03
    };
    
    /** Pull modes
     */
    enum PullModes {
        PullOff = 0x00,
        PullUp = 0x01,
        PullDown = 0x03
    };
    
    /** Reset the device
     */
    virtual void reset() = 0;

    /** Setup device interrupt mechanism
     * @param p_mirroring Set to 0x00 to disable INTA/B mirroring, 0x01 otherwise. Default: 0x00
     * @param p_openDrain Set to 0x00 for active driver output, 0x01 for opn drain output. Default: 0x00
     * @param p_polarity Set to 0x00 for interrupt active low, 0x01 otherwise. Default: 0x00
     */
    virtual void setupInterrupts(const unsigned char p_mirroring = 0x00, const unsigned char p_openDrain = 0x00, const unsigned char p_polarity = 0x00) = 0;
    
    /** Setup interrupt mode for a specific IO port
     * @param p_gpioId The IO port identifier
     * @param p_mode  The interrupt mode
     * @return 0 on success, -1 on wrong parameters and -2 otherwise
     */
    virtual int setupInterruptPin(const unsigned char p_gpioId, const InterruptModes p_mode = OnRising) = 0;
    
    /** Setup pull mode for a specific IO port
     * @param p_gpioId The IO port identifier
     * @param p_mode  The interrupt mode
     * @return 0 on success, -1 on wrong parameters and -2 otherwise
     */
    virtual int setupPullPin(const unsigned char p_gpioId, const PullModes p_mode = PullOff) = 0;
    
    /** Get interrupt information and clear it
     * @param p_gpioId The IO port identifier where the interrupt occured
     * @param p_value  The logic value on the pin port where the interrupt occured
     * @return 0 on success, -1 on wrong parameters and -2 otherwise
     */
    virtual int getLastInterruptPinAndValue(unsigned char * p_gpioId, unsigned char * p_value) = 0;
    
    /** Attach a function to call when a interrupt occurs on the GPIOA input ports
     * @param p_fptr The pointer to the "C" callback function
     */
    virtual void setIntrACallback(void (* p_fptr)(void)) = 0;
    
    /** Attach a function to call when a interrupt occurs on the GPIOB input ports
     * @param p_fptr The pointer to the "C" callback function
     */
    virtual void setIntrBCallback(void (* p_fptr)(void)) = 0;
    
    /** Read the specific GPIO port pin
     * @param p_gpioId The GPIO port pin to be read 
     * @param p_value The GPIO port pin value
     * @return 0 on success, -1 on wrong parameters and -2 otherwise
     */
    virtual int read(const unsigned char p_gpioId, unsigned char * p_value) = 0;
    
    /** Write value to the specific GPIO port pin
     * @param p_gpioId The GPIO port pin to be written
     * @param p_value The GPIO port pin value
     * @return 0 on success, -1 on wrong parameters and -2 otherwise
     */
    virtual int write(const unsigned char p_gpioId, const unsigned char p_value) = 0;
    
    virtual unsigned char createBus(const std::list<unsigned char> p_lines, const PinMode p_mode = PullNone) = 0;
    virtual void deleteBus(const unsigned char p_busId) = 0;
    virtual int busRead(const unsigned char p_busId, unsigned short * p_value) = 0;
    virtual int busWrite(const unsigned char p_busId, const unsigned short p_value) = 0;
    
protected:
    virtual bool writeRegister(const unsigned char p_registerId, const unsigned char p_value) = 0;
    virtual bool readRegister(const unsigned char p_registerId, unsigned char * p_value) = 0;

}; // End of class AbstractGpioExpender 
