/* mbed simplified access to RAMTRON FV24xx Serial 3V F-RAM Memory (I2C)
 * Copyright (c) 20103 ygarcia, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <iostream>
#include <sstream>

#include "FM24Vxx_I2C.h"

namespace _FM24VXX_I2C {

    unsigned char CFM24VXX_I2C::I2CModuleRefCounter = 0;

    CFM24VXX_I2C::CFM24VXX_I2C(const PinName p_sda, const PinName p_scl, const unsigned char p_address, const PinName p_wp, const unsigned int p_frequency) : _internalId("") {
        DEBUG_ENTER("CFM24VXX_I2C")

        if (CFM24VXX_I2C::I2CModuleRefCounter != 0) {
            error("CFM24VXX_I2C: Wrong params");
        }
#ifdef __DEBUG
        std::ostringstream out(std::ostringstream::out);
        out << "CFM24VXX_I2C #" << CFM24VXX_I2C::I2CModuleRefCounter;
        _internalId.assign(out.str());
        DEBUG("CFM24VXX_I2C: _internalId='%s'", _internalId.c_str())
#endif // __DEBUG
        _i2cInstance = new I2C(p_sda, p_scl);
        CFM24VXX_I2C::I2CModuleRefCounter += 1;
        DEBUG_ENTER("CFM24VXX_I2C: refCounter=%d", CFM24VXX_I2C::I2CModuleRefCounter)
        // Memory page select is set to 0
        _slaveAddress = (p_address << 2) | 0xa0; // Slave address format is: 1 0 1 0 A2 A1 PS R/W, PS set to 0
        DEBUG("CFM24VXX_I2C: I2C slave adress:  0x%02x", _slaveAddress)
        _i2cInstance->frequency(p_frequency); // Set the frequency of the I2C interface

        if (p_wp != NC) {
            DEBUG("CFM24VXX_I2C: WP managed");
            _wp = new DigitalOut(p_wp);
            _wp->write(0); // Disable write protect
        } else {
            DEBUG("CFM24VXX_I2C: WP not managed");
            _wp = NULL; // Not used
        }
 
        // Retrieve device identifiers
        _deviceID = NULL;
        GetDevideIDs();
        
        _sn = NULL;
        GetSerialNumbers();
    
        DEBUG_LEAVE("CFM24VXX_I2C")
    }

    CFM24VXX_I2C::~CFM24VXX_I2C() {
        DEBUG_ENTER("~CFM24VXX_I2C")
    
        // Release I2C instance
        DEBUG_ENTER("~CFM24VXX_I2C: refCounter=%d", CFM24VXX_I2C::I2CModuleRefCounter)
        CFM24VXX_I2C::I2CModuleRefCounter -= 1;
        if (CFM24VXX_I2C::I2CModuleRefCounter == 0) {
            delete _i2cInstance;
            _i2cInstance = NULL;
            if (_deviceID != NULL) {
                delete _deviceID;
                _deviceID = NULL;
            }
        }
        // Release _wp if required
        if (_wp != NULL) {
            _wp->write(0);
            delete _wp;
        }
    
        DEBUG_LEAVE("~CFM24VXX_I2C")
    }
    
    bool CFM24VXX_I2C::WriteProtect(const bool p_writeProtect) {
        if (_wp != NULL) {
            DEBUG("WP set to: %x", (int)p_writeProtect)
            _wp->write((int)(p_writeProtect));
            return true;
        }
    
        return false;
    }

    bool CFM24VXX_I2C::GetDevideIDs() {
         DEBUG_ENTER("CFM24VXX_I2C::GetDevideIDs")
    
        // 1. Memory address
        char i2cBuffer[1];
        i2cBuffer[0] = (unsigned char)(_slaveAddress & 0xfc); //FIXME Change 0xfc into a const SET_PAGE_SELECT_0
        DEBUG("CFM24VXX_I2C::GetDevideIDs: pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        // 2. Send I2C start + 0xF8 + I2C ReStart
       if (_i2cInstance->write(0xf8, i2cBuffer, 1, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::GetDevideIDs: Write F8 done")
            // 3. read data + I2C stop
            unsigned char buffer[3];
            int result = _i2cInstance->read(0xf9, (char *)buffer, 3);
            //wait(0.02);
            if (result == 0) {
                // 4. Setup the device IDs
                _deviceID = new CFM24VXX_IDs(buffer[0], buffer[1], buffer[2]);
                DEBUG("CFM24VXX_I2C::GetDevideIDs: %02x - %02x - %02x", buffer[0], buffer[1], buffer[2])
   
                DEBUG_LEAVE("CFM24VXX_I2C::GetDevideIDs: %x", (bool)(result == 0))
                return (bool)(result == 0);
            }
        }
    
        _deviceID = new CFM24VXX_IDs(0xff, 0xff, 0xff);
        DEBUG_LEAVE("CFM24VXX_I2C::GetDevideIDs (false)")
        return false;
    }

    bool CFM24VXX_I2C::GetSerialNumbers() {
        DEBUG_ENTER("CFM24VXX_I2C::GetSerialNumber")
        // 1. Memory address
        char i2cBuffer[1];
        i2cBuffer[0] = (unsigned char)(_slaveAddress & 0xfc); //FIXME Change 0xfc into a const SET_PAGE_SELECT_0
        DEBUG("CFM24VXX_I2C::GetSerialNumber: pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        // 2. Send I2C start + 0xF8 + I2C ReStart
       if (_i2cInstance->write(0xf8, i2cBuffer, 1, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::GetSerialNumber: Write F8 done")
            // 3. read data + I2C stop
            unsigned char buffer[8]; // See FM24V10_ds.pdf Page 10/16 Figure 15. 8-Byte Serial Number (read-only)
            int result = _i2cInstance->read(0xcd, (char *)buffer, 8);
            //wait(0.02);
            if (result == 0) {
                // 4. Check if it is supported
                if (buffer[7] != 0x00) { // SN supported
                    // 5. Compute CRC
                    unsigned char crc = ChecksumSN(buffer);
                    // 6. Check CRCs
                    if (buffer[7] == crc) {
                        _sn = new CFM24VXX_SN(buffer);
                        DEBUG_LEAVE("CFM24VXX_I2C::GetSerialNumber: true")
                        return true;
                     } else { // SN supported
                        DEBUG_LEAVE("CFM24VXX_I2C::GetSerialNumber: Checksum mismatch")
                        return false;
                     }
               } else { // SN supported
                    DEBUG_LEAVE("CFM24VXX_I2C::GetSerialNumber: Serial number not supported")
                    return true;
                }
            }
        }
    
        DEBUG_LEAVE("CFM24VXX_I2C::GetSerialNumber (false)")
        return false;
    }
    
    bool CFM24VXX_I2C::EraseMemoryArea(const short p_startAddress, const int p_count, const unsigned char p_pattern) {
        DEBUG_ENTER("CFM24VXX_I2C::EraseMemoryArea: 0x%02x - %d - 0x%02x", p_startAddress, p_count, p_pattern)
    
        std::vector<unsigned char> eraseBuffer(p_count, p_pattern);
        return Write(p_startAddress, eraseBuffer, false);
    }

    bool CFM24VXX_I2C::Write(const short p_address, const unsigned char p_byte) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (byte): Memory address: 0x%02x - 0x%02x", p_address, p_byte)
    
        // 1.Prepare buffer
        char i2cBuffer[3]; // Memory address + one byte of data
        // 1.1. Memory address
        short address = p_address + 1; // Index start to 1
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Write (byte): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Write (byte): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
        // 1.2. Datas
        i2cBuffer[2] = p_byte;
        DEBUG("CFM24VXX_I2C::Write (byte): value=0x%02x", i2cBuffer[2])
    
        // 2. Send I2C start + I2C address + Memory Address + Datas + I2C stop
        int result = _i2cInstance->write(_slaveAddress, i2cBuffer, 3);
        //wait(0.02);
    
        DEBUG_LEAVE("CFM24VXX_I2C::Write (byte) %x", (bool)(result == 0))
        return (bool)(result == 0);
    }
    
    bool CFM24VXX_I2C::Write(const short p_address, const short p_short, const CFM24VXX_I2C::Mode p_mode) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (short): Memory address:0x%02x, Mode:%d", p_address, p_mode)
    
        // 1.Prepare buffer
        char i2cBuffer[4]; // Memory address + one short (2 bytes)
        // 1.1. Memory address
        short address = p_address + 1; // Index start to 1
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Write (short): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Write (short): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
        // 1.2. Datas
        if (p_mode == BigEndian) {
            i2cBuffer[2] = (unsigned char)(p_short >> 8);
            i2cBuffer[3] = (unsigned char)((unsigned char)p_short & 0xff);
        } else {
            i2cBuffer[2] = (unsigned char)((unsigned char)p_short & 0xff);
            i2cBuffer[3] = (unsigned char)(p_short >> 8);
        }
        DEBUG("CFM24VXX_I2C::Write (short): value=0x%02x%02x", i2cBuffer[2], i2cBuffer[3])
    
        // 2. Send I2C start + I2C address + Memory Address + Datas + I2C stop
        int result = _i2cInstance->write(_slaveAddress, i2cBuffer, 4);
        //wait(0.02);
    
        DEBUG_LEAVE("CFM24VXX_I2C::Write (short) %x", (bool)(result == 0))
        return (bool)(result == 0);
    }
    
    bool CFM24VXX_I2C::Write(const short p_address, const int p_int, const CFM24VXX_I2C::Mode p_mode) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (int): Memory address:0x%02x, Mode:%d", p_address, p_mode)
    
        // 1.Prepare buffer
        char i2cBuffer[6]; // Memory address + one integer (4 bytes)
        // 1.1. Memory address
        short address = p_address + 1; // Index start to 1
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Write (int): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Write (int): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
        // 1.2. Datas
        if (p_mode == BigEndian) {
            i2cBuffer[2] = (unsigned char)(p_int >> 24);
            i2cBuffer[3] = (unsigned char)(p_int >> 16);
            i2cBuffer[4] = (unsigned char)(p_int >> 8);
            i2cBuffer[5] = (unsigned char)((unsigned char)p_int & 0xff);
        } else {
            i2cBuffer[2] = (unsigned char)((unsigned char)p_int & 0xff);
            i2cBuffer[3] = (unsigned char)(p_int >> 8);
            i2cBuffer[4] = (unsigned char)(p_int >> 16);
            i2cBuffer[5] = (unsigned char)(p_int >> 24);
        }
        DEBUG("CFM24VXX_I2C::Write (int): value=0x%02x%02x%02x%02x", i2cBuffer[2], i2cBuffer[3], i2cBuffer[4], i2cBuffer[5])
    
        // 2. Send I2C start + I2C address + Memory Address + Datas + I2C stop
        int result = _i2cInstance->write(_slaveAddress, i2cBuffer, 6);
        //wait(0.02);
    
        DEBUG_LEAVE("CFM24VXX_I2C::Write (int) %x", (bool)(result == 0))
        return (bool)(result == 0);
    }
    
    bool CFM24VXX_I2C::Write(const short p_address, const std::string & p_string, const bool p_storeLength, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (std::string)")
        return Write(p_address, p_string.c_str(), p_storeLength, p_length2write);
    }
    
    bool CFM24VXX_I2C::Write(const short p_address, const std::vector<unsigned char> & p_datas, const bool p_storeLength, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (std::vector)")
    
        int length = (p_length2write == -1) ? p_datas.size() : p_length2write;
        unsigned char array[length];
        std::copy(p_datas.begin(), p_datas.end(), array);
        bool result = Write(p_address, array, p_storeLength, length);
        //wait(0.02);
    
        DEBUG_LEAVE("CFM24VXX_I2C::Write (std::vector): %d", result)
        return result;
    }
    
    bool CFM24VXX_I2C::Write(const short p_address, const char *p_string, const bool p_storeLength, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (char *): Memory address: 0x%02x - %x - %d", p_address, p_storeLength, p_length2write)
    
        DEBUG("CFM24VXX_I2C::Write (char *): Slave address: %02x", _slaveAddress)
        // 1.Prepare buffer
        int length = (p_length2write == -1) ? strlen(p_string) : p_length2write;
        if (p_storeLength) {
            length += 4; // Add four bytes for the length as integer
        }
        DEBUG("CFM24VXX_I2C::Write (char *): length:%d", length)
        
        char i2cBuffer[2 + length];
        // 1.1. Memory address
        short address = p_address + 1;
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Write (char *): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Write (char *): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
        // 1.2. Datas
        if (p_storeLength) {
            // Fill the length
            i2cBuffer[2] = (unsigned char)(length >> 24);
            i2cBuffer[3] = (unsigned char)(length >> 16);
            i2cBuffer[4] = (unsigned char)(length >> 8);
            i2cBuffer[5] = (unsigned char)((unsigned char)length & 0xff);
            for (int i = 0; i < length - 4; i++) {
                i2cBuffer[6 + i] = *(p_string + i);
            }
        } else { // The length was not stored
            for (int i = 0; i < length; i++) {
                i2cBuffer[2 + i] = *(p_string + i);
            }
        }
        
        // 2. Send I2C start + I2C address + Memory Address + Datas + I2C stop
        int result = _i2cInstance->write(_slaveAddress, i2cBuffer, 2 + length);
        //wait(0.02);
    
        DEBUG_LEAVE("CFM24VXX_I2C::Write (char *) %x", (bool)(result == 0))
        return (bool)(result == 0);
    }
    
    bool CFM24VXX_I2C::Write(const short p_address, const unsigned char *p_datas, const bool p_storeLength, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Write (byte *): Memory address: 0x%02x - %x - %d", p_address, p_storeLength, p_length2write)
        return Write(p_address, (const char *)p_datas, p_storeLength, p_length2write);
    }
    
    bool CFM24VXX_I2C::Read(const short p_address, unsigned char * p_byte) {
        DEBUG_ENTER("CFM24VXX_I2C::Read (byte): Memory address:0x%02x", p_address)
    
        // 1.Prepare buffer
        char i2cBuffer[2];
        // 1.1. Memory address
        i2cBuffer[0] = (unsigned char)(p_address >> 8);
        DEBUG("CFM24VXX_I2C::Read (byte): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)p_address & 0xff);
        DEBUG("CFM24VXX_I2C::Read (byte): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
    
        // 2. Send I2C start + memory address
        if (_i2cInstance->write(_slaveAddress, i2cBuffer, 2, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (byte): Write memory done")
            // 2. Read data + I2C stop
            int result = _i2cInstance->read(_slaveAddress, (char *)p_byte, 1);
            //wait(0.02);
    
            DEBUG_LEAVE("CFM24VXX_I2C::Read (byte): %x", (bool)(result == 0))
            return (bool)(result == 0);
        }
    
        DEBUG_LEAVE("CFM24VXX_I2C::Read (byte) (false)")
        return false;
    }
    
    bool CFM24VXX_I2C::Read(const short p_address, short *p_short, const CFM24VXX_I2C::Mode p_mode) {
        DEBUG_ENTER("CFM24VXX_I2C::Read (short): Memory address:0x%02x, Mode:%d", p_address, p_mode)
    
        // 1.Prepare buffer
        char i2cBuffer[2];
        // 1.1. Memory address
        i2cBuffer[0] = (unsigned char)(p_address >> 8);
        DEBUG("CFM24VXX_I2C::Read (short): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)p_address & 0xff);
        DEBUG("CFM24VXX_I2C::Read (short): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
    
        // 2. Send I2C start + memory address
        if (_i2cInstance->write(_slaveAddress, i2cBuffer, 2, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (short): Write memory done")
            // 2. Read data + I2C stop
            int result = _i2cInstance->read(_slaveAddress, i2cBuffer, 2);
            if (result == 0) {
                DEBUG("CFM24VXX_I2C::Read (short): value: 0x%02x - 0x%02x", i2cBuffer[0], i2cBuffer[1])
                if (p_mode ==  BigEndian) {
                    *p_short = (short)(i2cBuffer[0] << 8 | i2cBuffer[1]);
                } else {
                    *p_short = (short)(i2cBuffer[1] << 8 | i2cBuffer[0]);
                }
        
                DEBUG_LEAVE("CFM24VXX_I2C::Read (short): 0x%04x", *p_short)
                return true;
            }
        }
    
        DEBUG_LEAVE("CFM24VXX_I2C::Read (short) (false)")
        return false;
    }
    
    bool CFM24VXX_I2C::Read(const short p_address, int *p_int, const CFM24VXX_I2C::Mode p_mode) {
        DEBUG_ENTER("CFM24VXX_I2C::Read (int): Memory address:0x%02x, Mode:%d", p_address, p_mode)
    
        // 1.Prepare buffer
        char i2cBuffer[4];
        // 1.1. Memory address
        i2cBuffer[0] = (unsigned char)(p_address >> 8);
        DEBUG("CFM24VXX_I2C::Read (int): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)p_address & 0xff);
        DEBUG("CFM24VXX_I2C::Read (int): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
    
        // 2. Send I2C start + memory address
        if (_i2cInstance->write(_slaveAddress, i2cBuffer, 2, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (int): Write memory done")
            // 2. Read data + I2C stop
            int result = _i2cInstance->read(_slaveAddress, i2cBuffer, 4);
            if (result == 0) {
                DEBUG("CFM24VXX_I2C::Read (int): value: 0x%02x - 0x%02x - 0x%02x - 0x%02x", i2cBuffer[0], i2cBuffer[1], i2cBuffer[2], i2cBuffer[3])
                //wait(0.02);
                if (p_mode ==  BigEndian) {
                    *p_int = (int)(i2cBuffer[0] << 24 | i2cBuffer[1] << 16 | i2cBuffer[2] << 8 | i2cBuffer[3]);
                } else {
                    *p_int = (int)(i2cBuffer[3] << 24 | i2cBuffer[2] << 16 | i2cBuffer[1] << 8 | i2cBuffer[0]);
                }
    
                DEBUG_LEAVE("CFM24VXX_I2C::Read (int): %d", *p_int)
                return true;
            }
    
            DEBUG_LEAVE("CFM24VXX_I2C::Read (int):false")
            return false;
        }
    
        DEBUG_LEAVE("CFM24VXX_I2C::Read (int) (false)")
        return false;
    }
    
    bool CFM24VXX_I2C::Read(const short p_address, std::vector<unsigned char> & p_datas, const bool p_readLengthFirst, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Read (vector): Memory address:0x%02x, readLength:%01x, Length:%d", p_address, p_readLengthFirst, p_length2write)
    
        // 1.Prepare buffer
        short address = p_address;
        int length = 0;
        if (p_readLengthFirst) {
            if (!Read(address, &length)) { // Read the length in big endian mode
                DEBUG_LEAVE("CFM24VXX_I2C::Read (vector) Failed to read length")
                return false;
            }
            DEBUG("CFM24VXX_I2C::Read (vector): length= %d", length)
            if (length == 0) {
                return true;
            }
            address += 4; // Skip the length value 
            length -= 4; // length is the size of (string length + string)
        } else {
            if (p_length2write == -1) {
                length = p_datas.size();
            } else {
                length = p_length2write;
            }
        }
        DEBUG("CFM24VXX_I2C::Read (vector): length= %d", length)
    
        // 2. Memory address
        char i2cBuffer[2];
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Read (vector): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Read (vector): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
    
        // 3. Send I2C start + memory address
        if (_i2cInstance->write(_slaveAddress, i2cBuffer, 2, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (vector): Write memory done")
            // 4. read data + I2C stop
            unsigned char buffer[length];
            int result = _i2cInstance->read(_slaveAddress, (char *)buffer, length);
            //wait(0.02);
            if (result == 0) {
                p_datas.assign(buffer, buffer + length);
    
                DEBUG_LEAVE("CFM24VXX_I2C::Read (vector): %x", (bool)(result == 0))
                return (bool)(result == 0);
            }
        }
    
        DEBUG_LEAVE("CFM24VXX_I2C::Read (vector) (false)")
        return false;
    }
    
    bool CFM24VXX_I2C::Read(const short p_address, std::string & p_string, const bool p_readLengthFirst, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Read (string): Memory address:0x%02x, readLength:%01x, Length:%d", p_address, p_readLengthFirst, p_length2write)
    
        // 1.Prepare buffer
        short address = p_address;
        int length = -1;
        if (p_readLengthFirst) { // The string was stored with its length
            if (!Read(address, &length)) { // Read the length as integer in big endian mode
                DEBUG_ERROR("CFM24VXX_I2C::Read (string): Failed to read length")
                return false;
            }
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (string): length=%d", length)
            if (length == 0) {
                DEBUG_ERROR("CFM24VXX_I2C::Read (string): empty")
                return true;
            }
            address += 4; // Skip the length value size 
            length -= 4; // length is the size of (string length + string)
        } else { // The string length is provided by p_length2write parameter
            if (p_length2write == -1) {
                length = p_string.size();
            } else {
                length = p_length2write;
                p_string.resize(p_length2write);
            }
        }
        DEBUG("CFM24VXX_I2C::Read (string): Address=0x%02x - Length=%d", address, length)
    
        // 2. Memory address
        char i2cBuffer[2];
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Read (string): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Read (string): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
    
        // 3. Send I2C start + memory address with repeat start
        if (_i2cInstance->write(_slaveAddress, i2cBuffer, 2, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (string): Write memory done")
            // 4. Read data + I2C stop
            char buffer[length];
            int result = _i2cInstance->read(_slaveAddress, (char *)buffer, length);
            if (result == 0) {
                p_string.assign(buffer, length);
        
                return true;
            }
        }
        
        DEBUG_LEAVE("CFM24VXX_I2C::Read (string) (false)")
        return false;
    }
    
    bool CFM24VXX_I2C::Read(const short p_address, char *p_string, const bool p_storeLength, const int p_length2write) {
        DEBUG_ENTER("CFM24VXX_I2C::Read (char *): Memory address:0x%02x, readLength:%01x, Length:%d", p_address, p_storeLength, p_length2write)
    
        // 1.Prepare buffer
        short address = p_address;
        int length = -1;
        if (p_storeLength) { // The string was stored with its length
            if (!Read(address, &length)) { // Read the length as integer in big endian mode
                DEBUG_ERROR("CFM24VXX_I2C::Read (char *): Failed to read length")
                return false;
            }
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (char *): length=%d", length)
            if (length == 0) {
                DEBUG_ERROR("CFM24VXX_I2C::Read (char *): empty")
                return true;
            }
            address += 4; // Skip the length value size 
            length -= 4; // length is the size of (string length + string)
        } else { // The string length is provided by p_length2write parameter
            if (p_length2write == -1) {
                DEBUG_ERROR("CFM24VXX_I2C::Read (char *): undefined length")
                return false;
            } else {
                length = p_length2write;
            }
        }
        DEBUG("CFM24VXX_I2C::Read (char *): Address=0x%02x - Length=%d", address, length)
    
        // 2. Memory address
        char i2cBuffer[2];
        i2cBuffer[0] = (unsigned char)(address >> 8);
        DEBUG("CFM24VXX_I2C::Read (char *): pI2CBuffer[0]: 0x%02x", i2cBuffer[0])
        i2cBuffer[1] = (unsigned char)((unsigned char)address & 0xff);
        DEBUG("CFM24VXX_I2C::Read (char *): pI2CBuffer[1]: 0x%02x", i2cBuffer[1])
    
        // 3. Send I2C start + memory address with repeat start
        if (_i2cInstance->write(_slaveAddress, i2cBuffer, 2, true) == 0) {
            //wait(0.02);
            DEBUG("CFM24VXX_I2C::Read (char *): Write memory done")
            // 4. Read data + I2C stop
            char buffer[length];
            int result = _i2cInstance->read(_slaveAddress, buffer, length);
            memcpy((void *)p_string, (const void *)buffer, length);

            DEBUG_LEAVE("CFM24VXX_I2C::Read (char *): %x", (bool)(result == 0))
            return (bool)(result == 0);
        }
                
        DEBUG_LEAVE("CFM24VXX_I2C::Read (char *) (false)")
        return false;
    }
    
    unsigned char CFM24VXX_I2C::ChecksumSN(const unsigned char *pdatas, const unsigned int length) {
        DEBUG_ENTER("CFM24VXX_I2C::ChecksumSN")
   
        unsigned char crctable[256] = {
            0x00, 0x07, 0x0E, 0x09, 0x1C, 0x1B, 0x12, 0x15,
            0x38, 0x3F, 0x36, 0x31, 0x24, 0x23, 0x2A, 0x2D,
            0x70, 0x77, 0x7E, 0x79, 0x6C, 0x6B, 0x62, 0x65,
            0x48, 0x4F, 0x46, 0x41, 0x54, 0x53, 0x5A, 0x5D,
            0xE0, 0xE7, 0xEE, 0xE9, 0xFC, 0xFB, 0xF2, 0xF5,
            0xD8, 0xDF, 0xD6, 0xD1, 0xC4, 0xC3, 0xCA, 0xCD,
            0x90, 0x97, 0x9E, 0x99, 0x8C, 0x8B, 0x82, 0x85,
            0xA8, 0xAF, 0xA6, 0xA1, 0xB4, 0xB3, 0xBA, 0xBD,
            0xC7, 0xC0, 0xC9, 0xCE, 0xDB, 0xDC, 0xD5, 0xD2,
            0xFF, 0xF8, 0xF1, 0xF6, 0xE3, 0xE4, 0xED, 0xEA,
            0xB7, 0xB0, 0xB9, 0xBE, 0xAB, 0xAC, 0xA5, 0xA2,
            0x8F, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9D, 0x9A,
            0x27, 0x20, 0x29, 0x2E, 0x3B, 0x3C, 0x35, 0x32,
            0x1F, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0D, 0x0A,
            0x57, 0x50, 0x59, 0x5E, 0x4B, 0x4C, 0x45, 0x42,
            0x6F, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7D, 0x7A,
            0x89, 0x8E, 0x87, 0x80, 0x95, 0x92, 0x9B, 0x9C,
            0xB1, 0xB6, 0xBF, 0xB8, 0xAD, 0xAA, 0xA3, 0xA4,
            0xF9, 0xFE, 0xF7, 0xF0, 0xE5, 0xE2, 0xEB, 0xEC,
            0xC1, 0xC6, 0xCF, 0xC8, 0xDD, 0xDA, 0xD3, 0xD4,
            0x69, 0x6E, 0x67, 0x60, 0x75, 0x72, 0x7B, 0x7C,
            0x51, 0x56, 0x5F, 0x58, 0x4D, 0x4A, 0x43, 0x44,
            0x19, 0x1E, 0x17, 0x10, 0x05, 0x02, 0x0B, 0x0C,
            0x21, 0x26, 0x2F, 0x28, 0x3D, 0x3A, 0x33, 0x34,
            0x4E, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5C, 0x5B,
            0x76, 0x71, 0x78, 0x7F, 0x6A, 0x6D, 0x64, 0x63,
            0x3E, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2C, 0x2B,
            0x06, 0x01, 0x08, 0x0F, 0x1A, 0x1D, 0x14, 0x13,
            0xAE, 0xA9, 0xA0, 0xA7, 0xB2, 0xB5, 0xBC, 0xBB,
            0x96, 0x91, 0x98, 0x9F, 0x8A, 0x8D, 0x84, 0x83,
            0xDE, 0xD9, 0xD0, 0xD7, 0xC2, 0xC5, 0xCC, 0xCB,
            0xE6, 0xE1, 0xE8, 0xEF, 0xFA, 0xFD, 0xF4, 0xF3
        };
        unsigned char crc = 0x00;
        unsigned int length_ = length;
        unsigned char *pdata = (unsigned char *)pdatas;
        while (length_-- != 0) {
            crc = crctable[crc ^ *pdata++];
            DEBUG("CFM24VXX_I2C::ChecksumSN: current checksum= 0x%02x - pdata:%08x", crc, pdata)
        }
        
        DEBUG_LEAVE("CFM24VXX_I2C::ChecksumSN: 0x%02x", crc)
        return crc;
    }
    
#if defined(__DEBUG)
    void CFM24VXX_I2C::DumpMemoryArea(const int p_address, const int p_count) {
        DEBUG_ENTER("CFM24VXX_I2C::DumpMemoryArea: %d - %d", p_address, p_count)
    
        DEBUG("CFM24VXX_I2C::DumpMemoryArea: Reading datas...");
        std::vector<unsigned char> datas(p_count);
        if (!Read(p_address, datas, false)) { // Read bytes, including the lenght indication, buffer size is not set before the call
            std::cout << "CFM24VXX_I2C::DumpMemoryArea: read failed\r" << std::endl;
        } else {
            std::cout << "CFM24VXX_I2C::DumpMemoryArea: Read bytes:\r" << std::endl;
            HEXADUMP(&datas[0], p_count);
            std::cout << "\r" << std::endl;
        }
    }
#endif // _DEBUG

} // End of namespace _FM24VXX_I2C
