#include <string>

#include "24LCxx_I2C.h"
#include "EthernetNetIf.h"
#include "NTPClient.h"

/*
 * Declare functions
 */

void AvailableIndicator(); // LED1 flashing for program while program is alive
char DisplayMenuAndGetChoice(); // Display and get the user choice
void RunStateMachine_WithStdStringValue(); // Write and read a string
void RunStateMachine_WithByteValue(); // Write and read a byte
void RunStateMachine_WithShortValue(C2424LCXX_I2C::Mode p_mode); // Write and read a short
void RunStateMachine_WithIntegerValue(C2424LCXX_I2C::Mode p_mode); // Write and read an integer
void RunStateMachine_WithStdVectorOfByteValue(); // Write and read a buffer of bytes
void RunStateMachine_WithLengthPlusStdVectorOfByteValue(); // Write and read a buffer of bytes

/*
 * Declare statics
 */
EthernetNetIf g_eth;

enum States {
    Idle, // Idle state / Read memory operation done
    Write, // Write memory operation
    Written, // Write memory operation done
    Read // Read memory operation
};
States g_state; // Process states for memory operations

DigitalOut g_availableLed(LED1); // To verify if program in running
Ticker g_available; // LED1 will flash with a period of 2s

#define __3_3V__
#ifdef __3_3V__
C2424LCXX_I2C g_myEEPROM(p28, p27, 0x01, p5, 400000); // Create an instance of the class C2424LCXX_I2C, p28: SDA, p29:SDL, p5: wired to WP input of 24LCxx, address: A3=0,A2=0,A=1, on 3.3V I2C bus
#else // __3_3V__
C2424LCXX_I2C g_myEEPROM(p28, p27, 0x07, p7, 400000); // Create an instance of the class C2424LCXX_I2C, p28: SDA, p29:SDL, p7: wired to WP input of 24LCxx, address: A3=1,A2=1,A=1, on 5V I2C bus
#endif // __3_3V__

int main() {

    // Launch available indicator
    g_available.attach(&AvailableIndicator, 2.0);

    // Initialize Ethernet
    g_eth.setup();
    
    wait(1); // Needed after Ethernet startup
   
    IpAddr ip = g_eth.getIp();
    printf("IP address: %d:%d:%d:%d\r\n", ip[0], ip[1], ip[2], ip[3]);

    // Update time
    Host ntpServer(IpAddr(), 123, "ntp.unice.fr");
    NTPClient ntp;
    ntp.setTime(ntpServer);

    g_myEEPROM.WriteProtect(false); // Disabe WP (pin 7) - See DS21203M - Clause 2.4 Write-Protect (WP)
    
    while (true) { // Interrupt driven processing
        g_state = Idle;
        switch (DisplayMenuAndGetChoice()) {
            case 'a':
                do {
                    RunStateMachine_WithStdStringValue();
                } while (g_state != Idle);
                break;
            case 'b':
                do {
                    RunStateMachine_WithByteValue();
                } while (g_state != Idle);
                break;
            case 'c':
                do {
                    RunStateMachine_WithShortValue(C2424LCXX_I2C::BigEndian);
                } while (g_state != Idle);
                break;
            case 'd':
                do {
                    RunStateMachine_WithShortValue(C2424LCXX_I2C::LittleEndian);
                } while (g_state != Idle);
                break;
            case 'e':
                do {
                    RunStateMachine_WithIntegerValue(C2424LCXX_I2C::BigEndian);
                } while (g_state != Idle);
                break;
            case 'f':
                do {
                    RunStateMachine_WithIntegerValue(C2424LCXX_I2C::LittleEndian);
                } while (g_state != Idle);
                break;
            case 'g':
                do {
                    RunStateMachine_WithStdVectorOfByteValue();
                } while (g_state != Idle);
                break;
            case 'h':
                do {
                    RunStateMachine_WithLengthPlusStdVectorOfByteValue();
                } while (g_state != Idle);
                break;
            default:
                printf("Invalid user choice\r\n");
                break;
         } // End of 'switch' statement
    } // End of 'while' statement
} // End of program - nerver reached

void AvailableIndicator()
{
    g_availableLed = !g_availableLed;
} // End of AvailableIndicator

char DisplayMenuAndGetChoice()
{
    printf("\r\n\r\n24LCxx_I2C v0.1\r\n");
    printf("\tWrite/Read with std::string:\t\t\ta\r\n");
    printf("\tWrite/Read with a byte:\t\t\t\tb\r\n");
    printf("\tWrite/Read with a short (Big Endian):\t\tc\r\n");
    printf("\tWrite/Read with a short (Little Endian):\td\r\n");
    printf("\tWrite/Read with an integer (Big Endian):\te\r\n");
    printf("\tWrite/Read with an integer (Little Endian):\tf\r\n");
    printf("\tWrite/Read with std::vector<byte>:\t\tg\r\n");
    printf("\tWrite/Read with std::vector<byte> + length:\th\r\n");
    printf("Enter your choice:\r\n");
    return getchar();
}

void RunStateMachine_WithStdStringValue()
{
    DEBUG_ENTER("RunStateMachine_WithStdStringValue")
    
    switch (g_state) {
        case Idle:
            g_state = Write;
            break;
        case Write: {
                DEBUG("Writing data...");
                time_t ctTime;
                ctTime = time(NULL);  
                std::string str("Current time is: ");
                str += ctime(&ctTime);
                str += " UTC";
                printf("RunStateMachine_WithStdStringValue: Write '%s'\r\n", str.c_str());
                if (!g_myEEPROM.Write(256, str)) { // Write the string, including the length indicator
#ifdef __DEBUG
                    DEBUG_FATAL("RunStateMachine_WithStdStringValue: write failed at address 256")
#else // __DEBUG
                    printf("RunStateMachine_WithStdStringValue: write failed at address 256\r\n");
#endif // __DEBUG
                    g_state = Idle;
                } else {
                    g_state = Written;
                }
            }
            break;
        case Written:
            g_state = Read;
            break;
        case Read: {
                DEBUG("Reading datas...");
                std::string readtext;
                if (!g_myEEPROM.Read(256, readtext)) { // Read the string, including the length indicator
#ifdef __DEBUG
                    DEBUG_FATAL("RunStateMachine_WithStdStringValue: write failed at address 256")
#else // __DEBUG
                    printf("RunStateMachine_WithStdStringValue: write failed at address 256\r\n");
#endif // __DEBUG
                } else {
                    printf("RunStateMachine_WithStdStringValue: Read:'%s'\r\n", readtext.c_str());
                }
            }
            g_state = Idle;
            break;
        default:
            printf("RunStateMachine_WithStdStringValue: Default!\r\n");
            g_state = Idle;
            break;
    }

    DEBUG_LEAVE("RunStateMachine_WithStdStringValue")
}

void RunStateMachine_WithByteValue() {
    DEBUG_ENTER("RunStateMachine_WithByteValue")
    
    switch (g_state) {
        case Idle:
            g_state = Write;
            break;
        case Write:
            DEBUG("Writing data...")
            printf("RunStateMachine_WithByteValue: Write 0xaa\r\n");
            if (!g_myEEPROM.Write(128, (unsigned char)0xaa)) {
#ifdef __DEBUG
                DEBUG_FATAL("RunStateMachine_WithByteValue: write failed at address 128")
#else // __DEBUG
                printf("RunStateMachine_WithByteValue: write failed at address 128\r\n");
#endif // __DEBUG
                g_state = Idle;
            } else {
                g_state = Written;
            }
            break;
        case Written:
            g_state = Read;
            break;
        case Read: {
                DEBUG("Reading datas...")
                unsigned char value = 0x00;
                if (!g_myEEPROM.Read(128, &value)) {
#ifdef __DEBUG
                    DEBUG_FATAL("RunStateMachine_WithByteValue: Read operation failed at address 128")
#else // __DEBUG
                    printf("RunStateMachine_WithByteValue: Read operation failed at address 128\r\n");
#endif // __DEBUG
                } else {
                    printf("RunStateMachine_WithByteValue: Read '0x%02x'\r\n", value);
                }
            }
            g_state = Idle;
            break;
        default:
            printf("RunStateMachine_WithByteValue: Default!\r\n");
            g_state = Idle;
            break;
    }

    DEBUG_LEAVE("RunStateMachine_WithByteValue")
}

void RunStateMachine_WithShortValue(C2424LCXX_I2C::Mode p_mode) {
    DEBUG_ENTER("RunStateMachine_WithShortValue")
    
    switch (g_state) {
        case Idle:
            g_state = Write;
            break;
        case Write:
            DEBUG("Writing data...")
            printf("RunStateMachine_WithShortValue: Write 0xbeef\r\n");
            if (!g_myEEPROM.Write(64, (short)0xbeef, p_mode)) { // See http://en.wikipedia.org/wiki/Hexspeak for more ideas on hexadecimal wording!!!
                DEBUG_FATAL("RunStateMachine_WithShortValue: write failed at address 64")
                g_state = Idle;
            } else {
                g_state = Written;
            }
            break;
        case Written:
            g_state = Read;
            break;
        case Read: {
                DEBUG("Reading datas...");
                short value = 0;
                if (!g_myEEPROM.Read(64, &value, p_mode)) {
#ifdef __DEBUG
                DEBUG_FATAL("RunStateMachine_WithShortValue: write failed at address 64")
#else // __DEBUG
                printf("RunStateMachine_WithShortValue: write failed at address 64\r\n");
#endif // __DEBUG
                } else {
                    printf("RunStateMachine_WithShortValue: Read '0x%04x' / '%hu'\r\n", value, (unsigned short)value);
                }
            }
            g_state = Idle;
            break;
        default:
            printf("RunStateMachine_WithShortValue: Default!\r\n");
            g_state = Idle;
            break;
    }

    DEBUG_LEAVE("RunStateMachine_WithShortValue")
}

void RunStateMachine_WithIntegerValue(C2424LCXX_I2C::Mode p_mode) {
    DEBUG_ENTER("RunStateMachine_WithIntegerValue")
    
    switch (g_state) {
        case Idle:
            g_state = Write;
            break;
        case Write:
            DEBUG("Writing data...")
            printf("RunStateMachine_WithIntegerValue: Write 0xdeaddead\r\n");
            if (!g_myEEPROM.Write(32, (int)0xdeaddead, p_mode)) {
                DEBUG_FATAL("RunStateMachine_WithIntegerValue: write failed at address 32")
                g_state = Idle;
            } else {
                g_state = Written;
            }
            break;
        case Written:
            g_state = Read;
            break;
        case Read: {
                DEBUG("Reading datas...")
                int value = 0;
                if (!g_myEEPROM.Read(32, &value, p_mode)) {
#ifdef __DEBUG
                    DEBUG_FATAL("RunStateMachine_WithIntegerValue: write failed at address 32")
#else // __DEBUG
                    printf("RunStateMachine_WithIntegerValue: write failed at address 32\r\n");
#endif // __DEBUG
                } else {
                    printf("RunStateMachine_WithIntegerValue: Read '0x%08x' / '%d'\r\n", value, (unsigned int)value);
                }
            }
            g_state = Idle;
            break;
        default:
            printf("RunStateMachine_WithIntegerValue: Default!\r\n");
            g_state = Idle;
            break;
    }

    DEBUG_LEAVE("RunStateMachine_WithIntegerValue")
}

void RunStateMachine_WithStdVectorOfByteValue() {
    DEBUG_ENTER("RunStateMachine_WithStdVectorOfByteValue")
    
    switch (g_state) {
        case Idle:
            g_state = Write;
            break;
        case Write: {
                std::vector<unsigned char> datas;
                datas.push_back(0xfe);
                datas.push_back(0xed);
                datas.push_back(0xfa);
                datas.push_back(0xce);
                DEBUG("Writing data...")
                printf("RunStateMachine_WithStdVectorOfByteValue: Write {0xfe, 0xed, 0xfa, 0xce}\r\n");
                if (!g_myEEPROM.Write(16, datas, false)) { // Write the full buffer, not including the length indication
                    DEBUG_FATAL("RunStateMachine_WithStdVectorOfByteValue: write failed at address 16")
                    g_state = Idle;
                } else {
                    g_state = Written;
                }
            }
            break;
        case Written:
            g_state = Read;
            break;
        case Read: {
                DEBUG("Reading datas...")
                std::vector<unsigned char> datas(4);
                if (!g_myEEPROM.Read(16, datas, false)) { // Read bytes, without the lenght indication, buffer size shall be set before the call
#ifdef __DEBUG
                    DEBUG_FATAL("RunStateMachine_WithStdVectorOfByteValue: write failed at address 16")
#else // __DEBUG
                    printf("RunStateMachine_WithStdVectorOfByteValue: write failed at address 16\r\n");
#endif // __DEBUG
                } else {
                    printf("RunStateMachine_WithStdVectorOfByteValue: Read {%02x, %02x, %02x, %02x}\r\n", datas[0], datas[1], datas[2], datas[3]);
                }
            }
            g_state = Idle;
            break;
        default:
            printf("RunStateMachine_WithStdVectorOfByteValue: Default!\r\n");
            g_state = Idle;
            break;
    }

    DEBUG_LEAVE("RunStateMachine_WithStdVectorOfByteValue")
}

void RunStateMachine_WithLengthPlusStdVectorOfByteValue() {
    DEBUG_ENTER("RunStateMachine_WithLengthPlusStdVectorOfByteValue")
    
    switch (g_state) {
        case Idle:
            g_state = Write;
            break;
        case Write: {
                DEBUG("Writing data...")
                std::vector<unsigned char> datas;
                datas.push_back(0xde);
                datas.push_back(0x5e);
                datas.push_back(0xa5);
                datas.push_back(0xed);
                printf("RunStateMachine_WithLengthPlusStdVectorOfByteValue: Write {0xde, 0x5e, 0xa5, 0xed}\r\n");
                if (!g_myEEPROM.Write(8, datas)) { // Write the full buffer, including the length indication
                    DEBUG_FATAL("RunStateMachine_WithLengthPlusStdVectorOfByteValue: write failed at address 8")
                    g_state = Idle;
                } else {
                    g_state = Written;
                }
            }
            break;
        case Written:
            g_state = Read;
            break;
        case Read: {
                DEBUG("Reading datas...")
                std::vector<unsigned char> datas;
                if (!g_myEEPROM.Read(8, datas)) { // Read bytes, including the lenght indication, buffer size is not set before the call
#ifdef __DEBUG
                    DEBUG_FATAL("RunStateMachine_WithStdVectorOfByteValue: write failed at address 8")
#else // __DEBUG
                    printf("RunStateMachine_WithStdVectorOfByteValue: write failed at address 8\r\n");
#endif // __DEBUG
                } else {
                    printf("RunStateMachine_WithLengthPlusStdVectorOfByteValue: Read bytes: ");
                    vector<unsigned char>::iterator it;
                    for (it = datas.begin() ; it < datas.end(); it++) {
                        printf("0x%02x ", *it);
                    }
                    printf("\r\n");
                }
            }
            g_state = Idle;
            break;
        default:
            printf("RunStateMachine_WithLengthPlusStdVectorOfByteValue: Default!\r\n");
            g_state = Idle;
            break;
    }

    DEBUG_LEAVE("RunStateMachine_WithLengthPlusStdVectorOfByteValue")
}
