#include "mbed.h"

// I/Oピン
DigitalOut eps_on(p21);

// EPS用
SPI spi_a(p5,p6,p7);  //mosi,miso,sclk
DigitalOut cs_eps(p8);
DigitalOut cs_com(p9);

// メモリ用
SPI spi(p11,p12,p13);  //mosi,miso,sclk
DigitalOut cs(p14);

// 諸設定
Ticker IR_getHK;
Ticker IR_saveHK;
Ticker IR_HKtoSD;

Serial pc(USBTX,USBRX);

// **** グローバル変数 ******
uint8_t eps_survival_hk[256];
uint8_t eps_HK_IR[256];
uint8_t i_HK_IR=0;   // eps_HK_IRのカウンター
uint16_t IRHK_length = 0;  // データ長さ（byte）

uint8_t seq_flag = 0;
uint8_t EPS_cmd = 0; 
uint16_t HK_length = 0;  // HKのデータ数．1Hz割込で使用
// ************************


// ************ Flashメモリ設定 *************
/*Flashのコマンドを発行*/
#define CMD_RDID 0x9f   // 
#define CMD_READ 0x03 // Address Read Data
#define CMD_WREN 0x06 // Write Enable
#define CMD_SE 0x20  // Sector Erase
#define CMD_HBE 0x52;    // Block Erase
#define CMD_BE 0xD8;    // Block Erase
#define CMD_PP 0x02  // Page Programming
#define CMD_RDSR1 0x05  // Read Status Resistor (WIP check)
// *****************************************


// **** Flashメモリ用 グローバル変数*****
uint32_t addr = 0x0;    // survival_HK用
uint32_t IRHK_addr = 0x000F000;    // survival_HK用
uint8_t tx_buf[256];    // 送信バッファ用意
uint8_t rx_buf[256];    // 受信バッファ用意
// ***********************************

// ************ Flashメモリ用関数 *************
void spi_init(){   /*spi通信設定*/
     spi.format(8,0);
     spi.frequency(1000000); // 1,000,000=1MHz clock rate
     cs = 1; //初期化
     }

void spi_transfer(const uint8_t* tx_buf, uint8_t* rx_buf, uint16_t length) {   // 送受信両方判断し通信実行する関数
    //pc.printf("#spi_transfer length= 0x%X\r\n",length);
    for(uint16_t i = 0 ; i<length ; i++){
        uint8_t data = spi.write(tx_buf != NULL ? tx_buf[i] : 0x0);  // txDataが有るならtxDataを送る，無いなら0を送る
        if(rx_buf != NULL){                                         // もし受信データを表示したい場合
            rx_buf[i] = data;
            }
        }
    }
 void spi_read_data(uint8_t* rx_buf, uint16_t length){  /*受信*/
    spi_transfer(NULL, rx_buf, length);
    }
 void spi_write_data(const uint8_t* tx_buf, uint16_t length){  /*受信*/
    spi_transfer(tx_buf, NULL, length);
    }
    
/*Flash関数*/ 
void flash_send_addr(uint32_t addr) {
    const uint8_t data[] = {
         (addr >> 16) & 0xff,   // 1のところだけ残す
         (addr >> 8) & 0xff,
         (addr >> 0) & 0xff,
         };
    for(int i=0; i<3; i++){ // addr : 3byte
        spi.write(data[i]);
        }
     }
     
void flash_rdid(uint8_t* rx_buf, uint16_t length){  /*RDID読み出し*/   
    cs = 0;
    spi.write(CMD_RDID);
    spi_read_data(rx_buf, length);
    cs = 1;
    }

void flash_read(uint32_t addr, uint8_t* rx_buf, uint16_t length) {  /*読み出し*/
    cs = 0;
    spi.write(CMD_READ);
    flash_send_addr(addr);
    spi_read_data(rx_buf, length);
    cs = 1;
    }

void flash_wren(){  // write enable
    cs = 0;
    spi.write(CMD_WREN);
    cs = 1;
    }    

void flash_se(uint32_t addr) {  // sector erase
    cs = 0;
    spi.write(CMD_SE);
    flash_send_addr(addr);
    cs = 1;
    }

void flash_pp(uint32_t addr, const uint8_t* wr_data, uint16_t length) {   // Page Program
    cs = 0;
    spi.write(CMD_PP);
    flash_send_addr(addr);
    spi_write_data(wr_data, length);
    cs = 1;
    }

void flash_rdsr(uint8_t* rx_buf, uint16_t length) {
    cs = 0;
    spi.write(CMD_RDSR1);
    spi_read_data(rx_buf, length);  // rx_bufにRDSRの値が入っている
    cs = 1;
    }

uint8_t flash_is_wip() { // (WIP)Write In Progress(書込み中)かどうか確認
    uint8_t data;
    flash_rdsr(&data, 0x01);    // 1byte（StatusRegister情報）
    return(data & 0x01);    //WIP // ビット演算_&
    }
    
    
void data_debug_print(const uint8_t* data, uint16_t bytes){ //配列を表示
    pc.printf("+0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +A +B +C +D +E +F\r\n");
    pc.printf("-----------------------------------------------\r\n");
    uint16_t depth = ((bytes - 1) >> 4) + 1;
    for(uint16_t j = 0 ; j < depth ; ++j){      //depthは何?
        for(uint16_t i = 0 ; i < 16 ; ++i){
            uint8_t d = data[i + (j << 4)];
            pc.printf("%X", (d >> 4) & 0x0f);
            pc.printf("%X", (d >> 0) & 0x0f);
            pc.printf(" ");
            }
        pc.printf("\r\n");
        }
    pc.printf("\r\n");
    }
// ******************************************Flashメモリ用関数ここまで

// ************* 割り込み関数 ***************
void getHK_eps_IR() { // 1Hz getHK 割り込み関数
    pc.printf("get EPS 1Hz HK\r\n");
    EPS_cmd = 2;
    
    cs_eps = 0;
    //pc.printf("send cmd to eps\r\n");
    spi_a.write(EPS_cmd); // send command
    
    HK_length = 4;  // 1回で取得するHKデータのbyte数
    for(uint8_t i=0; i < HK_length; i++) {    // get data
        eps_HK_IR[i_HK_IR] = spi_a.write(0x0);
        pc.printf("get eps 1Hz hk: %d\r\n",eps_HK_IR[i_HK_IR]);
        i_HK_IR++;
        }
    cs_eps = 1;
    }

void flash_saveHK_eps_IR() {    // 1分毎にFlashにIRHKを保存
    
    // eps_HK_IR[]をフラッシュに保存
    flash_wren();   // 書込み・消去の前に必ず必要
    IRHK_length = 4*60;  // データ長さ（byte）//
    flash_pp(IRHK_addr, eps_HK_IR, IRHK_length);
    while(flash_is_wip()){
        }
    pc.printf("Flash Write complete! \r\n");
    pc.printf("Write Data check (Read the Written address.)\r\n");
    flash_read(IRHK_addr, rx_buf, IRHK_length);
    data_debug_print(rx_buf, IRHK_length);
    IRHK_addr += IRHK_length;
    }

void sendHK_from_Flash_to_SD() {
    // Flashから読み出し(IRHK_addr-180*4)
    pc.printf("send to sd\r\n");
    flash_read(IRHK_addr-(IRHK_length*3), rx_buf, IRHK_length*3);
    data_debug_print(rx_buf, IRHK_length*3);
    
    // send to com
    cs_com = 0;
    spi.write(0x10);
    cs_com = 1;
    
    wait(1);
    
    uint8_t rep;
    for(uint8_t i=0; i<IRHK_length*3; i++) {    // get data
        cs_com = 0;
        spi.write(rx_buf[i]); // 1byte HK 送る
        while(rep!=0x01){   // ビジー確認．rep=1になるまで，永遠にダミーデータを送る
            rep = spi.write(0x0);
            }
        cs_com = 1;
        }
        
    // Flash転送終了コマンド
    spi.write(0xFF);
    
    }
// *****************************************割り込み関数定義終了


// Sequence 関数
void auto_sequence(uint8_t cmd) {
    if(cmd==0) {    // Flashメモリ全消去
        /*flash_wren();   // 書込み・消去の前に必ず必要
        flash_se(addr);
        while(flash_is_wip()){  // 消去待ち(WIPが 1 の間)  // While文が1周回る毎にflash_is_wipを実行している？
            }*/
        }
    if(cmd==1) {
        pc.printf("EPS ON \r\n");
        eps_on = 1;
        wait(0.5);
        eps_on = 0;
        }
    
    else if(cmd==2) {
        
        pc.printf("EPS ack check \r\n");
        EPS_cmd = 1;
        
        cs_eps = 0;
        uint8_t rdata = spi_a.write(EPS_cmd); // send command
        cs_eps = 1;
        
        pc.printf("eps ack: %d\r\n",rdata);
        }
            
    else if(cmd==3) { // eps Survival Check
        EPS_cmd = 2;
        
        pc.printf("get eps Survival hk\r\n");
        cs_eps = 0;        
        for(uint8_t i=0; i<5; i++) {    // get data
            spi_a.write(EPS_cmd); // send command
            //pc.printf("send cmd to eps\r\n");
            
            wait(0.5);    // epsのHK取得待ち時間（仮）
            
            eps_survival_hk[i] = spi_a.write(0x0);
            pc.printf("eps Survival hk: %d\r\n",eps_survival_hk[i]);
            wait(0.5);    // epsのHK取得待ち時間（仮）
            }
        cs_eps = 1;
        
        // flag更新
        seq_flag = 1;
        }
        
    else if(cmd==3) {
        // eps_survival_hkをフラッシュに保存
        flash_wren();
        addr = 0x0; // Flashメモリのデータを入れる場所指定
        HK_length = 2;  // データ長さ（byte）
        flash_pp(addr, eps_survival_hk, HK_length);
        while(flash_is_wip()){
            }
        pc.printf("Flash Write complete! \r\n");
        pc.printf("Write Data check (Read the Written address.)\r\n");
        flash_read(addr, rx_buf, HK_length);
        data_debug_print(rx_buf, HK_length);
        }
    
    else if(cmd==4) {
        pc.printf("COM ON \r\n");
        EPS_cmd = 3;
        cs_eps = 0;
        spi_a.write(EPS_cmd); // send command
        cs_eps = 1;        
        }
    else if(cmd==5) { // 1Hz HK取得開始
    
        // 1Hz getHK 割り込み開始
        IR_getHK.attach(&getHK_eps_IR, 10);    // 1HzでgetHK開始（0.1Hzでテスト）
        seq_flag = 3;
        pc.printf("start 1Hz getHK Interrupt!!\r\n");
        pc.printf("seq_flag: %d\r\n", seq_flag);
        }

    else if(cmd==6) {   // 1分毎にFlashに保存 開始
        
        IR_saveHK.attach(&flash_saveHK_eps_IR, 60);
        seq_flag = 4;
        pc.printf("start save HK to Flash memory Interrupt!!\r\n");
        pc.printf("seq_flag: %d\r\n", seq_flag);
        }
        
    else if(cmd==7) {
        // SP deployment
        }


    else if(cmd==8) {   // SD転送割り込みスタート
        IR_HKtoSD.attach(&sendHK_from_Flash_to_SD, 180);    // 1hで割り込み（今回は120s）
        seq_flag = 5;
        pc.printf("start send HK from Flash memory to SD!!\r\n");
        pc.printf("seq_flag: %d\r\n", seq_flag);
        }

//    else if(cmd==) {
//        }
    if(cmd == 99) {
        // 1Hz getHK 割り込み終了
        IR_getHK.detach();
        IR_saveHK.detach();
        }    
    }
    

int main() {
    uint8_t cmd = 0
    ;
    // ***Flashメモリ用ローカル変数・初期化******
    //uint32_t addr = 0x0;    // 3byte (or 4byte)
    uint16_t length = 0x100;    // 例だと1Page=256
    spi_init(); // spi初期設定
    // ***************************************
    
    pc.printf("CDH auto sequence start!!\r\n\r\n");
    
    while(cmd<2) {
        
        pc.printf(">>cmd %d: ", cmd);
        auto_sequence(cmd);            
            
        cmd++;
        wait(1);
    }
}
