#include "mbed.h"

#ifndef LCD_SPLC792A_I2C
#define LCD_SPLC792A_I2C

/* USER SETTINGS =================== */
#define LCD_COLS                        16          //16 digits
#define LCD_ROWS                        2           //2 Line
#define LCD_LOOP_MODE                   2
    //LCD_LOOP_MODE
    // 0: カーソルが16桁よりはみ出しても、そのまま出力し続けます(はみ出した文字は見えません)
    // 1: カーソルが16桁に達した場合、同じ行の先頭に戻り、出力を継続します
    // 2: カーソルが16桁に達した場合、別の行の先頭に移動し、出力を継続します

#define SPLC792A_DEFAULT_CONTRAST       45          //6bit Contrast(0 to 63)
    //SPLC792A_DEFAULT_CONTRAST
    // 液晶の初期化時に使用される表示濃度を指定
    // 表示濃度はいつでも変更することが可能です(setContrastを使用)

#define SPLC792A_I2C_SLAVEADDR          0x3E        //7bit I2C slave address
    //SPLC792A_I2C_SLAVEADDR
    // I2Cで使用するスレーブアドレスです Arduinoなどで使用される7bitのものを指定
    // mbedはスレーブアドレスが8bitですが、プログラム内で1bit左シフトしていますので問題ありません
    // aitendoで販売されているすべてのSPLC792A液晶は0x3Eだと思います

#define SPLC792A_I2C_FREQ               100000      //Min:80,000Hz - Max:400,000Hz
    //SPLC792A_I2C_FREQ
    // I2Cの規格によると、100KHzがノーマルモード、400KHzがハイスピードモードです
    // それ以外の周波数は記載されていませんが、動作することは確認しています
    // プルアップ抵抗の値は重要です 10K-ohmで80KHz - 300KHzでの動作を確認しています
/* ================================= */

class LCDI2C_SPLC792A{
private:
    I2C *lcd;

    char lcdCols;
    char lcdRows;
    char nowCol;
    char nowRow;
    char loopMode;
    char nowContrast;
    char nowVoltage;
    char box[2];
    char i2c_addr;

    void sendCmd(char Lbyte);
    void sendVal(char Lbyte);

public:
    /** Create a SPLC792A instance connected to I2C pins
    * default LCD cols = 16, rows = 2
    *
    * @param i2c_SDA I2C-bus SDA pin
    * @param i2c_SCL I2C-bus SCL pin
    * @param lcd_LOOP LCD display loop-mode
    *   0: Continue to output, If it overflowing 16 digits.(The overflowing characters disappear)
    *   1: Back to the beginning of the same line, If it overflowing 16 digits
    *   2: Back to the beginning of the different line, If it overflowing 16 digits
    */
    LCDI2C_SPLC792A(PinName i2c_SDA, PinName i2c_SCL, unsigned char lcd_LOOP=LCD_LOOP_MODE);

    /** Create a SPLC792A instance connected to I2C pins
    *
    * @param lcd_Cols LCD display cols
    * @param lcd_Rows LCD display rows
    * @param i2c_SDA I2C-bus SDA pin
    * @param i2c_SCL I2C-bus SCL pin
    * @param lcd_LOOP LCD display loop-mode
    *   0: Continue to output, If it overflowing 16 digits.(The overflowing characters disappear)
    *   1: Back to the beginning of the same line, If it overflowing 16 digits
    *   2: Back to the beginning of the different line, If it overflowing 16 digits
    */
    LCDI2C_SPLC792A(unsigned char lcd_Cols, unsigned char lcd_Rows, PinName i2c_SDA, PinName i2c_SCL, unsigned char lcd_LOOP=LCD_LOOP_MODE);

    /** Destructor
    */
    ~LCDI2C_SPLC792A();

    /** LCD initialize
    */
    void init();

    /** LCD clear (すべての表示を消すが、カーソル位置は保持する)
    */
    void clear();

    /** LCD clear : alias clear() (すべての表示を消すが、カーソル位置は保持する)
    */
    void cls();

    /** LCD Homing of the cursor
    */
    void home();

    /** LCD re-initialize : alias init()
    */
    void reset();

    /** Display single-character
    *
    * @param characterCode display single-character code
    */
    void setCharCode(const char characterCode);

    /** Display contrast setting
    *
    * @param voltage_0to7 Voltage Regulator amplified ratio
    *   0: 1.818V
    *   1: 2.222V
    *   2: 2.667V
    *   3: 3.333V (SPLC792A default)
    *   4: 3.636V (This Library - initialize default)
    *   5: 4.000V
    *   6: 4.444V
    *   7: 5.000V
    *
    * @param contrast_0to64 Contrast adjust fot internal Voltage Regulator
    *   38: (default?)
    *   45: best!!(3.636V)
    */
    void setContrast(char voltage_0to7, char contrast_0to64);

    /** LCD cursor set
    *
    * @param cursorAddr LCD cursor address
    *   0x00 - 0x27 (Row 1)
    *   0x40 - 0x67 (Row 2)
    */
    void setCursorfromAddr(char cursorAddr);

    /** LCD cursor set (from Row,Col)
    *
    * @param col LCD column 1 to 16
    * @param row LCD row 1 or 2
    */
    void setCursorfromColRow(unsigned char col, unsigned char row);

    /** LCD cursor set: alias setCursorfromColRow
    *
    * @param x LCD column 1 to 16
    * @param y LCD row 1 or 2
    */
    void setCursor(unsigned char x, unsigned char y);

    /** Display mode set
    *
    * @param display_on 0(false):Display Off, 1(true):Display On
    * @param cursor_underbar_on 0(false):cursor underbar Off, 1(true):cursor underbar On
    * @param blink_on 0(false):cursor blinking Off, 1(true):cursor blinking On
    */
    void setDisplay(bool display_on, bool cursor_underbar_on, bool blink_on);

    /** Entry mode set
    *
    * @param shift_on 0(false):shift Off, 1(true):shift On
    * @param isIncrement
    *   0(false):Decrement cursor(printing RTL)
    *   1(true):Inclement cursor(printing LTR)
    */
    void setEntryMode(bool shift_on, bool isIncrement);

    /** Entry mode set (よくわからない)
    *
    * @param isTwoLineLCD I do not know
    * @param isDoubleHeightMode I do not know
    */
    void setFunction(bool isTwoLineLCD, bool isDoubleHeightMode);

    /** Entry(shift?) mode set (よくわからない)
    *
    * @param cursor_shift_on I do not know
    * @param display_shift_on I do not know
    */
    void setShift(bool cursor_shift_on, bool display_shift_on);

    /** Print String
    *
    * @param *str string characters
    *   example:
    *
    *       char str[]="this is str.";
    *       lcd.writeString(str);
    *
    *   標準関数のprintfのような使い方をしたい場合は、sprintfを使ってください
    *   If you want to use, such as "printf", use "sprintf"
    *
    *       int my_No = 5;
    *       char str[20];
    *       sprtinf(str, "my number is %d", my_No);
    *       lcd.writeString(str);
    *
    * @param dispWaitTime_ms delay time
    *   1文字表示するごとに、何ミリ秒休むかを指定します
    *   空白 ' '(Blank-space)は2倍の時間休みます
    *   Consume mili-second to 1 character display
    *   character ' '(Blank-space): Consume (mili-second * 2) to 1 character display
    */
    void writeString(const char *str, unsigned short dispWaitTime_ms=0);

    /** Print String : alias writeString
    * @param *str string characters
    * @param dispWaitTime_ms delay time
    */
    void print(const char *str, unsigned short dispWaitTime_ms=0);

    /** Print String : alias writeString
    * @param *str string characters
    * @param dispWaitTime_ms delay time
    */
    void write(const char *str, unsigned short dispWaitTime_ms=0);

    /** Print String : alias writeString
    * @param *str string characters
    * @param dispWaitTime_ms delay time
    */
    void o(const char *str, unsigned short dispWaitTime_ms=0);


    //get functions
    char getCol();
    char getContrast();
    char getRow();
    char getVoltage0to7();
    float getVoltageValue();

};

#endif //LCD_SPLC792A_I2C
