#include "SPLC792A_I2C.h"

LCDI2C_SPLC792A::LCDI2C_SPLC792A(PinName i2c_SDA, PinName i2c_SCL, unsigned char lcd_LOOP){
    lcdCols = LCD_COLS;
    lcdRows = LCD_ROWS;

    lcd = new I2C(i2c_SDA, i2c_SCL);

    loopMode = (lcd_LOOP > 2) ? 2 : lcd_LOOP;

    box[0] = 0;
    box[1] = 0;
    i2c_addr = SPLC792A_I2C_SLAVEADDR << 1; //mbedはスレーブアドレスが8bitなので左シフトする
    wait_ms(50);
    lcd->frequency(SPLC792A_I2C_FREQ);
}

LCDI2C_SPLC792A::LCDI2C_SPLC792A(unsigned char lcd_Cols, unsigned char lcd_Rows, PinName i2c_SDA, PinName i2c_SCL, unsigned char lcd_LOOP){
    lcdCols = lcd_Cols;
    lcdRows = lcd_Rows;

    lcd = new I2C(i2c_SDA, i2c_SCL);

    loopMode = (lcd_LOOP > 2) ? 2 : lcd_LOOP;

    box[0] = 0;
    box[1] = 0;
    i2c_addr = SPLC792A_I2C_SLAVEADDR << 1; //mbedはスレーブアドレスが8bitなので左シフトする
    wait_ms(50);
    lcd->frequency(SPLC792A_I2C_FREQ);
}

LCDI2C_SPLC792A::~LCDI2C_SPLC792A(){
    delete lcd;
}

void LCDI2C_SPLC792A::sendCmd(char Lbyte){
    //A0 bit = 0 then Command
    box[0] = 0x00;
    box[1] = Lbyte;
    lcd->write(i2c_addr, box, 2);
}

void LCDI2C_SPLC792A::sendVal(char Lbyte){
    //A0 bit = 1 then Data Value
    box[0] = 0x40;
    box[1] = Lbyte;
    lcd->write(i2c_addr, box, 2);
}

void LCDI2C_SPLC792A::init(){
    //カーソルカウンタを初期化
    nowRow = 1;
    nowCol = 1;
    //各モードの初期化
    sendCmd(0x3F); //Data-transfer 8bit mode, Two-Lines Display
    wait_us(15);
    sendCmd(0x39); //DoubleHeightMode = false, 5x8 dots x Two-Lines
    wait_us(15);
    nowContrast = SPLC792A_DEFAULT_CONTRAST;
    sendCmd(0x50 | ((nowContrast & 30) >> 4));  //LCD Contrast setting High-2bit
    sendCmd(0x70 | (nowContrast & 0x0F));   //LCD Contrast setting Low-4bit
    sendCmd(0x60 | 0x04);   //Set LCD voltage: 3.636V(default 3.333V)
    wait_us(15);
    sendCmd(0x0D); //Display ON, Blink ON, Cursor OFF
    sendCmd(0x01); //Clear Display
}

void LCDI2C_SPLC792A::reset(){
    init();
    wait_ms(5);
}

void LCDI2C_SPLC792A::cls(){
    sendCmd(0x01);
    wait_ms(5);
}

void LCDI2C_SPLC792A::clear(){
    cls();
}

void LCDI2C_SPLC792A::home(){
    nowRow = 1;
    nowCol = 1;
    sendCmd(0x02);
    wait_ms(5);
}

void LCDI2C_SPLC792A::setCharCode(const char characterCode){
    //ASCII code (e.g. A = 0x41, z = 0x7a)
    sendVal(characterCode);
    nowCol++;

    //画面右端にカーソルが来ていたら
    if(nowCol > lcdCols && loopMode != 0){
        if(loopMode == 1){  //モード1:同じ行の先頭に戻る
            nowCol = 1;
            setCursorfromAddr((nowRow == 1)?0x00:0x40);
        }else if(loopMode == 2){ //モード2:別の行の先頭に戻る
            nowRow = (nowRow%LCD_ROWS)+1;
            nowCol = 1;
            setCursorfromAddr((nowRow == 1)?0x00:0x40);
        }
    }
}

void LCDI2C_SPLC792A::writeString(const char *str, unsigned short dispWaitTime_ms){
    while(*str){
        setCharCode(*str);
        str++;
        if(dispWaitTime_ms < 1) continue;
        wait_ms((*str == ' ') ? dispWaitTime_ms << 1 : dispWaitTime_ms);
    }
}

void LCDI2C_SPLC792A::print(const char *str, unsigned short dispWaitTime_ms){
    writeString(str, dispWaitTime_ms);
}

void LCDI2C_SPLC792A::write(const char *str, unsigned short dispWaitTime_ms){
    writeString(str, dispWaitTime_ms);
}

void LCDI2C_SPLC792A::o(const char *str, unsigned short dispWaitTime_ms){
    writeString(str, dispWaitTime_ms);
}

void LCDI2C_SPLC792A::setCursorfromAddr(char cursorAddr){
    if(cursorAddr < 0x00 || cursorAddr > 0x67) cursorAddr = 0x67;
    if(cursorAddr > 0x27 && cursorAddr < 0x40) cursorAddr = 0x67;
    sendCmd(0x80 | cursorAddr);
    wait_us(20);
}

void LCDI2C_SPLC792A::setCursorfromColRow(unsigned char col, unsigned char row){
    col = (col == 0) ? 1 : (col > lcdCols) ? lcdCols : col;
    row = (row == 0) ? 1 : (row > lcdRows) ? lcdRows : row;
    nowCol = col;
    nowRow = row;
    setCursorfromAddr(((row == 1) ? 0x00 : 0x40) | (col - 1));
}

void LCDI2C_SPLC792A::setCursor(unsigned char x, unsigned char y){
    setCursorfromColRow(x, y);
}

void LCDI2C_SPLC792A::setDisplay(bool display_on, bool cursor_underbar_on, bool blink_on){
    sendCmd(0x08 | (display_on?4:0 | cursor_underbar_on?2:0 | blink_on?1:0));
    wait_us(20);
}

void LCDI2C_SPLC792A::setEntryMode(bool shift_on, bool isIncrement){
    sendCmd(0x04 | shift_on?2:0 | isIncrement?1:0);
    wait_us(20);
}

void LCDI2C_SPLC792A::setFunction(bool isTwoLineLCD, bool isDoubleHeightMode){
    sendCmd(0x20 | isDoubleHeightMode?16:0 | isTwoLineLCD?8:0);
    wait_us(15);
    sendCmd(0x21 | isDoubleHeightMode?16:0 | isTwoLineLCD?8:0);
    wait_us(15);
}

void LCDI2C_SPLC792A::setShift(bool cursor_shift_on, bool display_shift_on){
    sendCmd(0x01 | cursor_shift_on?8:0 | display_shift_on?4:0);
}

void LCDI2C_SPLC792A::setContrast(char voltage_0to7, char contrast_0to64){
    if(voltage_0to7 < 0 || voltage_0to7 > 7) voltage_0to7 = 4;
    if(contrast_0to64 < 0 || contrast_0to64 > 64) contrast_0to64 = SPLC792A_DEFAULT_CONTRAST;

    sendCmd(0x50 | ((contrast_0to64 & 30) >> 4));   //LCD Contrast setting High-2bit
    sendCmd(0x70 | (contrast_0to64 & 0x0F));    //LCD Contrast setting Low-4bit
    sendCmd(0x60 | voltage_0to7);   //Set LCD voltage

    nowContrast = contrast_0to64;
    nowVoltage = voltage_0to7;
}

char LCDI2C_SPLC792A::getRow(){
    return nowRow;
}

char LCDI2C_SPLC792A::getCol(){
    return nowCol;
}

char LCDI2C_SPLC792A::getContrast(){
    return nowContrast;
}

char LCDI2C_SPLC792A::getVoltage0to7(){
    return nowVoltage;
}

float LCDI2C_SPLC792A::getVoltageValue(){
    float val = 0.0f;
    switch(nowVoltage){
        case 0:
            val = 1.818f;
            break;
        case 1:
            val = 2.222f;
            break;
        case 2:
            val = 2.667f;
            break;
        case 3:
            val = 3.333f;
            break;
        case 4:
            val = 3.636f;
            break;
        case 5:
            val = 4.000f;
            break;
        case 6:
            val = 4.444f;
            break;
        case 7:
            val = 5.000f;
            break;
    }
    return val;
}
