/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "MyUARTService.h"

#define NEED_CONSOLE_OUTPUT 0 /* Set this if you need debug messages on the console; \ \ \ \
                               * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
#define DEBUG(STR)                                   \
    {                                                \
        if (uartServicePtr)                          \
            uartServicePtr->write(STR, strlen(STR)); \
    }
#else
#define DEBUG(...) /* nothing */
#endif             /* #if NEED_CONSOLE_OUTPUT */

DigitalOut led1(LED1, 1);
DigitalOut led2(LED2, 1);

const static char DEVICE_NAME[] = "BLE UART";

static uint8_t hrmCounter = 100; // init HRM to 100bps
static MyUARTService *uartServicePtr;

Serial serial(USBTX, USBRX);

#define RING_BUF_SIZE 256
class RingBuffer
{
  public:
    RingBuffer() : read_idx(0), write_idx(0)
    {
    }
    void clear()
    {
        read_idx = 0;
        write_idx = 0;
    }
    int putc(int c)
    {
        if (((write_idx + 1) % RING_BUF_SIZE) == read_idx)
        {
            //  over flow
            return EOF;
        }
        else
        {
            buffer[write_idx] = c;
            write_idx++;
            write_idx %= RING_BUF_SIZE;
            return 1;
        }
    }

    int getc()
    {
        if (read_idx == write_idx)
        {
            return EOF;
        }

        int ret = buffer[read_idx];
        read_idx++;
        read_idx %= RING_BUF_SIZE;
        return ret;
    }

  private:
    uint8_t buffer[RING_BUF_SIZE];
    uint32_t read_idx;
    uint32_t write_idx;
};

RingBuffer buf;
unsigned char serial_buf[256];
unsigned char send_buf[256];
int serial_buf_idx = 0;

void onDataWrittenCallback(const GattWriteCallbackParams *params)
{
    //  backup receive data to ring buffer
    //DEBUG("BLE onDataWritten!\n\r");
    //    char res[] ={ 0x01, 0x03, 0x04, 0x00, 0x7f, 0x00, 0x01, 0x0a, 0x2b };
    //    if (uart) uart->write(res, 9);

    if (params->handle == uartServicePtr->getTXCharacteristicHandle())
    {
        uint16_t bytesRead = params->len;
        for (int i = 0; i < bytesRead; i++)
        {
            led2 = 1;
            buf.putc(params->data[i]);
        }
    }
    else
    {
        // do nothing.
    }
    led2 = 0;
}

void writeToSerial()
{
    int c;

    while ((c = buf.getc()) != EOF)
    {
        led1 = 1;
        serial.putc(c);
    }
    led1 = 0;
}
void readFromSerial()
{
    int c;
    while (serial.readable())
    {
        led1 = 1;
        c = serial.getc();
        serial_buf[serial_buf_idx++] = c;
    }
    led1 = 0;
}
void sendToBleUart()
{
    if (BLE::Instance().getGapState().connected)
    {
        if (serial_buf_idx > 0)
        {
            led2 = 1;
            uartServicePtr->write(serial_buf, serial_buf_idx);
            serial_buf_idx = 0;
        }
    }
    led2 = 0;
}
void mainProc()
{
    writeToSerial();
    readFromSerial();
    sendToBleUart();
}
static EventQueue eventQueue(
    /* event count */ 16 * /* event size */ 32);

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising(); // restart advertising
}


void periodicCallback(void)
{
    // led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */

    // if (BLE::Instance().getGapState().connected)
    // {
    //     // eventQueue.call(updateSensorValue);
    //     eventQueue.call(writeToSerial);
    //     eventQueue.call(readFromSerial);
    //     eventQueue.call(sendToBleUart);
    // }
    // else
    // {
    //     buf.clear();
    //     eventQueue.call(writeToSerial);
    //     eventQueue.call(readFromSerial);
    // }
    // eventQueue.call(mainProc);
}

void onBleInitError(BLE &ble, ble_error_t error)
{
    (void)ble;
    (void)error;
    /* Initialization error handling should go here */
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE &ble = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE)
    {
        onBleInitError(ble, error);
        return;
    }

    if (ble.getInstanceID() != BLE::DEFAULT_INSTANCE)
    {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    uartServicePtr = new MyUARTService(ble);

    ble.gattServer().onDataWritten(onDataWrittenCallback);

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME) - 1);

    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS, (uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.gap().setAdvertisingInterval(100); /* 100ms */
    ble.gap().startAdvertising();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext *context)
{
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

int main()
{
    eventQueue.call_every(50, periodicCallback);

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    // printf("Start up...\n\r");

    buf.clear();
    serial.baud(19200);

    while (1){
        mainProc();
        eventQueue.dispatch(50);
    }

    // eventQueue.dispatch_forever();

    return 0;
}
