#include "ESP8266.h"

// Constructor
ESP8266::ESP8266(PinName tx, PinName rx, int br) : comm(tx, rx) {
    comm.baud(br);
}

// Destructor
ESP8266::~ESP8266() { }

// Add <CR> + <LF> at the end of the string
void ESP8266::AddEOL(char * s) {
    char k   = strlen(s); // Finds position of NULL character
    s[k]     = '\r';    // switch NULL for <CR>
    s[k + 1] = '\n';    // Add <LF>
    s[k + 2] = '\0';    // Add NULL at the end
}

// Add one ASCII character at the end of the string
void ESP8266::AddChar(char * s, char c) {
    char k   = strlen(s);
    s[k]     = c;
    s[k + 1] = '\0';
}

// Sends command to ESP8266. Receives the command string
void ESP8266::SendCMD(char* s) {
    comm.printf("%s%s", s, CMD_END);
}

// Resets the ESP8266
void ESP8266::Reset(void) {
    SendCMD("AT+RST");
}

// Receive reply until no character is received after a given timeout in miliseconds
bool ESP8266::RcvReply(char* r, int to) {
    Timer t;
    strcpy(r, "");
    t.start();
    
    do {
        if(comm.readable()) {
            AddChar(r, comm.getc());
            t.start();
        }
    } while(t.read_ms() < to);

    AddChar(r, '\0');
    return r[0] != '\0';
}

// Gets the AP list. Parameter: the string to receive the list
bool ESP8266::GetList(char *l) {
    SendCMD("AT+CWLAP");
    return RcvReply(l, 5000); // Needs big timeout because it takes long to start replying
}

// Joins a Wifi AP. Parameters: SSID and Password (strings)
void ESP8266::Join(char *id, char *pwd) {
    char cmd[255];
    sprintf(cmd, "AT+CWJAP=\"%s\",\"%s\"", id, pwd);
    SendCMD(cmd);
}

// Gets ESP IP. Parameter: string to contain IP
bool ESP8266::GetIP(char *ip) {
    SendCMD("AT+CIFSR");
    return RcvReply(ip, 2000);
}

//Defines wifi mode; Parameter: mode; 1= STA, 2= AP, 3=both
void ESP8266::SetMode(WiFiMode mode) {
    char cmd[15];
    strcpy(cmd, "AT+CWMODE=");
    AddChar(cmd, mode + '0'); // Completes command string
    SendCMD(cmd);
}

// Quits the AP
void ESP8266::Quit(void) {
    SendCMD("AT+CWQAP");
}

// Sets single connection
void ESP8266::SetSingle(void) {
    SendCMD("AT+CIPMUX=0");
}

// Sets multiple connection
void ESP8266::SetMultiple(void) {
    SendCMD("AT+CIPMUX=1");
}

// Gets connection status. Parameter: string to contain status
bool ESP8266::GetConnStatus(char * st) {
    SendCMD("AT+CIPSTATUS");
    return RcvReply(st, 2000);
}

// Starts server mode. Parameter: port to be used
void ESP8266::StartServerMode(int port) {
    char rs[25];
    sprintf(rs, "AT+CIPSERVER=1,%d", port);
    SendCMD(rs);
}

// Close server mode.
void ESP8266::CloseServerMode(void) {
    SendCMD("AT+CIPSERVER=0");
}

void ESP8266::setTransparent(void){
    SendCMD("AT+CIPMODE=0");
}

void ESP8266::startTCPConn(char *IP, int port){
    char rs[100];
    sprintf(rs, "AT+CIPSTART=0,\"TCP\",\"%s\",%d", IP, port);
    SendCMD(rs);
}

void ESP8266::sendURL(char *URL, char* IP, char *command){
    char snd[20], http_cmd[300];

    sprintf(http_cmd, "GET %s HTTP/1.1%sHost: %s%sConnection: close%s%s", URL, CMD_END, IP, CMD_END, CMD_END, CMD_END);
    sprintf(snd,"AT+CIPSEND=0,%d",strlen(http_cmd));
    strcpy(command, http_cmd);
    
    SendCMD(snd);
    wait(1);
    SendCMD(http_cmd);
}
    
void ESP8266::deepsleep(size_t ms) {
    char snd[20];
    sprintf(snd, "AT+GSLP=%d", ms);
    SendCMD(snd);
}
    