/*
  @file BMP180.h
  @brief Barometric Pressure and Temperature Sensor BMP180 Breakout I2C Library
*/

#ifndef BMP180_H
#define BMP180_H

#include "mbed.h"

///  default address is 0xEF => (0x77<<1 + 1)
#define BMP180_I2C_ADDRESS  0xEF
#define DEGREES             '\u00B0'
#define DEGREES_CELCIUS     "\u00B0C"

// Oversampling settings
typedef enum {
    ULTRA_LOW_POWER       = 0, ///< 1 pressure sample  :  4.5 ms delay
    STANDARD              = 1, ///< 2 pressure samples :  7.5 ms delay
    HIGH_RESOLUTION       = 2, ///< 4 pressure samples : 13.5 ms delay
    ULTRA_HIGH_RESOLUTION = 3  ///< 8 pressure samples : 25.5 ms delay
} OverSamplingSetting;

/** BMP180 class.
 *  Read Pressure and temperature from the BMP180 Breakout I2C sensor
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "BMP180.h"
 *
 * int main() {
 *     BMP180 bmp180(PIN_SDA, PIN_SCL);
 *     float pressure, temperature;
 *
 *     // bmp180.Initialize(); // no altitude compensation and normal oversampling
 *     bmp180.Initialize(64, BMP180_OSS_ULTRA_LOW_POWER); // 64m altitude compensation and low power oversampling
 *
 *     while(1) {
 *         if (bmp180.ReadData(&pressure, &temperature))
 *             printf("Pressure(hPa): %8.2f \t Temperature(C): %8.2f\n", pressure, temperature);
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class BMP180 {
    public:
        /** Create a BMP180 instance
         * @param i2c object
         */
        BMP180(I2C& i2c);

        /** Initialization: set member values and read BMP180 calibration parameters
         * @param altitude (in meter)
         * @param overSamplingSetting
         * @returns
         *   1 on success,
         *   0 on error
         */
        int initialize(float altitude = 0.F, OverSamplingSetting oss = STANDARD);

        /** Read pressure and temperature from the BMP180.
         * @param pressure (hPa)
         * @param temperature (C)
         * @returns
         *   1 on success,
         *   0 on error
         */
        int readData(float& pTemperature, float& pPressure);

    protected:
        /** Perform temperature measurement
         *
         * @returns
         *   temperature (C)
         */
        int ReadRawTemperature(long* pUt);

        /** Perform pressure measurement
         *
         * @returns
         *   pressure (mbar)
         */
        int ReadRawPressure(long* pUp);

        /** Calculation of the temperature from the digital output
         */
        float TrueTemperature(long ut);

        /** Calculation of the pressure from the digital output
         */
        float TruePressure(long up);

        OverSamplingSetting m_oss;
        float m_altitude;

        I2C m_i2c;
        char m_data[2];

        //short mb; // Not used?
        uint16_t ac4, ac5, ac6;
        int16_t ac1, ac2, ac3, b1, b2, b5, md;
        long x1, x2, x3, b3, b6, mc;
        unsigned long b4, b7;
};

#endif //BMP180_H
