#ifndef ICAL_H
#define ICAL_H
#include "mbed.h"

// This defines the total number of events that can be handled - kind of limiting
// but maybe ok for now. This is a kind of trade-off, based on having to receive
// the whole iCal file into a buffer, and that not leaving a lot left over for
// events themselves. Instead, the receive process should do more "on the fly"
// with a then smaller buffer, leaving room for many more events. Possibly too,
// the events could be quickly rejected if not matching the criteria.
#define EVENT_COUNT 10


// Maximum storage space for each item as free-form text
#ifdef WIN32
#define SUMMARY_CHARS 100
#define LOCATION_CHARS 100
#define CATEGORY_CHARS 20
#else
#define SUMMARY_CHARS 40
#define LOCATION_CHARS 20
#define CATEGORY_CHARS 20
#endif

typedef enum {
    rptfNone,
    rptfDaily,
    rptfWeekly,
    rptfMonthly,
    rptfYearly
} RepeatFreq_t;

#if 0
typedef enum {
    Sunday    = 0x01,
    Monday    = 0x02,
    Tuesday   = 0x04,
    Wednesday = 0x08,
    Thursday  = 0x10,
    Friday    = 0x20,
    Saturday  = 0x40
} RepeatDays_t;
#endif

typedef int32_t tz_sec_t;
typedef int16_t tz_min_t;

/// A single event consists of quite a number of attributes.
typedef struct {
    time_t Start;
    time_t End;
    time_t Until;
    uint16_t Count;
    uint16_t Interval;
    RepeatFreq_t RepeatFreq;
    uint8_t RepeatDays;             // bit mapped (bit 0 = sunday, bit 1=monday, ...)
    uint16_t RepeatMonths;          // bit mapped (bit 0 = jan, 1=feb, ...)
    uint32_t RepeatMonthDay;        // bit mapped (bit 1 = 1st, 2=2nd, ...)
    uint32_t RepeatMonthDayRev;     // reverse -1 = last day = bit 1, -2=bit 2, ...
    char Summary[SUMMARY_CHARS];
    char Location[LOCATION_CHARS];
    char Category[CATEGORY_CHARS];  // "Green", ...
    int Priority;                   // 1 == High, 5 == Normal, 9 == Low
} Event_T;

extern Event_T EventList[EVENT_COUNT];

/// Parse an iCal stream, and extract everything useful.
///
/// This accepts a pointer to a [large] buffer, which is the contents
/// of an iCal stream. It walked through all of the available
/// information to extract the Event list. 
///
/// @param[in] pStart is a pointer to the start of the stream.
/// @param[in] gridStartTime is a time value representing the start of the time-window of interest.
/// @param[in] gridEndTime is a time value representing the end of the time-window of interest.
/// @param[in] tzoMin is the time-zone offset in minutes.
/// @param[in] showEvents when true causes it to print the events as parsed.
/// @returns number of events in range.
///
int ParseICalStream(char * pStart, time_t gridStartTime, time_t gridEndTime, tz_min_t tzoMin, bool showEvents = true);


/// Get the number of events that have been cached.
///
/// @returns the number of events.
///
int GetNumEvents(void);

/// Compute the intersection of two time ranges, and evaluate the recurringing events.
///
/// This compares a pair of time ranges, each by a start and end time. If they overlap
/// it then computes the intersection of those two ranges. Additionally, for a 
/// specified Event, it will evaluate the recurring events that may also fall into
/// the target time range.
///
/// @note This is usually the only API you need, as this will first call
///     the TimeIntersects function, and if that fails, then it will evaluate
///     repeat information.
///
/// @param[in,out] start1 is the starting time of range 1.
/// @param[in,out] end1 is the ending time of range 1.
/// @param[in] start2 is the starting time of range 2.
/// @param[in] end2 is the ending time of range 2.
/// @param[in] Event is a pointer to the event of interest that may have recurring series.
/// @returns true if the ranges overlap, and then start1 and end1 are set to the
///     intersection.
///
bool RepeatIntersects(time_t * start1, time_t * end1, time_t * start2, time_t * end2, Event_T * Event = NULL);


/// Compute the intersection of two time ranges, and returns that intersection.
///
/// This compares a pair of time ranges, each by a start and end time. If they overlap
/// it then computes the intersection of those two ranges.
///
/// @param[in,out] start1 is the starting time of range 1.
/// @param[in,out] end1 is the ending time of range 1.
/// @param[in] start2 is the starting time of range 2.
/// @param[in] end2 is the ending time of range 2.
/// @returns true if the ranges overlap, and then start1 and end1 are set to the
///     intersection.
///
bool TimeIntersects(time_t * start1, time_t * end1, time_t * start2, time_t * end2);


// Private Functions - no real value external to the iCal public interface
// other than for test code.

/// Computes the next interval for iCal events that have recurrence.
/// 
/// @param[in] baseT is the base time value which is a factor only in leap-year.
/// @param[in] repeatFreq is a value representing the frequency of recurrence - 
///     0=none, 1=daily, 2=weekly, 3=monthly, 4=yearly
/// @param[in] interval is the multiplier of that repeat frequency to the next
///     event.
/// @returns a time_t value which is the incremental interval, and would be added
///     to a reference time.
///
time_t NextInterval(time_t baseT, int repeatFreq, int interval);


/// Format a ctime value as a string, without the trailing <cr>
///
/// This uses the normal ctime function, which appends a <cr>, but
/// it then removes that trailing line ending character. Additionally,
/// this keeps a few local static buffers to return the time string in
/// which permits a printf (for example) to call this api a few times
/// and get the proper representation of each.
///
/// @param[in] t is a time value;
/// @returns a pointer to a static buffer containing the converted time.
///
char * FormatCTime(time_t t);

/// Sort the Events that have been extracted from the iCal results.
///
void SortEvents();

/// Show the details for a specific Event, on the specified serial stream.
///
/// Most useful during development, and perhaps no value after that.
///
/// @param[in] Event is the event of interest.
///
void ShowEventInfo(Event_T & Event);

/// Access the 2-letter abbreviation for the day of the week.
/// 
/// @param[in] i is the day of the week, where 0 = sunday
/// @returns a pointer to the 2-letter abbreviation.
///
const char * RepeatDayAbbrev(int i);

/// Parse a Datestamp string into a time value.
///
/// Parses a string which can look like this: 20140505T200000
/// into a time_t value.
///
/// @param[in] string is the string to parse.
/// @param[in] tzoSec is the time-zone offset in seconds.
/// @returns time_t value.
///
time_t ParseDateStamp(char * string, tz_sec_t tzoSec);

/// Parse a Time Zone ID value from the front-end of a Datestamp
///
/// Parses a string which can look like one of these:
/// @li TZID="(GMT -06:00)":20140519T063000
/// @li TZID:(UTC-06:00) Central Time (US & Canada)
/// @li TZID:(GMT -06:00)
///
/// @param[in] string to be parsed.
/// @returns time zone offset in seconds.
///
tz_sec_t ParseTZID(char * string);




#endif // ICAL_H
