/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "TCPSocketServer.h"
#include <string>

TCPSocketServer::TCPSocketServer()
{
    acceptIndex = 0;
}

// Server initialization
int TCPSocketServer::bind(int port)
{
    char cmd[20];

    // set TCP protocol
    wifi->setProtocol(TCP);

    // set ip local port #
    sprintf(cmd, "set i l %d\r", port);
    if (!wifi->sendCommand(cmd, "AOK"))
        return -1;

    // save
    if (!wifi->sendCommand("save\r", "Stor", NULL, 5000))
        return -1;

    // reboot
    wifi->reboot();

    // connect the network
    if (wifi->isDHCP()) {
        if (!wifi->sendCommand("join\r", "DHCP=ON", NULL, 10000))
            return -1;
    } else {
        if (!wifi->sendCommand("join\r", "Associated", NULL, 10000))
            return -1;
    }

    // exit
    wifi->exit();

    wait(0.2);
    wifi->flush();
    return 0;
}

int TCPSocketServer::listen(int backlog)
{
    if (backlog != 1)
        return -1;
    return 0;
}

// sometimes simply "*OPEN*" comes in, and sometimes
// a prior connection closes too, so we get "*CLOS**OPEN*".
// When accepting a connection from a browser, it can be
// *CLOS**OPEN*GET / HTTP/1.1
// The point here is to remove everything to the left
// of, and including, "*OPEN*" in order to accept this
// connection, and the "GET / HTTP/1.1" is available
// for consumption.
//
int TCPSocketServer::accept(TCPSocketConnection& connection)
{
    int nb_available = 0;
    char c;
    const char OPEN[] = "*OPEN*";   // seeking this to accept

    acceptTimerStart();
    while (1) {
        nb_available = wifi->readable();
        if (nb_available == 0 && !_blocking && acceptTimeout())
            return -1;
        for (int i = 0; i < nb_available; i++) {
            c = wifi->getc();
            if (c != OPEN[acceptIndex])
                acceptIndex = 0;
            if (c == OPEN[acceptIndex]) {
                acceptIndex++;
                if (acceptIndex == strlen(OPEN)) {
                    wifi->setConnectionState(true);
                    acceptIndex = 0;    // for next pass
                    return 0;
                }
            }
        }
    }
}

void TCPSocketServer::acceptTimerStart()
{
    acceptTimer.start();
}

bool TCPSocketServer::acceptTimeout()
{
    if ((_timeout == 0) || (acceptTimer.read_ms() >= _timeout))
        return true;
    else
        return false;
}