/* EthernetInterface.h */
/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Changes relative to mbed official version and others as identified
 * in the thread http://mbed.org/forum/team-165-components-community/topic/4844/?page=1#comment-24108
 * EthernetInterface:
 * @li Improve documentation
 * @li add setSecurity() api
 * @li correct connect() api return value (it was inverted)
 * @li derived from 4:0bcec6272784
 */

#ifndef EthernetInterface_H_
#define EthernetInterface_H_

#include "Wifly.h"

/** Interface using Wifly to connect to an IP-based network
*
*/
class EthernetInterface: public Wifly
{
public:

    /**
    * Construct the WiFlyInterface.
    *
    * API Precedence:
    * @verbatim
    * EthernetInterface() -->
    *            +------------------+              +---------------------------+
    *            v                  ^              v                           ^
    * -> init() -+----------------->+-> connect() -+-------------------------->+-> disconnect() --|
    *            |                  ^              +-> getName() --------------+
    *            +-> setName() -----+              +-> getIPAddress() ---------+
    *            |                  |              +-> getMACAddress() --------+
    *            +-> setSecurity() -+              +-> Wifly APIs -------------+
    *                                              +-> get_connection_speed() -+
    * @endverbatim
    *
    * @note The optional ssid, phrase, and sec parameters may be deferred, or
    *       overridden prior to the connect().
    *
    * @param tx mbed pin to use for tx line of Serial interface
    * @param rx mbed pin to use for rx line of Serial interface
    * @param reset reset pin of the wifi module ()
    * @param tcp_status connection status pin of the wifi module (GPIO 6)
    * @param ssid optional parameter which is the ssid of the network
    * @param phrase optional parameter which is the WEP or WPA key/phrase
    * @param sec optional security type (NONE[default], WEP_128 or WPA)
    */
    EthernetInterface(PinName tx, PinName rx, PinName reset, PinName tcp_status, const char * ssid = NULL, const char * phrase = NULL, Security sec = NONE);

    /** Initialize the interface with DHCP.
    * Initialize the interface and configure it to use DHCP (no connection at this point).
    * @return 0 on success, a negative number on failure
    */
    int init(); //With DHCP

    /** Initialize the interface with a static IP address.
    *
    * Initialize the interface and configure it with the following static
    * configuration (no connection at this point).
    *
    * @param ip the IP address to use
    * @param mask the IP address mask
    * @param gateway the gateway to use
    * @return 0 on success, a negative number on failure
    */
    int init(const char* ip, const char* mask, const char* gateway);

    /** Set security parameters for the interface.
    *
    * Prior to connect, this interface permits changing the security parameters. This
    * can be most useful when the Wifly module may be used where there are multiple access
    * points to be connected to. With this, the interface can be brought online far
    * enough to scan for available access points.
    *
    * @param ssid optional parameter which is the ssid of the network
    * @param phrase optional parameter which is the WEP or WPA key/phrase
    * @param sec optional security type (NONE[default], WEP_128 or WPA)
    */
    void setSecurity(const char * ssid = NULL, const char * phrase = NULL, Security sec = NONE);

    /** Connect
    * Bring the interface up, start DHCP if needed.
    * @return 0 on success, a negative number on failure
    */
    int connect();

    /** Disconnect
    * Bring the interface down
    * @return 0 on success, a negative number on failure
    */
    int disconnect();

    /** Get the MAC address of your Ethernet interface.
    *
    * @return a pointer to a string containing the MAC address
    */
    char * getMACAddress();
  
    /** Get IP address
    *
    * @return a pointer to a string containing the IP address
    */
    char * getIPAddress();

    /** setName 
    *
    * Set the network name for this device. Apply this before
    * calling 'connect'.
    *
    * @example
    * EthernetInterface eth;
    * ...
    *     if (0 == eth.init()) {
    *         eth.setName("Sensor 3");
    *         if (0 == eth.connect()) {
    *             ...
    *
    * @param myname is the name to assign for this node. 
    *        Only the first 32 characters will be used if the 
    *        name is longer.
    *        Only '0'-'9', 'A'-'Z', 'a'-'z' are accepted,
    *        any others are converted to '-'.
    * @return 0 on success, a negative number on failure.
    */
    int setName(const char * myname);
    
    /** getName
    *
    * Get the network name for this device.
    *
    * @return pointer to the name (or null)
    */
    const char * getName(void);

    /** get the speed of the connection.
    *
    * @return the connection speed, which can range from 1 to 54 (Mb/s)
    * @return 0 if the information could not be acquired.
    */
    int get_connection_speed(void);

private:
    char ip_string[16];     // "129.168.100.123\0"
    bool ip_set;            // true when we have captured the IP address
    char mac_string[18];    // "11:22:33:44:55:66\0"
    bool mac_set;           // true when we have captured the MAC address
};

#include "TCPSocketConnection.h"
#include "TCPSocketServer.h"
#include "UDPSocket.h"

#endif /* EthernetInterface_H_ */
