/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Wifly RN131-C, wifi module
 *
 * Datasheet:
 * http://dlnmh9ip6v2uc.cloudfront.net/datasheets/Wireless/WiFi/WiFly-RN-UM.pdf
 *
 * Changes relative to mbed official version and others as identified
 * in the thread http://mbed.org/forum/team-165-components-community/topic/4844/?page=1#comment-24108
 * Wifly:
 * @li increase default timeout to 2500 msec
 * @li add additional security options
 * @li add ability to set security after constructor 
 * @li improve commands on the APIs
 * @li add baud() api
 * @li add setConnectionState() api
 * @li add getWiflyVersionString() api
 * @li add getWiflyVersion() api
 * @li add SWUpdateWifly() api
 * @li update for RawSerial instead of Serial
 * @li revise #define DEBUG interface
 * @li reduced some of the C++ string processing for reduced memory
 * @li changed size of auto-send from 1024 to 1420
 * @li add ability to disable remote (telnet) config
 * @li add retry to to the sendCommnand() api
 * @li correct is_connected() api to return true if associated
 * @li revise reset to gather Wifly version number and version string
 * @li reduce compiler warnings
 * @li derived from 4:0bcec6272784
 */

#ifndef WIFLY_H
#define WIFLY_H

#include "mbed.h"
#include "RawSerial.h"
#include "CBuffer.h"

#define DEFAULT_WAIT_RESP_TIMEOUT 5000

enum Security {     // See Wifly user manual Table 2-13.
    NONE        = 0,
    WEP_128     = 1,
    WPA1        = 2,
    WPA_MIXED   = 3,
    WPA         = 3,  // maintained for backward compatibility
    WPA2_PSK    = 4,
    ADHOC       = 6,
    WPE_64      = 8,
    WEP_64      = 8   // probably what the last one should have been
};

enum Protocol {
    UDP = (1 << 0),
    TCP = (1 << 1)
};

/** the Wifly object
*
* This object controls the Wifly module.
*/
class Wifly
{

public:
    /**
    * Constructor
    *
    * @param tx mbed pin to use for tx line of Serial interface
    * @param rx mbed pin to use for rx line of Serial interface
    * @param reset reset pin of the wifi module ()
    * @param tcp_status connection status pin of the wifi module (GPIO 6)
    * @param ssid ssid of the network
    * @param phrase WEP or WPA key
    * @param sec Security type (NONE, WEP_128, WPA1, WPA | WPA_MIXED, WPA2_PSK, WEP_64 )
    */
    Wifly(PinName tx, PinName rx, PinName reset, PinName tcp_status, const char * ssid, const char * phrase, Security sec);

    /**
    * Destructor to clean up
    */
    ~Wifly();
    
    /** Optional means to set/reset the security
    *
    * If join() has not been called, then you can revise the security parameters
    * from those in the constructor.
    *
    * @param ssid ssid of the network
    * @param phrase WEP or WPA key
    * @param sec Security type (NONE, WEP_128, WPA1, WPA | WPA_MIXED, WPA2_PSK, WEP_64 )
    */
    void SetSecurity(const char * ssid, const char * phrase, Security sec);

    /**
    * Configure the wifi module with the parameters contained in the constructor.
    *
    * @return true if successful, false otherwise.
    */
    bool configure();

    /**
    * Connect the wifi module to the ssid contained in the constructor.
    *
    * @return true if connected, false otherwise
    */
    bool join();

    /**
    * Disconnect the wifly module from the access point
    *
    * @return true if successful
    */
    bool disconnect();

    /**
    * Open a tcp connection with the specified host on the specified port
    *
    * @param host host (can be either an ip address or a name. If a name is provided, a dns request will be established)
    * @param port port
    * @return true if successful
    */
    bool connect(const char * host, int port);


    /**
    * Set the protocol (UDP or TCP)
    *
    * @param p protocol
    * @return true if successful
    */
    bool setProtocol(Protocol p);

    /**
    * Reset the wifi module
    */
    void reset();
    
    /**
    * Reboot the wifi module
    *
    * @return true if it could send the command, false otherwise
    */
    bool reboot();

    /**
    * Check if characters are available
    *
    * @return number of available characters
    */
    int readable();

    /**
    * Check if characters are available
    *
    * @return number of available characters
    */
    int writeable();

    /**
    * Check if associated with an access point (was checking if tcp link is active)
    *
    * Follow this example to improve the automatic recovery
    * after a lost association.
    *
    * @code
    * WiflyInterface eth(p28, p27, p23, p24, "ssid", "pass", WPA);
    * 
    * if (0 == eth.init()) {
    *     eth.baud(230400);     // speed up the interface
    *     do {
    *         if (0 == eth.connect()) {
    *             linkup = true;    // led indicator
    *             while (eth.is_connected()) {
    *                 wait_ms(1000);
    *             }
    *             linkup = false;   // led indicator
    *             eth.disconnect();
    *             wait_ms(1000);
    *         }
    *     } while (1);
    * } else {
    *     wait(5);    // ... failed to initialize, rebooting...
    *     mbed_reset();
    * }
    * @endcode
    *
    * @return true if successful
    */
    bool is_connected();

    /**
    * Read a character
    *
    * @return the character read
    */
    char getc();

    /**
    * Flush the buffer
    */
    void flush();

    /**
    * Write a character
    *
    * @param the character which will be written
    * @return the character written
    */
    int putc(char c);


    /**
    * To enter in command mode (we can configure the module)
    *
    * @return true if successful, false otherwise
    */
    bool cmdMode();

    /**
    * To exit the command mode
    *
    * @return true if successful, false otherwise
    */
    bool exit();

    /**
    * Close a tcp connection, and exit command mode.
    *
    * @return true if successful
    */
    bool close();

    /**
    * Send a string to the wifi module by serial port. This function desactivates 
    * the user interrupt handler when a character is received to analyze the response 
    * from the wifi module.
    *
    * Useful to send a command to the module and wait a response.
    *
    * @param str string to be sent
    * @param len string length
    * @param ACK string which must be acknowledge by the wifi module. 
    *           If ACK == NULL, no string has to be acknowledged. (default: NULL)
    * @param res this field will contain the response from the wifi module, 
    *           result of a command sent. This field is available only 
    *           if ACK == NULL AND res != NULL (default: NULL)
    * @param timeout is the time in msec to wait for the acknowledge
    *
    * @return true if ACK has been found in the response from the wifi module. 
    * @return false if there is not a correct response in the timeout.
    */
    int send(const char * str, int len, const char * ACK = NULL, char * res = NULL, int timeout = DEFAULT_WAIT_RESP_TIMEOUT);

    /**
    * Put the device in command mode.
    *
    * @param str string to be sent
    * @param ACK string which must be acknowledge by the wifi module. If ACK == NULL, no string has to be acknoledged. (default: "NO")
    * @param res this field will contain the response from the wifi module, result of a command sent. This field is available only if ACK = "NO" AND res != NULL (default: NULL)
    * @param timeout is the time in msec to wait for the acknowledge
    *
    * @return true if successful, false if it failed.
    */
    bool sendCommand(const char * cmd, const char * ack = NULL, char * res = NULL, int timeout = DEFAULT_WAIT_RESP_TIMEOUT);
    
    /**
    * Return true if the module is using dhcp
    *
    * @return true if the module is using dhcp
    */
    bool isDHCP() {
        return state.dhcp;
    }

    /**
    * gets the host ip address
    *
    * @param host is a pointer to the host string to look up
    * @param ip contains the host IP in a string after the lookup.
    * @param sizeof_ip is the size of the buffer pointed to by ip
    * @return true if successful
    */
    bool gethostbyname(const char * host, char * ip);

    /**
    * Set the baud rate between the ARM and the WiFly.
    *
    * This will set the WiFly module baud rate first and then
    * set the ARM interface to match it. If it cannot get the proper 
    * acknowledge response, it will go on a hunt through the range 
    * of standard baud rates.
    *
    * @note Baud rate must be one of 2400, 4800, 9600, 19200, 38400, 57600,
    *       115200, 230400, 460800, or 921600. (See Wifly manual 2.3.64)
    * @note Baud rate of 230400 has been seen to be marginal w/o flow control.
    * @note Setting a baud rate to a 460800 or above may be unrecoverable
    *       without resetting the Wifly module.
    *
    * @param baudrate is the desired baudrate.
    *
    * @return true if it succeeded, which means that communications can continue, 
    * @return false if it failed to establish a communication link.
    */
    bool baud(int baudrate);


    /**
    * Sets the connection state. 
    *
    * Typically used by external apps that detect an incoming connection.
    *
    * @param state sets the connection state to true or false
    */
    void setConnectionState(bool state);


    /**
    * Get the version information from the Wifly module.
    *
    * @return the version information as a string, or NULL
    */
    char * getWiflyVersionString();
    
    
    /**
    * Get the software version from the Wifly module.
    *
    * This extracts the basic version number (e.g. 2.38, 4.00)
    * as a float.
    *
    * @return the software version number as a float.
    */
    float getWiflyVersion();

    /**
    * Update the software in the Wifly module.
    *
    * This attempts to connect to the microchip site, download
    * a software update, install it as the primary image, and 
    * reboot to activate that image. It is compile-time defined
    * to try for 30 seconds.
    *
    * @param file is the name of the file to fetch, to update to.
    *        The strlen(file) cannot exceed 60, and is typically
    *        like this: "Wifly7-475.mif"
    * @return true if success or false for failure.
    */
    bool SWUpdateWifly(const char * file);
    
    /**
    * determine if the module is in command mode
    *
    * @return true if in command mode, false otherwise
    */
    bool isCmdMode() {
        return state.cmd_mode;
    }

    static Wifly * getInstance() {
        return inst;
    };

protected:
    RawSerial wifi;
    DigitalOut reset_pin;
    DigitalIn tcp_status;
    int baudrate;
    char phrase[65];
    char ssid[33];
    char * wiflyVersionString;
    float swVersion;
    const char * ip;
    const char * netmask;
    const char * gateway;
    CircBuffer<char> buf_wifly;

    static Wifly * inst;

    void attach_rx(bool null);
    void handler_rx(void);
    void flushIn(int timeout_ms = -1);
    uint16_t hextoi(char * p);

    typedef struct STATE {
        bool associated;
        bool tcp;
        bool dhcp;
        Security sec;
        Protocol proto;
        bool cmd_mode;
    } State;

    State state;
    char * getStringSecurity();
    
    /**
    * Allocate space for the parameter (ssid or passphrase)
    * and then fix it (change ' ' to '$').
    *
    * @param dst is a reference to the private member pointer.
    * @param dstLen is the size of the destination buffer
    * @param src is a pointer to a passed in string.
    * @returns true if the src phrase was placed in the dst buffer.
    */
    bool FixPhrase(char * dst, size_t dstLen, const char * src);

    /**
    * Gather the Wifly module version information for later queries
    */
    void GatherLogonInfo();

};

#endif