/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "mbed.h"
#include "Wifly.h"
#include <string>
#include <algorithm>
#include <ctype.h>

// Defined to disable remote configuration via telnet which increases security of this device.
// Available in Wifly module SW 2.27 and higher. If you want to retain remote telnet, undefine
// or comment this.
#define INCREASE_SECURITY


//#define DEBUG "WiFi"      //Debug is disabled by default

// How to use this debug macro
//
// ...
// INFO("Stuff to show %d", var); // new-line is automatically appended
// [I myfile  23] Stuff to show 23\r\n
//
#if (defined(DEBUG) && !defined(TARGET_LPC11U24))
#define INFO(x, ...) std::printf("[INF %s %3d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define WARN(x, ...) std::printf("[WRN %s %3d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define ERR(x, ...)  std::printf("[ERR %s %3d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#else
#define INFO(x, ...)
#define WARN(x, ...)
#define ERR(x, ...)
#endif


#define MAX_TRY_JOIN 3

Wifly * Wifly::inst;

Wifly::Wifly(   PinName tx, PinName rx, PinName _reset, PinName tcp_status, const char * ssid, const char * phrase, Security sec):
    wifi(tx, rx), reset_pin(_reset), tcp_status(tcp_status), baudrate(9600), buf_wifly(256)
{
    INFO("Wifly constructor");
    SetSecurity(ssid, phrase, sec);
    inst = this;
    attach_rx(false);
    state.cmd_mode = false;
    wiflyVersionString = NULL;
    INFO("  const. exit");
}

Wifly::~Wifly()
{
    INFO("~Wifly()");
    if (wiflyVersionString) {
        free(wiflyVersionString);
        wiflyVersionString = NULL;
    }
}


void Wifly::SetSecurity(const char * ssid, const char * phrase, Security sec)
{
    memset(&state, 0, sizeof(state));
    state.sec = sec;
    FixPhrase(this->ssid, sizeof(this->ssid), ssid);
    FixPhrase(this->phrase, sizeof(this->phrase), phrase);
}


bool Wifly::configure()
{
    char cmd[80];   // room for command with maximum ssid or passphrase

    INFO("configure");
    for (int i= 0; i < MAX_TRY_JOIN; i++) {

        // no auto join
        if (!sendCommand("set w j 0\r", "AOK"))
            continue;

        // no echo
        if (!sendCommand("set u m 1\r", "AOK"))
            continue;

        // set comm time to flush (ms)
        if (!sendCommand("set c t 30\r", "AOK"))
            continue;

        // set comm size to auto-send
        if (!sendCommand("set c s 1420\r", "AOK"))
            continue;

        // set comm idle time to auto-close (sec)
        //if (!sendCommand("set c i 5\r", "AOK"))
        //    continue;

        // red led on when tcp connection active
        if (!sendCommand("set s i 0x40\r", "AOK"))
            continue;

        // no hello string sent to the tcp client
        if (!sendCommand("set c r 0\r", "AOK"))
            continue;

        // tcp protocol
        if (!sendCommand("set i p 2\r", "AOK"))
            continue;

        // tcp retry (retry enabled, Nagle alg, retain link)
        if (!sendCommand("set i f 0x7\r", "AOK"))
            continue;

#ifdef INCREASE_SECURITY
        // tcp-mode 0x10 = disable remote configuration
        // only in SW 2.27 and higher (see 2.3.39)
        if ((swVersion >= 2.27) && (!sendCommand("set i t 0x10\r", "AOK")))
            continue;
#endif

        // set dns server
        if (!sendCommand("set d n rn.microchip.com\r", "AOK"))
            continue;

        //dhcp
        sprintf(cmd, "set i d %d\r", (state.dhcp) ? 1 : 0);
        if (!sendCommand(cmd, "AOK"))
            continue;

        // ssid
        sprintf(cmd, "set w s %s\r", ssid);
        if (!sendCommand(cmd, "AOK"))
            continue;

        //auth
        sprintf(cmd, "set w a %d\r", state.sec);
        if (!sendCommand(cmd, "AOK"))
            continue;

        // if no dhcp, set ip, netmask and gateway
        if (!state.dhcp) {
            INFO("not dhcp");
            sprintf(cmd, "set i a %s\r", ip);
            if (!sendCommand(cmd, "AOK"))
                continue;

            sprintf(cmd, "set i n %s\r", netmask);
            if (!sendCommand(cmd, "AOK"))
                continue;

            sprintf(cmd, "set i g %s\r", gateway);
            if (!sendCommand(cmd, "AOK"))
                continue;
        }

        //key step
        cmd[0] = '\0';
        switch (state.sec) {
            case WPE_64:        // google searching suggests this is a typo and should be WEP_64
            case WEP_128:
                sprintf(cmd, "set w k %s\r", phrase);
                break;
            case WPA1:
            case WPA_MIXED: // alias WPA
            case WPA2_PSK:
                sprintf(cmd, "set w p %s\r", phrase);
                break;
            case ADHOC:
            case NONE:
            default:
                break;
        }
        if (cmd[0] && !sendCommand(cmd, "AOK"))
            continue;

        if (!sendCommand("save\r", "Stor", NULL, 5000))
            continue;

        exit();
        return true;
    }
    return false;
}


bool Wifly::join()
{
    INFO("join");
    //join the network (10s timeout)
    if (state.dhcp && swVersion < 4.75) {
        if (!sendCommand("join\r", "DHCP=ON", NULL, 10000))     // possibly older SW did this
            return false;
    } else {
        if (!sendCommand("join\r", "Associated!", NULL, 10000))  // This for most uses
            return false;
    }
    INFO("  join exit");
    exit();
    INFO("  join end.");
    state.associated = true;
    return true;
}


bool Wifly::setProtocol(Protocol p)
{
    // use udp auto pairing
    char cmd[20];
    sprintf(cmd, "set i p %d\r", p);
    if (!sendCommand(cmd, "AOK"))
        return false;

    switch(p) {
        case TCP:
            // set ip flags: tcp retry enabled
            if (!sendCommand("set i f 0x07\r", "AOK"))
                return false;
            break;
        case UDP:
            // set ip flags: udp auto pairing enabled
            if (!sendCommand("set i h 0.0.0.0\r", "AOK"))
                return false;
            if (!sendCommand("set i f 0x40\r", "AOK"))
                return false;
            break;
    }
    state.proto = p;
    return true;
}


char * Wifly::getStringSecurity()
{
    switch(state.sec) {
        case NONE:              // 0
            return "NONE";
        case WEP_128:           // 1
            return "WEP_128";
        case WPA1:              // 2
            return "WPA1";
        case WPA:               // 3
            return "WPA";
        case WPA2_PSK:          // 4
            return "WPA2_PSK";
        case ADHOC:             // 6
            return "ADHOC";
        case WPE_64:            // 8
            return "WPE_64";
        default:                // ?
            return "UNKNOWN";
    }
}


bool Wifly::connect(const char * host, int port)
{
    char rcv[20];
    char cmd[20];

    // try to open
    sprintf(cmd, "open %s %d\r", host, port);
    if (sendCommand(cmd, "OPEN", NULL, 10000)) {
        setConnectionState(true);
        exit();
        return true;
    }

    // if failed, retry and parse the response
    if (sendCommand(cmd, NULL, rcv, 5000)) {
        if (strstr(rcv, "OPEN") == NULL) {
            if (strstr(rcv, "Connected") != NULL) {
                if (!sendCommand("close\r", "CLOS"))
                    return false;
                if (!sendCommand(cmd, "OPEN", NULL, 10000))
                    return false;
            } else {
                return false;
            }
        }
    } else {
        return false;
    }

    setConnectionState(true);
    exit();
    return true;
}


bool Wifly::gethostbyname(const char * host, char * ip)
{
    string h = host;
    char cmd[30], rcv[100];
    int l = 0;
    char * point;
    int nb_digits = 0;

    // no dns needed
    int pos = h.find(".");
    if (pos != string::npos) {
        string sub = h.substr(0, h.find("."));
        nb_digits = atoi(sub.c_str());
    }
    //printf("substrL %s\r\n", sub.c_str());
    if (count(h.begin(), h.end(), '.') == 3 && nb_digits > 0) {
        strcpy(ip, host);
    }
    // dns needed
    else {
        nb_digits = 0;
        sprintf(cmd, "lookup %s\r", host);
        if (!sendCommand(cmd, NULL, rcv))
            return false;

        // look for the ip address
        char * begin = strstr(rcv, "=") + 1;
        for (int i = 0; i < 3; i++) {
            point = strstr(begin + l, ".");
            INFO("str: %s", begin + l);
            l += point - (begin + l) + 1;
        }
        INFO("str: %s", begin + l);
        while(*(begin + l + nb_digits) >= '0' && *(begin + l + nb_digits) <= '9') {
            INFO("digit: %c", *(begin + l + nb_digits));
            nb_digits++;
        }
        memcpy(ip, begin, l + nb_digits);
        ip[l+nb_digits] = 0;
        INFO("ip from dns: %s", ip);
    }
    return true;
}


void Wifly::flush()
{
#if 0 and defined(DEBUG)
    char chatter[500];
    int count = 0;
    char c;

    while (buf_wifly.available()) {
        buf_wifly.dequeue(&c);
        chatter[count++] = c;
    }
    chatter[count] = '\0';
    if (count)
        DBG("Wifly::flush {%s}", chatter);
#endif
    buf_wifly.flush();
}


bool Wifly::sendCommand(const char * cmd, const char * ack, char * res, int timeout)
{
    int tries = 1;
    Timer t;
    
    INFO("sendCommand");
    t.start();
    while (tries <= 3) {
        if (cmdMode()) {      // some influences to the wifi module sometimes kick it out
            if (send(cmd, strlen(cmd), ack, res, timeout) >= 0) {
                INFO("  sendCommand - success");
                t.stop();
                INFO("  sendCommand - success in %f", t.read());
                return true;
            }
        }
        state.cmd_mode = false;     // must not really be in cmd mode
        ERR("sendCommand: failure %d when sending: %s", tries, cmd);
        tries++;
    }
    INFO("  sendCommand - failure in %f", t.read());
    send("exit\r", 5, "EXIT");
    return false;
}


bool Wifly::cmdMode()
{
    char buf[200];
    // if already in cmd mode, return
    if (state.cmd_mode) {
        INFO("  is cmdMode");
        #if 0
        return true;
        #else  // for deeper debugging
        // Quick verify to ensure we really are in cmd mode
        //flushIn(0);
        //INFO("  send \\r to test for cmdMode");
        if (send("\r", 1, ">") == 1) {
            //INFO("  is cmdMode");
            return true;
        } else {
            ERR(" failed to detect command mode");
            state.cmd_mode = false;
        }
        #endif
    } else {
        wait_ms(460);   // manual 1.2.1 (250 msec before and after)
        #if 1
        if (send("$$$", 3, NULL, buf, 1500)) {
            INFO("Resp: [%s]", buf);
            if ( ! strstr(buf, "CMD")) {
                WARN("cannot enter cmd mode");
                send("exit\r", 5, "EXIT", NULL, 100);
                return false;
            }
        }
        #else
        if (send("$$$", 3, "CMD") == -1) {  // the module controls the 'after' delay
            ERR("cannot enter in cmd mode");

            return false;
        }
        #endif
        state.cmd_mode = true;
    }
    return true;
}


bool Wifly::disconnect()
{
    // if already disconnected, return
    if (!state.associated)
        return true;

    if (!sendCommand("leave\r", "DeAuth"))
        return false;
    exit();

    state.associated = false;
    return true;
}


uint16_t Wifly::hextoi(char *p)
{
    uint16_t res = 0;
    
    while ((*p >= '0' && *p <= '9') || (*p >= 'a' && *p <= 'f') || (*p >= 'A' && *p <= 'F')) {
        if (*p >= '0' && *p <= '9')
            res = (res * 16) + (*p - '0');
        else if (*p >= 'a' && *p <= 'f')
            res = (res * 16) + (*p - 'a' + 10);
        else if (*p >= 'A' && *p <= 'F')
            res = (res * 16) + (*p - 'A' + 10);
        p++;
    }
    return res;
}


bool Wifly::is_connected()
{
    char buf[30];
    uint16_t connectionStatus = 0;
    bool cnxStatus = false;
    
    if (sendCommand("show connection\r", NULL, buf)) {
        connectionStatus = hextoi(buf);
        exit();
    }
    //return (tcp_status.read() ==  1) ? true : false;  // hw pin
    if (connectionStatus & 0x0010) { // associated
        cnxStatus = true;
    } else {
        state.associated = false;
        cnxStatus = false;
    }
    return cnxStatus;
}


void Wifly::reset()
{
    reset_pin = 0;
    INFO("RESET ACTIVATED");
    wifi.baud(9600);
    wait_ms(400);
    reset_pin = 1;
    GatherLogonInfo();
    INFO("swver %3.2f, {%s}", getWiflyVersion(), getWiflyVersionString());
}


bool Wifly::reboot()
{
    if (sendCommand("reboot\r", "Reboot")) {
        state.cmd_mode = false;
        wait_ms(500);
        wifi.baud(9600);
        baud(baudrate);
        exit();
        return true;
    } else {
        return false;
    }
}


bool Wifly::close()
{
    if (!state.tcp) {
        return true;    // already closed
    }
    if (!sendCommand("close\r", "*CLOS*")) {
        return false;   // failed to close
    }
#if 1
    // It appears that the close exits cmd mode
    // so we won't bother trying to close which
    // could cause it to open command mode to
    // send the close (which add more 0.5s delays).
    state.cmd_mode = false;
#else
    flushIn();
    exit();
#endif
    setConnectionState(false);
    return true;        // succeeded to close
}


int Wifly::putc(char c)
{
    while (!wifi.writeable())
        ;
    return wifi.putc(c);
}


bool Wifly::exit()
{
    INFO("exit()");
    if (!sendCommand("exit\r", "EXIT")) {
        ERR("  failed to exit.");
        return false;
    }
    state.cmd_mode = false;
    return true;
}


int Wifly::readable()
{
    return buf_wifly.available();
}


int Wifly::writeable()
{
    return wifi.writeable();
}


char Wifly::getc()
{
    char c = 0xCC;  // avoid compiler warning of uninitialized var.

    while (!buf_wifly.available())
        ;
    buf_wifly.dequeue(&c);
    return c;
}


void Wifly::handler_rx(void)
{
    //read characters
    while (wifi.readable())
        buf_wifly.queue(wifi.getc());
}


void Wifly::attach_rx(bool callback)
{
    if (!callback)
        wifi.attach(NULL);
    else
        wifi.attach(this, &Wifly::handler_rx);
}

int Wifly::send(const char * str, int len, const char * ACK, char * res, int timeout)
{
    char read;
    int ackIndex = 0;
    Timer tmr;
    int result = 0;

    INFO("will send: %s",str);
    attach_rx(false);
    flushIn();

    for (int i = 0; i < len; i++)
        result = (putc(str[i]) == str[i]) ? result + 1 : result;
    INFO("  data sent.");
    tmr.start();
    if (ACK) {
        while (1) {
            if (tmr.read_ms() > timeout) {
                //flushIn();
                WARN("timeout awaiting '%s' in (%f)", ACK, tmr.read());
                attach_rx(true);
                return -1;
            } else if (wifi.readable()) {
                read = wifi.getc();
                if (tolower(read) != tolower(ACK[ackIndex]))
                    ackIndex = 0;
                if (tolower(read) == tolower(ACK[ackIndex])) {
                    ackIndex++;
                    if (ackIndex == strlen(ACK)) {
                        //flushIn();
                        break;
                    }
                }
            }
        }
        INFO("check: ACK '%s' received in (%f)", ACK, tmr.read());
        if (strcmp(str,"exit") != 0)
            flushIn();
        attach_rx(true);
        return result;
    }

    // the user wants the result from the command (ACK == NULL, res != NULL)
    if ( res != NULL) {
        int i = 0;
        int lastStamp = tmr.read_ms();
        //Timer timeout;
        //timeout.start();
        //tmr.reset();
        while (1) {
            if (tmr.read_ms() > timeout) {      // crash and burn timeout...
                if (i == 0) {
                    res = NULL;
                    break;
                }
                res[i] = '\0';
                WARN(" hit user %d msec timeout: %s", timeout, res);
                break;
            } else {
                if ((tmr.read_ms() - lastStamp) > 300) { // timeout since last char suggests done...
                    res[i] = '\0';
                    //WARN("user str: %s", res);
                    break;
                }
                while (wifi.readable()) {
                    lastStamp = tmr.read_ms();
                    read = wifi.getc();
                    res[i++] = read;
                }
            }
        }
        INFO("user str: %s", res);
    }
    flushIn();
    attach_rx(true);
    INFO("result: %d in (%f)", result, tmr.read())
    return result;
}

void Wifly::flushIn(int timeout_ms)
{
    Timer tmr;
    int lastStamp;
#if 1 and defined(DEBUG)
    char chatter[500];
    int count = 0;
    int c;
#endif

    if (timeout_ms <= 0) {
        timeout_ms = 30; // 2 * 10000 / baudrate;  // compute minimal timeout
    }
    tmr.start();
    lastStamp = tmr.read_ms();
    while (wifi.readable() || ((tmr.read_ms() - lastStamp) < timeout_ms)) {
        if (wifi.readable()) {
#if 1 and defined(DEBUG)
            c = wifi.getc();
            //printf("%02X ", c);
            if (count < sizeof(chatter)-1)  // guard overflow
                chatter[count++] = c;
#else
            wifi.getc();
#endif
            lastStamp = tmr.read_ms();
        }
    }
#if 1 and defined(DEBUG)
    chatter[count] = '\0';
    if (count && (count > 2 || chatter[0] != '\r' || chatter[1] != '\n')) {
        INFO("Wifly::flushIn(%d) {%s} in (%f)", count, chatter, tmr.read());
    } else {
        INFO("Wifly::flushIn() empty in [%d] (%f)", lastStamp, tmr.read());
    }
#endif
}


// The ARM uart and the Wifly uart have to be in sync or we get
// no meaningful response, so then have to try the possibilities.
//
// First try is at the currently configured ARM uart baud, if
// that fails then it shifts the ARM uart baud through the probable
// speeds, trying to establish contact with the Wifly module.
// Once contact is demonstrated (by response to the 'ver' command),
// then it sets the Wifly module and then the ARM uart.
bool Wifly::baud(int _targetBaud)
{
    // in testing, 460800 and 921600 may change the Wifly module where you can't
    // change it back w/o a reset. So, we won't even permit those speeds.
    const int baudrates[] = {2400, 4800, 9600, 19200, 38400, 57600, 115200, 230400}; //, 460800, 921600};
#define BRCOUNT (sizeof(baudrates)/sizeof(baudrates[0]))
    char cmd[26];       // sized for "set u i 460800\r" [15+1], plus margin [4]
    int tryIndex = 0;
    bool res = false;
    int userIndex;

    sprintf(cmd, "set uart instant %d\r", _targetBaud);
    // set u i # should cause it to exit command mode (manual 2.3.64),
    // but testing indicates that it does not.
    for (userIndex=0; userIndex < BRCOUNT; userIndex++) {
        if (_targetBaud == baudrates[userIndex]) {
            while (tryIndex <= BRCOUNT) {
                //INFO("baud() try: %d: %d", tryIndex, _targetBaud);
                sendCommand(cmd); // shift Wifly to desired speed [it may not respond (see 2.3.64)]
                flushIn(10);
                //state.cmd_mode = false;  // see note above why this is disabled
                wifi.baud(_targetBaud);     // shift the ARM uart to match
                if (sendCommand("ver\r", "wifly", NULL, 125)) {  // use this to verify communications
                    baudrate = _targetBaud;
                    res = true;
                    break;              // success
                }
                // keep trying baudrates between ARM and WiFly
                if (tryIndex < BRCOUNT) {
                    //INFO(" baud() set to %d", baudrates[tryIndex]);
                    wifi.baud(baudrates[tryIndex]);
                }
                tryIndex++;
            }
            break;  // if they selected a legitimate baud, try no others
        }
    }
    //INFO(" baud() result: %d", res);
    return res;
}


bool Wifly::FixPhrase(char * dst, size_t dstLen, const char * src)
{
    if (strlen(src) < dstLen) {
        strcpy(dst, src);
        // change all ' ' to '$' in ssid or passphrase
        for (int i = 0; i < strlen(dst); i++) {
            if ((dst)[i] == ' ')
                (dst)[i] = '$';
        }
        INFO("phrase: {%s} fr {%s}", dst, src);
        return true;
    } else {
        ERR("Source {%s} is too long for destination buffer of %d bytes", src, dstLen);
        return false;
    }
}


void Wifly::GatherLogonInfo()
{
    Timer timer;
    char logonText[200];
    int i = 0;
    char *p;

    timer.start();
    if (wiflyVersionString) {
        free(wiflyVersionString);
        wiflyVersionString = NULL;
    }
    logonText[i] = '\0';
    while (timer.read_ms() < 500) {
        while (wifi.readable() && (i <sizeof(logonText)-1)) {
            logonText[i++] = wifi.getc();
        }
    }
    logonText[i] = '\0';
    p = strchr(logonText, '\r');
    if (p)
        *p = '\0';
    wiflyVersionString = (char *)malloc(strlen(logonText)+1);
    if (wiflyVersionString) {
        strcpy(wiflyVersionString, logonText);
    }
    p = strstr(logonText, "Ver ");          // "Ver 4.00" for ver <= 4.00
    if (!p) p = strstr(logonText, "Ver: "); // "Ver: 4.40" new in ver 4.40
    if (p) {
        while (*p && (*p < '0' || *p > '9'))
            p++;
        swVersion = atof(p);
    }
    WARN("swVersion: %3.2f,\r\nverString: {%s}", swVersion, wiflyVersionString);
}


float Wifly::getWiflyVersion()
{
    INFO("swVersion: %3.2f", swVersion);
    return swVersion;
}


char * Wifly::getWiflyVersionString()
{
    INFO("version string: %s", wiflyVersionString);
    return wiflyVersionString;
}

bool Wifly::SWUpdateWifly(const char * file)
{
    bool success = false;
    char buf[80];
    
    INFO("\r\n\r\n\r\n");
    INFO("SWUpdateWifly %s", file);
    if (strlen(file) < (80 - 13)) {
        sprintf(buf, "ftp update %s\r", file);
        if (is_connected()) {
            // once connected, send command to update firmware
            if (sendCommand("set ftp address 0\r", "AOK")) {
                if (sendCommand("set dns name rn.microchip.com\r", "AOK")) {
                    if (sendCommand("save\r", "Stor", NULL, 5000)) {
                        if (sendCommand(buf, "UPDATE OK", NULL, 50000)) {
                            if (sendCommand("factory RESET\r")) {
                                if (reboot()) {
                                    success = true;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return success;
}


void Wifly::setConnectionState(bool value)
{
    state.tcp = value;
}
