#ifndef VT100_H
#define VT100_H included
#include "mbed.h"

/** vt100 class
 * Utility for handling text/letter on a terminal
 * which can handle VT100 escape command sequence.
 *
 * Derived from https://os.mbed.com/users/Rhyme/code/vt100/file/b7229a9eae1c/vt100.h/
 *              which had no licensing requirements.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "vt100.h"
 * 
 * vt100 tty;
 * 
 * int main() {
 *   int count = 0 ;
 *   tty.cls() ;
 *   tty.setFG(VT_COLOR::VT_BLUE) ;
 *   tty.frame(5, 5, 15, 9) ;
 *   while(1) {
 *       tty.locate(7, 7) ;
 *       tty.setFG(count % 8) ;
 *       printf("%d\r\n", count++) ;
 *       wait(1.0) ;
 *   }
 * }
 * @endcode
 * 
 */
 
class vt100 : public Stream {
public:
    /** constructor 
     * @param pc is a pointer to the Serial Interface to use.
     * @param name is the interface name
     */
    vt100(Serial * pc, const char * name) ;
    
    /** destructor */
    virtual ~vt100();
    
    /** clear screen */
    void cls(void) ;
    
    /** move cursor to (x, y) 
     * @param x start column of the next letter
     * @param y start row of the next letter
     * @note no value checking is performed.
     */
    void locate(int x, int y) ;
    
    /** print a letter c at (x,y)
     * @param c the letter to be written 
     * @param x column of the letter
     * @param y row of the letter
     */
    void putChar(int x, int y, char c) ;
    
    /** print a string str from (x,y) 
     * @param *str c-style string to be written
     * @param x column of the first letter
     * @param y row of the first letter
     */
    void putStr(int x, int y, const char *str) ;
    
    /** print a line of char 
     * @param x1 starting column
     * @param y1 starting row
     * @param x2 ending column
     * @param y2 ending row
     * @param c the letter to form the line
     */
    void line(int x1, int y1, int x2, int y2, char c='*') ;
    
    /** print a text frame 
     * @param x1 left column
     * @param y1 top row
     * @param x2 right column
     * @param y2 bottom row
     */
    void frame(int x1, int y1, int x2, int y2) ;
    
    /** print a text circle
     * @param x0 center column
     * @param y1 center row
     * @param r radius
     * @param c the optional letter to form the circle, default '*'
     */
    void circle(int x0, int y0, int r, char c='*') ;
    
    /// VT Color definitions
    typedef enum {
        VT_BLACK   , ///< VT Color Black
        VT_RED     , ///< VT Color Red
        VT_GREEN   , ///< VT Color Green
        VT_YELLOW  , ///< VT Color Yellow
        VT_BLUE    , ///< VT Color Blue
        VT_PURPLE  , ///< VT Color Purple
        VT_CYAN    , ///< VT Color Cyan
        VT_WHITE   , ///< VT Color White
    } VT_COLOR;

    /** set foreground color
     * @param newFG new foreground color
     * @returns previous foreground color
     * @note VT_COLOR::VT_BLACK
     * @note VT_COLOR::VT_RED
     * @note VT_COLOR::VT_GREEN
     * @note VT_COLOR::VT_YELLOW
     * @note VT_COLOR::VT_BLUE
     * @note VT_COLOR::VT_PURPLE
     * @note VT_COLOR::VT_CIAN
     * @note VT_COLOR::VT_WHITE
     */
    VT_COLOR setFG(VT_COLOR newFG) ;
    
    /** get current foreground color 
     * @returns current foreground color
     */
    VT_COLOR getFG(void) ;
    
    /** set background color
     * @param newBG new background color
     * @returns previous background color
     */
    VT_COLOR setBG(VT_COLOR newBG) ;
    
    /** get current background color 
     * @returns current background color
     */
    VT_COLOR getBG(void) ;

protected:
    /// a method to put a character to the display.
    ///
    /// @param value is the character value to send to the display
    /// @returns the character that was sent.
    ///
    virtual int _putc(int value);


    /// a method to get a character from the stdin
    ///

    /// @returns the fetched character.
    ///
    virtual int _getc();


    virtual int readable();

private:
    Serial * pc;
    VT_COLOR foreground ;
    VT_COLOR background ;
 
} ;
 
#endif /* VT100_H */
