/// @file TimeUtilities.h contains a real time clock interface for the mbed
///
/// @mainpage Time Management of the Real Time Clock module
/// 
/// APIs for showing the time from the RTC, or from a passed in value.
/// Also supports setting the clock, timezone offsets and calibration
/// of the RTC subsystem.
///
/// @note Copyright &copr; 2011 by Smartware Computing, all rights reserved.
///     Individuals may use this application for evaluation or non-commercial
///     purposes. Within this restriction, changes may be made to this application
///     as long as this copyright notice is retained. The user shall make
///     clear that their work is a derived work, and not the original.
///     Users of this application and sources accept this application "as is" and
///     shall hold harmless Smartware Computing, for any undesired results while
///     using this application - whether real or imagined.
///
/// @author David Smart, Smartware Computing
///
/// @note
/// History
/// v1.04 20111101
/// \li Minor change to the APIs to reduce some of the complexity
/// v1.03 20111030
/// \li Adding an API to adjust time and calibration based on how far it is
///     off "now", based on when it was set.
/// v1.02 
/// \li 20110716 Minor documentation update
/// \li 20110616 Minor documentation updates
/// v1.01 29 May 2011
/// \li Slightly improved documentation.
/// unlabeled 29 May 2011
/// \li Initial release as a library
///
#ifndef TIMEUTILITIES_H
#define TIMEUTILITIES_H

#ifndef WIN32
#include "mbed.h"
#else
typedef int int32_t;
#endif

#include <time.h>

/// RealTimeClock is an interface to the mbed RTC module
///
/// It provides interfaces to get and set the time in text representation
/// as well as a means to get and set the timezone offset so that it 
/// maintains time in UTC format.
/// 
/// Additionally, it provides an interface to adjust the calibration
/// registers of the RTC, to compensate for 32 kHz clock error, whether
/// based on the oscillator itself or on some bias to ambient temperature.
///
/// @note It is recommended that the system has a battery connected to
/// the mbed battery backup pin.
///
/// @note The timezone offset is stored in RTC register GPREG0 which
/// is battery backed, permitting recovery on the next startup.
///
/// Example:
/// @code
/// RealTimeClock rtc;
///
/// void main() {
///     bool success;
///     char TimeTest[] = "05/29/2011 14:15:18 (-6:00)";
///     char buf[50];
///     success = rtc.SetTime(TimeTest);
///     rtc.GetTimeString(buf);
///     printf("success: %i, time: %s\n", success, buf);
/// }
/// prints the following:
/// success: 1, time: Sun May 29 14:15:18 2011 (tzo: -6:00)
/// @endcode
///
class RealTimeClock
{
public:
    /// Constructor for the RealTimeClock interface, usually implemented
    /// outside of main( )
    ///
    RealTimeClock();

    /// Destructor for the RealTimeClock, usually not executed
    ///
    ~RealTimeClock();

    /// GetTimeValue gets the current time in time_t format
    ///
    /// Gets the current time from the rtc and returns the time in time_t format.
    /// This is functionally identical to "return time(NULL);", 
    /// simply wrapped in this class for convenience.
    ///
    /// @returns current time in time_t format
    ///
    time_t GetTimeValue();
    
    /// GetTimeString gets the formatted time applying the internal time-zone offset of hours and minutes
    ///
    /// It places the formatted string into the buffer in the format of
    /// Sun May 29 07:19:24 2011 (tzo: -6:00)
    ///
    /// The internal timezone offset can be set with SetTimeOffset
    ///
    /// @param buffer is a pointer to a buffer large enough (>= 40 chars) for the formatted time string
    /// @param tValue is the optional time value to convert to text, if zero, then the current 
    ///         time is used.
    /// @returns nothing
    ///
    void GetTimeString(char * buffer, time_t tValue = 0);

    /// GetTimeString gets the formatted time applying the time-zone offset of hours and minutes
    ///
    /// It places the formatted string into the buffer in the format of
    /// Sun May 29 07:19:24 2011 (tzo: -6:00)
    ///
    /// @param buffer is a pointer to a buffer large enough (>= 40 chars) for the formatted time string
    /// @param hOffset is the hours offset
    /// @param mOffset is the minutes offset
    /// @returns nothing
    ///
    void GetTimeString(char * buffer, int hOffset, int mOffset);

    /// GetTimeString gets the formatted time value applying the time-zone offset of hours and minutes
    ///
    /// It places the formatted string into the buffer in the format of
    /// Sun May 29 07:19:24 2011 (tzo: -6:00)
    ///
    /// @param buffer is a pointer to a buffer large enough (>= 40 chars) for the formatted time string
    /// @param tValue is the time value to convert to text, if zero, then the current 
    ///         time is used.
    /// @param hOffset is the hours offset
    /// @param mOffset is the minutes offset
    /// @returns nothing
    ///
    void GetTimeString(char * buffer, time_t tValue, int hOffset, int mOffset);

    /// GetTimeLastSet will extract the time_t when last set from RTC ram
    ///
    /// When the clock is correctly set, the time value will have been stored
    /// which permits later adjustments to the calibration based on 
    /// elapsed time and knowing when it was set. This function retrieves it.
    ///
    /// @returns time_t value when the time was last set
    ///
    time_t GetTimeLastSet();

    /// SetTimeOffset will set the hour and minute timezone offset
    ///
    /// @param offsetHour is the timezone offset in hours
    /// @param offsetMinute is the timezone offset in minutes
    /// @returns nothing
    ///
    void SetTimeOffset(int offsetHour, int offsetMinute);

    /// GetTimeCalibration will return the calibration register value
    ///
    /// This is the raw register value as a signed 32-bit value (even though
    /// it is actually a 17-bit unsigned value with an additional 'direction' flag).
    ///
    /// @returns calibration settings ranging from -131071 to +131071
    ///
    int32_t GetTimeCalibration();

    /// SetTimeCalibration will set the calibration register value
    ///
    /// This accepts a signed value to be used to set the calibration
    /// registers. Setting the calibration value to zero disables the
    /// calibration function. 
    /// It is important to know the register function in order to use 
    /// this command, and this API is normally not used by external
    /// application code. @See AdjustBySeconds for a user-friendly
    /// API.
    ///
    /// @param calibration value to use ranging from -131071 to +131071
    /// @returns nothing
    ///
    void SetTimeCalibration(int32_t calibration);


    /// SetTime will set the Real Time Clock from a well formatted string
    ///
    /// This will read a string, and if it is well formed, it will then
    /// set the RTC based on this string. The string should be of the form:
    /// 'MM/DD/YYYY HH:MM:SS' and may have the optional timezone offset 
    /// of the form '(+/-HH:MM)'
    ///
    /// example: 05/29/2011 13:15:07 (-6:00)
    ///
    /// @param timestring is the string to parse
    /// @returns true if the time was set
    ///
    bool SetTime(char * timestring);
    
    
    /// AdjustBySeconds adjusts both the time and the calibration by seconds
    ///
    /// This will take a signed value, which is the current adjustment in seconds
    /// to put the clock on the correct time. So, if the clock is behind by
    /// 3 seconds, the value should be +3 to advance the clock accordingly.
    /// It will then adjust the time, and it will attempt to adjust the
    /// calibration factor to make the time more accurate.
    ///
    /// The adjustment can only be made if it has retained when the clock was
    /// last set, in order to know by how much to adjust it. It is also most
    /// accurate if several days have elapsed since the time was set.
    ///
    /// @note The current version only works if the calibration value
    ///       is zero when this adjustment is made.
    /// 
    /// @param adjustSeconds is the signed value by which to adjust the time to
    ///        correct it to the current actual time.
    /// @returns true if the adjustment was made
    /// @returns false if the adjustment could not be made
    ///
    bool AdjustBySeconds(int32_t adjustSeconds);
    
    /// GetVersion will return a pointer to a constant text string
    ///
    /// This returns a pointer to a constant string which represents
    /// the version of this component.
    ///
    /// @returns pointer to a character string holding the version.
    ///
    const char * GetVersion();
    
private:

    /// SetTimeOffsetStore will store the timezone offset values in RTC ram
    ///
    /// This takes the current timezone offset values and stores them in 
    /// the RTC battery backed ram.
    ///
    /// @returns nothing
    ///
    void SetTimeOffsetStore();

    /// GetTimeOffsetStore will extract the timezone offset values from RTC ram
    ///
    /// This extracts the timezone offset values that were stored in RTC
    /// battery backed ram and uses them for time information.
    ///
    /// @returns nothing
    ///
    void GetTimeOffsetStore();
    
    /// SetTimeLastSet will store the time_t when last set in RTC ram
    ///
    /// When the clock is correctly set, the time value will be stored
    /// which permits later adjustments to the calibration based on 
    /// elapsed time and knowing when it was set.
    ///
    /// @param t is the time_t value that should be saved
    /// @returns nothing
    ///
    void SetTimeLastSet(time_t);
    
};

#endif
