/// Demonstration of dynamic page creation using the Smartware Web Server
///
/// Here is a sample for file upload.
#include "SW_HTTPServer.h"
#include "DynamicFileIn.h"

#define DEBUG "File"
#include "Utility.h"

// This defines the size of the largest file you are willing to accept.
// Hopefully, space is also available on the file system for it.
#define ACCEPT_MAX_BYTES (25000)

// This defines the local cache, which should be at least 2 x the size of ETHER_CHUNK.
// It can be any size above that, so long as RAM is available.
#define PCACHE_SIZE (3000)

// ETHER_CHUNK is the maximum size of a single packet handoff.
#define ETHER_CHUNK (1500)
#if ((2 * ETHER_CHUNK) > PCACHE_SIZE)
#error "Configuration error, PCACHE_SIZE must be at least 2 * ETHER_CHUNK size.
#endif

static int bytesInCache;
static char * pCache;
static char * boundary = "";        // for pointing at the boundary marker part of multipart/form-data in pCache
static char * contenttype = NULL;   // for "text/plain", or "image/bmp"
static char * fqfn = NULL;          // for the filepath/filename
static FILE * fp = NULL;
static int bytesWritten;

static int FileSize(const char * dir, const char * filename);
static char _tolower(char c);
static int _strnicmp(const char * left, const char * right, int len);
static char * _stristr(char * pHaystack, const char * pNeedle);
static char * _strnbin(char * pBinstack, const char *pNeedle, int len);
static void FreeResources(void);

typedef enum {
    IDLE,
    ACCEPTED,
    REJECTED_TOO_LARGE
} AcceptStatus;

//
//-----------------------------7de3ab1652086c
//Content-Disposition: form-data; name="InFile"; filename="Len0700.txt"
//Content-Type: text/plain
//
//__1 - _50 12345678901234567890123456789012345678
//_51 - 100 12345678901234567890123456789012345678
//101 - 150 12345678901234567890123456789012345678
//151 - 200 12345678901234567890123456789012345678
//201 - 250 12345678901234567890123456789012345678
//251 - 300 12345678901234567890123456789012345678
//301 - 350 12345678901234567890123456789012345678
//351 - 400 12345678901234567890123456789012345678
//401 - 450 12345678901234567890123456789012345678
//451 - 500 12345678901234567890123456789012345678
//501 - 550 12345678901234567890123456789012345678
//551 - 600 12345678901234567890123456789012345678
//601 - 650 12345678901234567890123456789012345678
//651 - 700 12345678901234567890123456789012345678
//
//-----------------------------7de3ab1652086c--
//^^ Boundary Definition                     ^^ Boundary Definition
//   is appended to "--"                        is suffixed "--" indicating the end.
//---
//Content-Disposition: form-data; name="InFile"; filename="somefile.txt"\r\n
//Content-Type: image/bmp\r\n
//\r\n
//BM6.... (where the ... is purely binary data data)
//\r\n------webKitFormBoundary9A3C872FAC9BE23--\r\n
//---
//
HTTPServer::CallBackResults ProcessCache(HTTPServer *svr, HTTPServer::CallBackType type)
{
    typedef enum {
        Seeking_Boundary_Start,
        Seeking_Filename,
        Seeking_ContentType,
        Seeking_Data
    } EF_State_E;
    //const char * statelist[] = {
    //    "Start",
    //    "Filename",
    //    "ContentType",
    //    "Data"
    //};
    static EF_State_E EF_State = Seeking_Boundary_Start;

    char * pDataStart = pCache;                 // start of data in this transaction
    char * pDataEnd = pCache + bytesInCache;    // end of data in this transaction
    char * pDBLcrlf;
    char *p1, *p2, *p3;                         // general purpose pointer for processing the results
    HTTPServer::CallBackResults ret = HTTPServer::ACCEPT_CONTINUE;

    //INFO("ProcessCache %s", statelist[EF_State]);
    INFO("### Cache Begin\r\n%s\r\n### Cache End", pCache);
    if (EF_State == Seeking_Boundary_Start) {
        INFO("Start");
        pDBLcrlf = _strnbin((char *)pDataStart, "\r\n\r\n", pDataEnd - pDataStart);    // find end of preamble
        if (pDBLcrlf) {
            p1 = _strnbin((char *)pDataStart, boundary, pDBLcrlf - pDataStart);   // is there a boundary in the cache?
            if (p1) {
                EF_State = Seeking_Filename;
            }
        }
    }
    
    if (EF_State == Seeking_Filename) {
        INFO("File");
        pDBLcrlf = _strnbin((char *)pDataStart, "\r\n\r\n", pDataEnd - pDataStart);    // find end of preamble
        if (pDBLcrlf) {
            p2 = _strnbin((char *)pDataStart, "Content-Disposition: ", pDBLcrlf - pDataStart);   // and the filename container?
            if (p2) {
                p2 = _strnbin((char *)p2, "filename=\"", pDBLcrlf - pDataStart);
                if (p2) {   // found filename
                    p2 += strlen("filename=\"");    // start of filename
                    p3 = strchr(p2, '"');           // end of filename
                    if (p3) {
                        *p3 = '\0';
                        fqfn = (char *)malloc(p3-p2 + strlen(svr->GetWebRoot())+2);
                        if (fqfn) {
                            strcpy(fqfn, svr->GetWebRoot());     // We could put it elsewhere
                            strcat(fqfn, "/");
                            strcat(fqfn, p2);
                            *p3 = '"';              // Put this back so we can print the cache
                            EF_State = Seeking_ContentType;
                            pDataStart = p3 + 1;
                        } else {
                            ERR("Failed to allocate space for filename");
                            ret = HTTPServer::ACCEPT_ERROR;
                            EF_State = Seeking_Boundary_Start;
                            bytesInCache = 0;
                            return ret;
                        }
                    }
                }
            }
        }
    }
    
    if (EF_State == Seeking_ContentType) {
        INFO("ContentType");
        pDBLcrlf = _strnbin((char *)pDataStart, "\r\n\r\n", pDataEnd - pDataStart);    // find end of preamble
        if (pDBLcrlf) {
            p2 = _strnbin((char *)pDataStart, "Content-Type: ", pDBLcrlf - pDataStart);
            if (p2) {
                p2 += strlen("Content-Type: ");
                p3 = strchr(p2, '\r');
                if (p3) {
                    *p3 = '\0';
                    contenttype = (char *)malloc(p3-p2 + 2);
                    if (contenttype) {
                        strcpy(contenttype, p2);
                        *p3 = '\r';             // put it back so we can print the cache
                    } else {
                        ERR("Failed to allocate space for content type");
                        ret = HTTPServer::ACCEPT_ERROR;
                        EF_State = Seeking_Boundary_Start;
                        bytesInCache = 0;
                        return ret;
                    }
                    pDataStart = pDBLcrlf + 4;  // Advance to the actual data
                    EF_State = Seeking_Data;
                }
            }
        }
    }
    
    if (EF_State == Seeking_Data) {
        INFO("Data");
        if (!fp && fqfn) {
            INFO("Open file [%s]", fqfn);
            fp = fopen(fqfn, "w");
            if (!fp) {
                ERR("failed to open [%s]", fqfn);
                FreeResources();
                ret = HTTPServer::ACCEPT_ERROR;
                return ret;
            }
        }
        if (fp) {
            Timer t;
            // The file is open, so we're writing data to it. 
            // We do this until we find the ending boundary, but we may not have the [whole]
            // ending boundary in the cache yet.
            p1 = _strnbin((char *)pDataStart, boundary, pDataEnd - pDataStart);   // is there a boundary in the cache?
            if (p1) {       // if we find the ending boundary, we can truncate and write the data ahead of the boundary
                p1 -= 4;    // Actual ending boundary is "\r\n--" + "boundary" + "--", so back up 4.
                *p1 = '\0';
                pDataEnd = p1;
                if (pDataEnd > pDataStart) {
                    bytesWritten += pDataEnd - pDataStart;
                    t.start();
                    fwrite(pDataStart, 1, pDataEnd - pDataStart, fp);
                    INFO("Writing %5d bytes, %5d total written, (reserve %d) in %d us", 
                        pDataEnd - pDataStart, bytesWritten, 0, t.read_us());
                    //INFO("Write: %s***", pDataStart);
                    bytesInCache = 0;
                    *pCache = '\0';         // this for debugging in case it is text and you want to print it.
                } else {
                    INFO("Not enough data to write (%d <= %d)", pDataEnd, pDataStart);
                }
            } else {  // didn't find the boundary, but it might be partially there...
                // write the data, but not the last (strlen(boundary) + 4) 
                // in case this transaction has some, but not all of the boundary
                int reserve = strlen(boundary) + 4;
                pDataEnd -= reserve;
                if (pDataEnd > pDataStart) {
                    bytesWritten += pDataEnd - pDataStart;
                    t.start();
                    fwrite(pDataStart, 1, pDataEnd - pDataStart, fp);
                    INFO("Writing %5d bytes, %5d total written, (reserve %d) in %d us", 
                        pDataEnd - pDataStart, bytesWritten, reserve, t.read_us());
                    //INFO("Write: %s###", pDataStart);
                    memcpy(pCache, pDataEnd, reserve+1);    // copies the trailing '\0'
                    bytesInCache = reserve;
                    //*(pCache + bytesInCache) = '\0';
                    INFO("remaining: %s", pCache);
                } else {
                    INFO("Not enough data to write (%d <= %d)", pDataEnd, pDataStart);
                }
            }
            ret = HTTPServer::ACCEPT_CONTINUE;
        }
    }

    if (type == HTTPServer::DATA_TRANSFER_END) {
        FreeResources();
        ret = HTTPServer::ACCEPT_COMPLETE;
    }
    return ret;
}



/// DynamicFileIn
///
/// This page lets you submit a file.
///
/// You can see in main how this page was registered.
///
HTTPServer::CallBackResults DynamicFileIn(
    HTTPServer *svr,
    HTTPServer::CallBackType type,
    const char * path,
    const HTTPServer::namevalue *params,
    int count)
{
    char buf[100];
    int clSize;     // content-length size to be transferred
    static AcceptStatus accept = IDLE;
    HTTPServer::CallBackResults ret = HTTPServer::ACCEPT_ERROR;
    DIR *d;
    struct dirent *p;

    switch (type) {
        case HTTPServer::CONTENT_LENGTH_REQUEST:
            // Here we can find out how much data is headed our way,
            // and choose to accept or reject it.
            clSize = atoi(svr->GetHeaderValue("Content-Length"));
            //printf("Content Len RQST(%d)\r\n", clSize);
            //printf("Content-Type: [%s]\r\n", svr->GetHeaderValue("Content-Type"));
            // also, for the multipart data, we have to learn and track
            // the boundary sequence, so we can parse the big block
            // successfully.
            boundary = _stristr((char *)svr->GetHeaderValue("Content-Type"), "boundary=");
            if (boundary)
                boundary += strlen("boundary=");
            //INFO("C-Len: %d, boundary: [%s]", clSize, boundary);
            //printf("InFile = [%s]\r\n", svr->GetHeaderValue("InFile"));
            // If we're happy with the size, and we have the boundary marker we need
            if (clSize < ACCEPT_MAX_BYTES && boundary) {
                pCache = (char *)malloc(PCACHE_SIZE);
                if (pCache) {
                    //INFO("pCache - allocated");
                    bytesInCache = 0;
                    accept = ACCEPTED;
                    ret = HTTPServer::ACCEPT_COMPLETE;
                } else {
                    accept = REJECTED_TOO_LARGE;    // can't get the RAM to host the stream
                    ret = HTTPServer::ACCEPT_ERROR;
                }
            } else {
                accept = REJECTED_TOO_LARGE;
            }
            break;

        case HTTPServer::DATA_TRANSFER_END:
            //INFO("\r\n\\\\\\\\\\\\ Transfer Begin [%d, %d]\r\n%s\r\n////// Transfer End", count, bytesInCache, path);
            INFO("TransferEnd [%d, %d, %d, %d]", count, bytesInCache, PCACHE_SIZE - ETHER_CHUNK, PCACHE_SIZE);
            // if there is anything there, copy it in.
            if (count) {
                memcpy(pCache + bytesInCache, path, count);
                bytesInCache += count;
                pCache[bytesInCache] = '\0';    // one past the data, so ok (and good for debugging text streams)
            }
            ret = ProcessCache(svr, type);
            break;

        case HTTPServer::DATA_TRANSFER:
            //INFO("\r\n\\\\\\\\\\\\ Transfer Begin [%d, %d]\r\n%s\r\n////// Transfer End", count, bytesInCache, path);
            // We chose to accept the transfer, and it may come in chunks.
            if ((bytesInCache + count) < (PCACHE_SIZE - ETHER_CHUNK)) { // room in the cache for more, juat accept it locally
                INFO("Transfer A  [%d, %d, %d, %d]", count, bytesInCache, PCACHE_SIZE - ETHER_CHUNK, PCACHE_SIZE);
                memcpy(pCache + bytesInCache, path, count);
                bytesInCache += count;
                pCache[bytesInCache] = '\0'; // one past the data, so ok (and good for debugging text streams)
                ret = HTTPServer::ACCEPT_CONTINUE;
            } else if ((bytesInCache + count) < PCACHE_SIZE) {        // room in the cache, now process it.
                INFO("Transfer B  [%d, %d, %d, %d]", count, bytesInCache, PCACHE_SIZE - ETHER_CHUNK, PCACHE_SIZE);
                memcpy(pCache + bytesInCache, path, count);
                bytesInCache += count;
                pCache[bytesInCache] = '\0';    // one past the data, so ok (and good for debugging text streams)
                ret = ProcessCache(svr, type);  // now hand it off
            } else {
                INFO("Transfer C  [%d, %d, %d, %d]", count, bytesInCache, PCACHE_SIZE - ETHER_CHUNK, PCACHE_SIZE);
                ret = ProcessCache(svr, type);  // panic, ah!, ok, first hand it off then cache it
                memcpy(pCache + bytesInCache, path, count);
                bytesInCache += count;
                pCache[bytesInCache] = '\0';    // one past the data, so ok (and good for debugging text streams)
                ERR("FATAL - didn't flush pCache, %d in cache.", bytesInCache);
                break;
            }
            break;

        case HTTPServer::SEND_PAGE:
            svr->header(200, "OK", svr->GetSupportedType(".htm"));
            svr->send("<html><head><title>Dynamic File Submit</title></head>\r\n");
            svr->send("<body>\r\n");
            svr->send("<h1>Smart WiFly Web Server</h1>\r\n");
            sprintf(buf, "This example permits you to upload a file of not more than %d bytes. ", ACCEPT_MAX_BYTES);
            svr->send(buf);
            svr->send("If the file is one of the supported types, you can then download it too. ");
            svr->send("Select the file using the form gadget and then submit it.");
            switch(accept) {
                case IDLE:
                default:
                    break;
                case ACCEPTED:
                    svr->send("<br/><b>Previous submission accepted.</b><br/>\r\n");
                    break;
                case REJECTED_TOO_LARGE:
                    svr->send("<br/><b>Previous submission rejected.</b><br/>\r\n");
                    break;
            }
            // note that to accept a file, the form enctype is multipart/form-data.
            svr->send("<blockquote>\r\n");
            sprintf(buf, "<form method='post' enctype='multipart/form-data' action='%s'>\r\n", path);
            svr->send(buf);
            svr->send("<table border='0'>\r\n");
            svr->send("<tr><td>File</td><td><input type='file' name='InFile' size='60'></td></tr>\r\n");
            svr->send("<tr><td>&nbsp;</td><td><input type='submit' value='Submit'><input type='reset' value='Clear'></td></tr>\r\n");
            svr->send("</table>\r\n");
            svr->send("</form>\r\n");
            svr->send("</blockquote>\r\n");
#if 1
            sprintf(buf, "<h2>Directory of [%s]:</h2>\r\n", svr->GetWebRoot());
            svr->send(buf);
            d = opendir(svr->GetWebRoot());
            if ( d != NULL ) {
                svr->send("<ul>\r\n");
                while ( (p = readdir(d)) != NULL ) {
                    if (svr->GetSupportedType(p->d_name) != NULL)
                        sprintf(buf, "<li><a href='%s'>%s</a> (%d bytes)</li>\r\n", p->d_name, p->d_name, FileSize(svr->GetWebRoot(), p->d_name)); // only supported types linked
                    else
                        sprintf(buf, "<li>%s (%d bytes)</li>\r\n", p->d_name, FileSize(svr->GetWebRoot(), p->d_name));     // We could skip/hide these if that makes more sense
                    svr->send(buf);
                }
                svr->send("</ul>\r\n");
                closedir(d);
            } else {
                svr->send("Unable to open directory!");
            }
#endif
            svr->send("<a href='/'>Back to main</a></body></html>\r\n");
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;

        default:
            printf("unknown command %d\r\n", type);
            ret = HTTPServer::ACCEPT_ERROR;
            break;
    }
    return ret;
}

static void FreeResources(void)
{
    if (fp) {
        fclose(fp);
        fp = NULL;
    }
    if (fqfn) {
        free(fqfn);
        fqfn = NULL;
    }
    if (contenttype) {
        free(contenttype);
        contenttype = NULL;
    }
    if (pCache) {
        free(pCache);
        pCache = NULL;
        bytesInCache = 0;
    }
}

static int FileSize(const char * dir, const char * filename)
{
    int size = 0;
    char * fqfn = (char *)malloc(strlen(dir) + strlen(filename) + 2);
    if (fqfn) {
        strcpy(fqfn, dir);
        strcat(fqfn, "\\");
        strcat(fqfn, filename);
        FILE * f = fopen(fqfn,"r");
        if (f) {
            fseek(f, 0, SEEK_END); // seek to end of file
            size = ftell(f);
            fclose(f);
        }
        free(fqfn);
    }
    return size;
}

// very simple ASCII only function to convert
// UPPERCASE to lowercase.
//
static char _tolower(char c)
{
    if (c >= 'A' && c <= 'Z')
        c = c - 'A' + 'a';
    return c;
}

// case insensitive compare
static int _strnicmp(const char * left, const char * right, int len)
{
    if (!left || !right)
        return 0;           // actually, can't compare so this is a small lie
    while (len) {
        if (_tolower(*left) < _tolower(*right))
            return -1;
        else if (_tolower(*left) > _tolower(*right))
            return 1;
        left++;
        right++;
        len--;
    }
    return 0;   // match
}

// Search a haystack string for a needle string - case insensitive.
//
static char * _stristr(char * pHaystack, const char * pNeedle)
{
    if (!pHaystack || !*pHaystack || !pNeedle || !*pNeedle)
        return NULL;    // bad params
    while (*pHaystack) {
        if (_strnicmp(pHaystack, pNeedle, strlen(pNeedle)) == 0)
            return pHaystack;
        pHaystack++;
    }
    return NULL;
}

// Search the [possibly] binary haystack for the Needle, which is a string.
// This is used to find the 'boundary' marker in a stream from the network
// interface. The contents between the boundary markers could be text, or
// it could be a binary stream. Since the haystack could be binary, we need
// to control the search by a length parameter.
//
static char * _strnbin(char * pBinstack, const char *pNeedle, int len)
{
    char * pLast;
    if (!pBinstack || !pNeedle || !*pNeedle || len == 0)
        return NULL;    // bad params
    len -= strlen(pNeedle); // only search [most] of the haystack
    pLast = pBinstack + len;
    while (pBinstack <= pLast) {
        if (_strnicmp(pBinstack, pNeedle, strlen(pNeedle)) == 0)
            return pBinstack;
        pBinstack++;
    }
    return NULL;
}

