/// Demonstration of dynamic page creation using the Smartware Web Server
///
/// There are a few samples here of the dynamically generated web pages.
/// Read the header above each one for more details.
///
#include "mbed.h"

#include "SW_HTTPServer.h"
#include "DynamicPages.h"

//#define DEBUG "dynP"    /* define DEBUG before "Utility.h" to activate the INFO, WARN, ERR macros. */
#include "Utility.h"


BusOut leds(LED4,LED3,LED2,LED1);       // dynamic page "/dyn2" lets you control these


/// SimplyDynamicPage1
///
/// This web page is generated dynamically as a kind of "bare minimum".
/// It doesn't do much.
///
/// You can see in main how this page was registered.
///
HTTPServer::CallBackResults SuperSimpleDynamicPage(HTTPServer *svr, HTTPServer::CallBackType type, const char * path, const HTTPServer::namevalue *params, int paramcount)
{
    HTTPServer::CallBackResults ret = HTTPServer::ACCEPT_ERROR;
    char contentlen[30];
    char buf[250];

    switch (type) {
        case HTTPServer::SEND_PAGE:
            // This sample drops it all into a local buffer, computes the length,
            // and passes that along as well. This can help the other end with efficiency.
            strcpy(buf, "<html><head><title>Home Page</title></head>\r\n");
            strcat(buf, "<body>\r\n");
            strcat(buf, "<h1>Smart WiFly Web Server</h1>\r\n");
            strcat(buf, "This page was generated dynamically.<br/>\r\n");
            strcat(buf, "<a href='/dyn'>To another dynamically generated page.</a><br/>\r\n");
            strcat(buf, "<a href='/led'>A page to control the LEDs.</a><br/>\r\n");
            strcat(buf, "</body></html>\r\n");
            sprintf(contentlen, "Content-Length: %d\r\n", strlen(buf));
            // Now the actual header response
            svr->header(200, "OK", "Content-Type: text/html\r\n", contentlen);
            // and data are sent
            svr->send(buf);
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
        case HTTPServer::CONTENT_LENGTH_REQUEST:
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
        case HTTPServer::DATA_TRANSFER:
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
        default:
            ret = HTTPServer::ACCEPT_ERROR;
            break;
    }
    return ret;
}


/// SimplyDynamicPage
///
/// This web page is generated dynamically and fills in part of the information from
/// data extracted from the URL as a query string. It also shows some additional
/// information - memory stats, server stats and so forth.
///
/// You can see in main how this page was registered.
///
HTTPServer::CallBackResults SimpleDynamicPage(HTTPServer *svr, HTTPServer::CallBackType type, const char * path, const HTTPServer::namevalue *params, int paramcount)
{
    static unsigned int pageCounter = 0;
    char buf[150];
    HTTPServer::CallBackResults ret = HTTPServer::ACCEPT_ERROR;
    HTTPServer::SW_PerformanceData perfData;
    //Wifly * w = svr->GetWifly()->getInstance();

    switch (type) {
        case HTTPServer::SEND_PAGE:
            INFO("dt: %d, %s", type, path);
            if (strcmp("true", svr->GetParameter("Reset")) == 0)
                svr->ResetPerformanceData();
            // Send the header
            svr->header(200, "OK", "Content-Type: text/html\r\n");

            // Send top of the page
            svr->send(
                "<html><head><title>Dynamic Page</title><meta http-equiv='refresh' content='15'></head>\r\n"
                "<body>\r\n"
                "<h1>Smart WiFly Web Server</h1>\r\n"
                "This page was generated dynamically. Create your own name=value pairs on the URL "
                "which uses the GET method.<br/>\r\n");

            sprintf(buf, "This page has been accessed %d times.<br/>\r\n", ++pageCounter);
            svr->send(buf);
            
            // Show Passed-in parameters
            svr->send("<h2>Query Parameters:</h2><blockquote>\r\n");
            sprintf(buf, "%d parameters passed to {%s}:<br/>\r\n", paramcount, path);
            svr->send(buf);
            // show each of the parameters passed on the URL
            for (int i=0; i<paramcount; i++) {
                sprintf(buf, "%d: %s = %s<br/>\r\n", i, params[i].name, params[i].value);
                svr->send(buf);
            }
            svr->send("</blockquote>\r\n");

            // Show Memory Stats
            svr->send(
                "<h2>System Statistics:</h2><blockquote>\r\n"
                "<table border='1'><tr><td>Parameter</td><td>Description</td></tr>\r\n");
            sprintf(buf,"<tr><td align='right'>%d</td><td>Free memory</td></tr>\r\n", Free());
            svr->send(buf);
            sprintf(buf,"<tr><td align='right'>%d</td><td>Max Header size</td></tr>\r\n", svr->GetMaxHeaderSize());
            svr->send(buf);
            //sprintf(buf,"<tr><td align='right'>%3.2f</td><td>Wifly SW version</td></tr>\r\n", w->getWiflyVersion());
            //svr->send(buf);
            //sprintf(buf,"<tr><td align='right'><font size='-1'>%s</font></td><td>Wifly Version Information</td></tr>\r\n", w->getWiflyVersionString());
            //svr->send(buf);
            svr->send("</table>\r\n");
            svr->send("</blockquote>\r\n");

            // Show Web Server Performance metrics
            svr->GetPerformanceData(&perfData);
            svr->send("<h2>Web Server Performance Metrics</h2><blockquote>\r\n");
            svr->send("<table border='1'><tr><td>avg time (uS)</td><td>samples</td><td>max time (uS)</td><td>description</td></tr>\r\n");

            sprintf(buf, "<tr><td align='right'>%d</td><td align='right'>%d</td><td align='right'>%d</td><td>%s</td></tr>\r\n",
                    (unsigned long)(perfData.ConnectionAccepted.TotalTime_us / perfData.ConnectionAccepted.Samples),
                    perfData.ConnectionAccepted.Samples,
                    (unsigned long)(perfData.ConnectionAccepted.MaxTime_us),
                    "Connection Accepted");
            svr->send(buf);

            sprintf(buf, "<tr><td align='right'>%d</td><td align='right'>%d</td><td align='right'>%d</td><td>%s</td></tr>\r\n",
                    (unsigned long)(perfData.HeaderParsed.TotalTime_us / perfData.HeaderParsed.Samples),
                    perfData.HeaderParsed.Samples,
                    (unsigned long)(perfData.HeaderParsed.MaxTime_us),
                    "Header Parsed");
            svr->send(buf);
            
            sprintf(buf, "<tr><td align='right'>%d</td><td align='right'>%d</td><td align='right'>%d</td><td>%s</td></tr>\r\n",
                    (unsigned long)(perfData.ResponseSent.TotalTime_us / perfData.ResponseSent.Samples),
                    perfData.ResponseSent.Samples,
                    (unsigned long)(perfData.ResponseSent.MaxTime_us),
                    "Response Sent");
            svr->send(buf);
            
            sprintf(buf, "<tr><td align='right'>%d</td><td align='right'>%d</td><td align='right'>%d</td><td>%s</td></tr>\r\n",
                    (unsigned long)(perfData.ConnectionClosed.TotalTime_us / perfData.ConnectionClosed.Samples),
                    perfData.ConnectionClosed.Samples,
                    (unsigned long)(perfData.ConnectionClosed.MaxTime_us),
                    "Connection Closed");
            svr->send(buf);

            svr->send(
                "<tr><td colspan='3'>&nbsp;</td><td><a href='?Reset=false'>Reload</a>"
                " <a href='?Reset=true'>Reset Metrics</a></td></tr>\r\n"
                "</table>\r\n"
                "</blockquote>\r\n");

            // Send bottom of the page
            svr->send("<br/><a href='/'>back to main</a></body></html>\r\n");
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
        case HTTPServer::CONTENT_LENGTH_REQUEST:
            INFO("dt: %d, %s", type, path);
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
        case HTTPServer::DATA_TRANSFER:
            INFO("dt: %d, %s", type, path);
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
        default:
            INFO("dt: %d, %s", type, path);
            ret = HTTPServer::ACCEPT_ERROR;
            break;
    }
    return ret;
}

/// SimplyDynamicForm
///
/// This web page is generated dynamically and fills in part of the information from
/// data extracted from the URL. It also generates a web form, and allows you to
/// update the data in the form using the POST method. This one also updates the
/// leds on the mbed module based on the form submission, so demonstrates a very
/// simple interaction with the hardware.
///
/// You can see in main how this page was registered.
///
HTTPServer::CallBackResults SimpleDynamicForm(HTTPServer *svr, HTTPServer::CallBackType type, const char * path, const HTTPServer::namevalue *params, int paramcount)
{
    char buf[150];
    const char * p1, *p2;
    HTTPServer::CallBackResults ret = HTTPServer::ACCEPT_ERROR;

    switch (type) {
        case HTTPServer::CONTENT_LENGTH_REQUEST:
            INFO("dt: %d, %s", type, path);
            // this callback asks this handler if it will accept a data transfer.
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
            
        case HTTPServer::SEND_PAGE:
            INFO("dt: %d, %s", type, path);
            // set the LEDs based on a passed in parameter.
            p1 = svr->GetParameter("leds");
            p2 = svr->GetPostParameter("leds");
            if (p1)
                leds = atoi(p1);
            else if (p2)
                leds = atoi(p2);

            // send the header
            svr->header(200, "OK", svr->GetSupportedType(".htm"));  //svr->header(200, "OK", "Content-Type: text/html\r\n");
            // send some data
            svr->send("<html><head><title>Dynamic Page</title></head>\r\n");
            svr->send("<body>\r\n");
            svr->send("<h1>Smart WiFly Web Server</h1>\r\n");
            svr->send("This form was generated dynamically. You can add name=value pairs on the URL "
                      "which will show up in the form, but the form is submitted using POST method.<br/>\r\n");
            svr->send("Hint: <a href='/led?leds=7&sky=Blue&car=Jaguar'>leds=7&sky=Blue&car=Jaguar</a> turns on"
                      " 3 of the 4 blue leds on the mbed and passes a couple other parameters.<br/>\r\n");
            sprintf(buf, "%d parameters passed to {%s}:<br/>\r\n", paramcount, path);
            svr->send(buf);
            
            // Create a user form for which they can post changes
            sprintf(buf, "POST Form: <form method='post' action='%s'>\r\n", path);
            //sprintf(buf, "<form method='post' enctype='multipart/form-data' action='%s'>\r\n", path); // not supported
            svr->send(buf);
            // show the parameters in a nice format
            svr->send("<table>\r\n");
            
            for (int i=0; i<svr->GetParameterCount(); i++) {
                HTTPServer::namevalue * p = svr->GetParameter(i);
                sprintf(buf, "<tr><td>%d</td><td>%s</td><td><input type='text' name='%s' value='%s'></td></tr>\r\n", 
                    i, p->name, p->name, p->value);
                svr->send(buf);
            }
            for (int i=0; i<svr->GetPostParameterCount(); i++) {
                HTTPServer::namevalue * p = svr->GetPostParameter(i);
                sprintf(buf, "<tr><td>%d</td><td>%s</td><td><input type='text' name='%s' value='%s'></td></tr>\r\n", 
                    i, p->name, p->name, p->value);
                svr->send(buf);
            }

            //svr->send("<tr><td>&nbsp;</td><td>File</td><td><input type='file' name='InFile' size='40'></td></tr>\r\n");
            svr->send("<tr><td>&nbsp;</td><td colspan='2'><input type='submit' value='submit'><input type='reset' value='clear'></td></tr>\r\n");
            svr->send("</table>\r\n");
            svr->send("</form>\r\n");

            // Create a user form for which they can post changes
            sprintf(buf, "GET Form: <form method='get' action='%s'>\r\n", path);
            //sprintf(buf, "<form method='post' enctype='multipart/form-data' action='%s'>\r\n", path); // not supported
            svr->send(buf);
            // show the parameters in a nice format
            svr->send("<table>\r\n");
            for (int i=0; i<paramcount; i++) {
                sprintf(buf, "<tr><td>%d</td><td>%s</td><td><input type='text' name='%s' value='%s'></td></tr>\r\n", 
                    i, 
                    params[i].name, params[i].name, params[i].value);
                svr->send(buf);
            }
            //svr->send("<tr><td>&nbsp;</td><td>File</td><td><input type='file' name='InFile' size='40'></td></tr>\r\n");
            svr->send("<tr><td>&nbsp;</td><td colspan='2'><input type='submit' value='submit'><input type='reset' value='clear'></td></tr>\r\n");
            svr->send("</table>\r\n");
            svr->send("</form>\r\n");

            svr->send("<br/><a href='/'>Back to main</a></body></html>\r\n");
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
            
        case HTTPServer::DATA_TRANSFER:
            INFO("dt: %d, %s", type, path);
            ret = HTTPServer::ACCEPT_COMPLETE;
            break;
            
        default:
            INFO("dt: %d, %s", type, path);
            ret = HTTPServer::ACCEPT_ERROR;
            break;
    }
    return ret;
}

