/** @file main.cpp contains the main program that a user would write.
 * see the documentation above "main"
 */
#include "mbed.h"               // testing ver 92, last stable v89
#include "RawSerial.h"

// My components
#include "IniManager.h"
#include "Utility.h"            // a couple of simple helper functions
#include "WiflyInterface.h"     // ver 53, derived from mbed official ver 4
#include "SW_HTTPServer.h"      // ver 31, derived from nweb
#include "DynamicPages.h"       // my dynamically generated pages
#include "SecurePage.h"         // my secure pages
#include "ServerConfig.h"       // various configuration options
#include "DynamicFileIn.h"      // Upload a file to the server
//#include "MSCFileSystem.h"      // ver 1, this and SDFileSystem
//#include "SDFileSystem.h"       // ver 1, this and MSCFileSystem

#define MBED_APP_BOARD 1        /* http://mbed.org/components/mbed-Application-Board/ */
#define SMART_BOARD    2        /* http://mbed.org/users/WiredHome/notebook/SmartBoard-baseboard/ */

#define HW_ADAPTER MBED_APP_BOARD   /* Which board are we compiling against? SMART_BOARD or MBED_APP_BOARD */

#if HW_ADAPTER == MBED_APP_BOARD
#include "C12832.h"

C12832 lcd(p5, p7, p6, p8, p11);
#endif

#define HTTP_SERVER_PORT 80

RawSerial pc(USBTX, USBRX);

LocalFileSystem local("local");         // some place to hold settings and maybe the static web pages
//MSCFileSystem msc("msc");               // Mass Storage on USB
//SDFileSystem sd(p5, p6, p7, p8, "sd");  // for the static web pages
#define WEBROOT "/local"

PwmOut signOfLife(LED1);

Timer onceinawhile;

/// ShowSignOfLife
///
/// Pulse an LED to indicate a sign of life of the program.
/// also has some moderate entertainment value.
///
void ShowSignOfLife() {
#define PI 3.14159265359
    static Timer activityTimer;
    static unsigned int activityStart;
    static bool init;
    //static float currentBrightness = 0.0;
    static int degrees = 0;
    float v;
    //static bool rampUp = true;

    if (!init) {
        activityTimer.start();
        activityStart = (unsigned int) activityTimer.read_ms();
        init = true;
    }
    if ((unsigned int)activityTimer.read_ms() - activityStart > 20) {

        v = sin(degrees * PI / 180);
        if (v < 0)
            v = 0;
        signOfLife = v;
        degrees += 2;
        activityStart = (unsigned int) activityTimer.read_ms();
    }
}

// A time-bound hook on startup to permit the user to access the wifly module
//
void WiFlyShell(Wifly & wifly, PC & pc)
{
    Timer userTimer;
    const int baudrates[] = {2400, 4800, 9600, 19200, 38400, 57600, 115200, 230400};
    static int i = 0;
    
    pc.printf("Pausing 5 sec for shell access to WiFly (press <enter>).\r\n");
    userTimer.start();
    do {
        if (pc.readable()) {
            bool loop = true;
            int c = pc.getc();
            pc.printf("Shell opened to WiFly module: <esc> to exit,\r\n  ctrl-B to reboot, ctrl-C to step baud\r\n");
            while (loop) {
                if (pc.readable()) {
                    int c = pc.getc();
                    switch (c) {
                        case '\x1B':        // <ESC>
                            loop = false;
                            break;
                        case '\x02':        // Ctrl-B
                            wifly.reboot();
                            break;
                        case '\x03':        // Ctrl-C
                            pc.printf("Setting to %d baud.\r\n", baudrates[i]);
                            wifly.baud(baudrates[i]);
                            i++;
                            if (i >= (sizeof(baudrates)/sizeof(baudrates[0])))
                                i = 0;
                            break;
                        default:
                            wifly.putc(c);
                            break;
                    }
                }
                if (wifly.readable())
                    pc.putc(wifly.getc());
            }
        }
    } while (userTimer.read_ms() < 5000);
    pc.printf("  WiFly shell closed.\r\n");
}


/// Smart-WiFly-WebServer is the creation of a web server using a WiFly module.
///
/// This is a working version, but it is not using the standardized wifly
/// library, which would not work for me... I had to make a number
/// of changes to get it to work well. After trying to minmimize those
/// changes, additional improvements became more and more clumsy, so
/// I have now been working to refactor where it makes sense, even as
/// it further deviates from the mbed-library version.
///
/// I created this because I couldn't find one that worked and wanted to
/// learn the WiFly module. There are a lot of possible improvements:
/// @li I think I'm not using the Socket interface as fully as I should.
/// @li I would like it to be faster (the interface from mbed to wifly is
///     limited to 230400 baud before it drops chars. HW handshake could
///     improve this, but the HW handshake pins on the LPC1768 are not
///     both brought out.
/// @li I would like to integrate this with the rtos.
/// @li If a page has multiple components (e.g. images), it appears
///     unreliable. It doesn't see the request for the extra component.
///     A poor workaround, for images, is to use a javascript to post-
///     load them. This is fundamentally a constraint of the WiFly module.
///
/// history:
/// @li 20130602 added .txt to the supported types (e.g. robots.txt), so
///         revised the credentials to .crd, which is an unsupported type
///         therefore won't be delivered to the user.
///
/// @note Copyright &copy; 2013 by Smartware Computing, all rights reserved.
///     Individuals may use this application for evaluation or non-commercial
///     purposes. Within this restriction, changes may be made to this application
///     as long as this copyright notice is retained. The user shall make
///     clear that their work is a derived work, and not the original.
///     Users of this application and sources accept this application "as is" and
///     shall hold harmless Smartware Computing, for any undesired results while
///     using this application - whether real or imagined.
///
/// @author David Smart, Smartware Computing
///
int main()
{
    char ssid[60], pass[60];
    INI ini("/local/config.ini");     // handles the credentials

    pc.baud(460800);    // I like a snappy terminal, so crank it up!
    pc.printf("\r\nSmart WiFly - Build " __DATE__ " " __TIME__ "\r\n");

    if (!ini.ReadString("Wifi", "ssid", ssid, sizeof(ssid))
    ||  !ini.ReadString("Wifi", "pass", pass, sizeof(pass))) {
        pc.printf("**** ERROR, credentials not found. ****\r\n");
        wait(1.0);
        error("     Waiting for user to fix this problem.         \r\n"); // flash and die
    }

    // first signals are tx, rx, reset, status pins.
    #if HW_ADAPTER == MBED_APP_BOARD
    WiflyInterface wifly( p9, p10, p30, p29, ssid, pass, WPA);
    #elif HW_ADAPTER == SMART_BOARD
    WiflyInterface wifly(p28, p27, p23, p24, ssid, pass, WPA);
    #endif

    // Bring the WiFly interface online
    do {
        wifly.init(); // start it up as a client of my network using DHCP
        wifly.baud(230400);     // Only 230K w/o HW flow control
        if (0 == wifly.connect())
            break;
        pc.printf(" Failed to connect, retrying...\r\n");
        wait(1.0);
        wifly.reset();
    } while (1);
    
    // Let the human know it is ready - if they are watching
    pc.printf("Connect to this server at http://%s:%d\r\n", wifly.getIPAddress(), HTTP_SERVER_PORT);

    #if HW_ADAPTER == MBED_APP_BOARD
    lcd.cls();
    lcd.locate(0,3);
    lcd.printf("mbed application board!\r\n");
    lcd.printf("http://%s:%d", wifly.getIPAddress(), HTTP_SERVER_PORT);
    #endif
    
    // Now let's instantiate the web server - along with a few settings:
    // the Wifly object, the port of interest (typically 80),
    // file system path to the static pages, if any.
    // the maximum parameters per transaction (in the query string),
    // the maximum number of dynamic pages that can be registered,
    // the serial port back thru USB (for development/logging)
    HTTPServer svr(HTTP_SERVER_PORT, WEBROOT, 15, 30, 10, &pc);

    // But for even more fun, I'm registering a few dynamic pages
    // You see the handlers for in DynamicPages.cpp.
    // Here you can see the path to place on the URL.
    // ex. http://192.168.1.140/dyn
    //svr.RegisterHandler("/",     SuperSimpleDynamicPage);
    svr.RegisterHandler("/dyn",  SimpleDynamicPage);
    svr.RegisterHandler("/led",  SimpleDynamicForm);
    
    svr.RegisterHandler("/dyn1", SimpleDynamicPage);
    svr.RegisterHandler("/dyn2", SimpleDynamicForm);
    svr.RegisterHandler("/pass", SimpleSecurityCheck);
    svr.RegisterHandler("/config", ServerConfig);
    svr.RegisterHandler("/FileIn", DynamicFileIn);

    onceinawhile.start();
    while (true) {
        if (onceinawhile.read_ms() >= 200)
            {
            onceinawhile.reset();
            //pc.printf("Largest free mem block is %d\r\n", Free());
            ShowSignOfLife();
            }
        svr.Poll();   // non-blocking, but also not deterministic
        if (pc.readable())
            WiFlyShell(wifly, pc);  // allow shell access at runtime (if user taps a key)
    }
}

