/// @file SW_String.h This is a small collection of string utilities that 
///         are not consistent across compilers.
///
/// @note Copyright &copy; 2016 by Smartware Computing, all rights reserved.
///     This software may be used to derive new software, as long as
///     this copyright statement remains in the source file.
/// @author David Smart
///
#include "string.h"


/// A more secure version of strcat
///
/// This function is like a wrapper on strcat, to first validate the concatination
/// and then if all parameters appear good, it will call strcat. It will not
/// permit overlapping source and destination.
///
/// If there is an error, no concatination is performed.
///
/// @note This has a different return value than the normal strcat.
///
/// @param[out] dst is a pointer to the start of the destination buffer (not necessarily
///             where the next string will appear).
/// @param[in] dstSize defines the size of the destination buffer.
/// @param[in] src is a pointer to the source.
///
/// @returns 
///     - 0 = no error
///     - -1 = destination pointer invalid
///     - -2 = source is too big to append into the destination
///     - -3 = overlap between src and dst
///
int strcat_s(char * dst, size_t dstSize, const char * src);

/// A more secure version of strcpy
///
/// This function is like a wrapper on strcpy, to first validate the concatination
/// and then if all parameters appear good, it will call strcpy. It will not
/// permit overlapping source and destination.
///
/// If there is an error, no copy is performed.
///
/// @note This has a different return value than the normal strcpy.
///
/// @param[out] dst is a pointer to the start of the destination buffer.
/// @param[in] dstSize defines the size of the destination buffer.
/// @param[in] src is a pointer to the source.
///
/// @returns 
///     - 0 = no error
///     - -1 = destination pointer invalid
///     - -2 = source is too big to append into the destination
///     - -3 = overlap between src and dst
///
int strcpy_s(char * dst, size_t dstSize, const char * src);

/// sw_tolower exists because not all compiler libraries have this function
///
/// This takes a character and if it is upper-case, it converts it to
/// lower-case and returns it.
///
/// @note an alternate means would be a 256-entry lookup table. Very fast...
///
/// @param a is the character to convert
/// @returns the lower case equivalent to a
///
char sw_tolower(char a);

/// sw_strnicmp exists because not all compiler libraries have this function.
///
/// In a case-insensitive compare, evaluate 'n' characters of the left and
/// right referenced strings.
///
/// @note Some compilers have strnicmp, others _strnicmp, and others have C++ 
/// methods, which is outside the scope of this C-portable set of functions.
///
/// @param l is a pointer to the string on the left
/// @param r is a pointer to the string on the right
/// @param n is the number of characters to compare
/// @returns -1 if l < r
/// @returns 0 if l == r
/// @returns +1 if l > r
///
int sw_strnicmp(const char *l, const char *r, size_t n);

/// sw_stristr exists because not all compiler libraries have this function.
///
/// In a case-insenstive search, try to find the needle in the haystack.
///
/// @param haystack is a pointer to string being searched
/// @param needle is a pointer to a string to find
/// @returns a pointer to the found needle in the haystack, or NULL
///
const char * sw_stristr(const char * haystack, const char * needle);


/// sw_stristr exists because not all compiler libraries have this function.
///
/// In a case-insenstive search, try to find the needle in the haystack.
///
/// @param haystack is a pointer to string being searched
/// @param needle is a pointer to a string to find
/// @returns a pointer to the found needle in the haystack, or NULL
///
char * sw_stristr(char * haystack, const char * needle);