//
// SSDP Server
//
// This block creates and manages the network interface for an SSDP Server,
// which can make this node easily discoverable on the network.
//
// @note SSDP is not a secure protocol and is cited as creating vulnerabilities,
//       so this should be understood or used with caution on a controlled
//       private network.
//

#include "mbed.h"
#include "rtos.h"

/// Configuration data for the SSDP server.
///
/// When internal to the SSDP server, it will manage the memory (dynamic allocations
/// and free).
///
/// When an instance of this structure is passed to the SSDP constructor, it is
/// up to the caller to ensure that the data being pointed to is persistent, where
/// it should be. The 'name' is one field that could be changed at runtime, in 
/// order to reflect the name change to the network.
///
typedef struct {
    char * name;        ///< pointer to the friendly name
    char * ident;       ///< pointer to a unique identity number, like a mac address
    char * ipAddr;      ///< pointer to the node IP address
    int port;           ///< port number of the local server
} SSDP_Config_T;


/// SSDP Server
/// 
/// This file documents the SSDP Server, which can be implanted into a node
/// so that it is discoverable on Ethernet, and in Windows Network Explorer
/// view.
///
/// Many basics are defined in order to satisfy the SSDP discovery process.
/// - the hosting node has a web server running, which will serve various responses.
/// - the hosting node shall provide a /setup.xml which satisfies the SSDP requirements.
/// - depending on the application, many additional files may need to be supported.
/// 
/// @code
///    HTTPServer svr(Server_Port, Server_Root, 15, 30, 20, 50, &pc);
///    svr.RegisterHandler("/", RootPage);
///    svr.RegisterHandler("/setup.xml", Setup_xml);
///    SSDP ssdp("Friendly Node", eth.getMACAddress(), eth.getIPAddress(), Server_Port);
///    
///    while (1) {
///        led4 = !led4;
///        svr.Poll();     // non-blocking, but also not deterministic
///        Thread::yield();
///        if (criteria)
///           ssdp.SetFriendlyName("New Node Name");
///    }
/// @endcode
///
class SSDP {
private:
    
public:
    /// Constructor for the SSDP server
    ///
    /// @param[in] name is a pointer to a string containing the name.
    /// @param[in] ident is a pointer to an identity string of this node, e.g. a mac address.
    /// @param[in] ipAddr is a pointer to an IP Address string of this node.
    /// @param[in] port is an integer port number for the local web server.
    ///
    SSDP(const char * name, const char * ident, const char * ipAddr, int port);
    
    /// Constructor for the SSDP server, as an alternate to the parameter version.
    ///
    /// @param[in] config is a pointer to configuration structure which contains
    ///             the individual elements;
    ///             - pointer to the const char * friendly name 
    ///             - pointer to the const char * identity
    ///             - pointer to the const char * IP Address of this node
    ///             - port number for the web server 
    ///
    /// @note the items being pointed to in this structure must be persistent. Only
    ///         the friendly name can change at runtime.
    ///
    SSDP(const SSDP_Config_T * config);
    
    /// Destructor
    ///
    ~SSDP();
    
    /// Set the friendly name for this node
    ///
    /// @param[in] name is a pointer to a string containing the name.
    /// @returns true if the name was set (memory was available).
    ///
    bool SetFriendlyName(const char * name);

    /// Set the identity for this node
    ///
    /// @param[in] ident is a pointer to a string containing the identity.
    /// @returns true if the name was set (memory was available).
    ///
    bool SetIdentity(const char * ident);

    /// Set the IP Address for this node
    ///
    /// @param[in] ipAddr is a pointer to a string containing the ipAddress.
    /// @returns true if the IP Address was set (memory was available).
    ///
    bool SetIPAddress(const char * ipAddr);

    /// Set the Port
    ///
    /// @param[in] port is the port number of the local web server.
    /// @returns true if the port number was set.
    ///
    bool SetPort(int port);
    
  
private:

    /// Delete the friendly name that was assigned.
    ///
    /// This frees the memory that was previously allocated.
    ///
    void DelFriendlyName();

    /// Delete the identity that was assigned.
    ///
    /// This frees the memory that was previously allocated.
    ///
    void DelIdentity();

    /// Delete the friendly IP Address that was assigned.
    ///
    /// This frees the memory that was previously allocated.
    ///
    void DelIPAddress();

    /// Starts a dedicated thread just to listen for SSDP Queries
    /// and respond to them.
    ///
    void StartListener();
    
    Thread * pThr;

    /// Type of Notification
    typedef enum {
        nt_root,        ///< upnp:rootdevice notification
    } NotifyType_t;
    
    /// On power-up and on state-change, notify events are sent
    /// for which listeners may react.
    ///
    void SendNotify(NotifyType_t nt = nt_root);
    
    SSDP_Config_T _config;       ///< the configuration

};