//
// The GSL1680 is a 32-bit RISC micro. It must be bootstrapped with a program before it can be used
// as a touchscreen controller.
//
// Below is the program that is installed into it to give it that capability.
//
// Each element consists of two values:
// * Register Offset to write into
// * 32-Bit Value to write into that register/memory location.
//
// The special register 0xf0 is used to set the memory Page, where each page is 128B (range 0x00 to 0x7F)
//
// This table layout appears quite memory inefficient, but is pretty well defacto standardized.
// I'm not sure if the compiler 32-bit aligns the elements, which wastes 3 bytes for every
// other value. Once functionality is proven, it is pretty easy to imagine some significant
// memory optimization.
//
// By online information, it is stated that some versions support only 5-finger touch and
// others support 10-finger touch. It is unknown what this version does, so some experiments
// with alternates may be appropriate (see below).

#ifndef RA8875_TOUCH_GSL1680_FIRMWARE_H
#define RA8875_TOUCH_GSL1680_FIRMWARE_H


#define GSL1680_FW_0 0
#define GSL1680_FW_1 1
#define GSL1680_FW_2 2

// SELECT WHICH FIRMWARE VERSION TO USE
// Two versions have been found at sites that report functioning controllers. It is unknown
// if either or both will work here and what may be different between them. Hopefully, one
// of them will support 10-touch.
//
// Presently, there are two versions in here -
//  - GSL1680_FW_0      // This special "null-ish" version, which does not compile in any GSL code
//  - GSL1680_FW_1      // This preferred version
//  - GSL1680_FW_2      // This second version

#define GSL1680_FW GSL1680_FW_0

// DECLARE NUMBER OF TOUCHPOINTS SUPPORTED
// Based on the chosen firmware, this declares how many touchpoints are supported, which
// affects RAM for managing the touches.
//
// Probably, somebody with this display will want to try FW_1 configured for 10-touch and
// see if it works. If it doesn't, dial it down to 5 and document that constraint. Then,
// try FW_2 for 10-touch and see if that works - documenting its capability.
//
#define GSL1680_TOUCH_POINTS 10


// Data Structure for the firmware installation table that is below
struct fw_data
{
    uint8_t offset;
    uint32_t val;
};

#define ARRAY_SIZE(a) (sizeof(a)/sizeof(a[0]))

#if GSL1680_FW == GSL1680_FW_0

const struct fw_data GSLX680_FW[] = {
    //GSL_1680E+1688E+2681B+2682B_V1.3.4_2013.02.28
    {0x00,0x00},
};

#elif GSL1680_FW == GSL1680_FW_1

// In searching online, you can find different code-bases.
const struct fw_data GSLX680_FW[] = {
    {0xf0,0x2},
    {0x00,0x00000000}, {0x04,0x00000000}, {0x08,0x00000000}, {0x0c,0x00000000}, {0x10,0x00000000}, {0x14,0x00000000}, {0x18,0x00000000}, {0x1c,0x00000000},
    {0x20,0x00000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0xf801000f}, {0x68,0x00066414}, {0x6c,0x1001020a}, {0x70,0x00000fff}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x0a0f0a0f},

    {0xf0,0x3},
    {0x00,0x3a6cc7ea}, {0x04,0x00000000}, {0x08,0x00000000}, {0x0c,0x00000000}, {0x10,0x00000000}, {0x14,0x00000000}, {0x18,0x00000000}, {0x1c,0x00000000},
    {0x20,0x00000000}, {0x24,0x00005100}, {0x28,0x00008e00}, {0x2c,0x00000000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x1a0ac00a}, {0x68,0x00000002}, {0x6c,0x0000000f}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0xffffffec}, {0x7c,0x00000000},

    {0xf0,0x4},
    {0x00,0x00000000}, {0x04,0x0001660b}, {0x08,0x00000064}, {0x0c,0x00000000}, {0x10,0xfe0cff06}, {0x14,0x00000000}, {0x18,0x00000000}, {0x1c,0x00000000},
    {0x20,0x00000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00010000}, {0x34,0x00000fff}, {0x38,0x0000000a}, {0x3c,0x00000258},
    {0x40,0x00000000}, {0x44,0x04020a00}, {0x48,0x0014012c}, {0x4c,0x9a080007}, {0x50,0x00000000}, {0x54,0x00010203}, {0x58,0x04050607}, {0x5c,0x08090a0b},
    {0x60,0x0c0d0e0f}, {0x64,0x10111213}, {0x68,0x14151617}, {0x6c,0x18191a1b}, {0x70,0x1c1d1e1f}, {0x74,0x0014000a}, {0x78,0x80808080}, {0x7c,0xcba981f4},

    {0xf0,0x5},
    {0x00,0x00000000}, {0x04,0x00000005}, {0x08,0x000000b4}, {0x0c,0x66666666}, {0x10,0x0000000a}, {0x14,0x00000000}, {0x18,0x00000fff}, {0x1c,0x10000000},
    {0x20,0x10000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000400}, {0x30,0x80808080}, {0x34,0x80808080}, {0x38,0x80808080}, {0x3c,0x80808080},
    {0x40,0x80808080}, {0x44,0x80808080}, {0x48,0x80808080}, {0x4c,0x80808000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000221}, {0x78,0x0000000f}, {0x7c,0x0000000a},

    {0xf0,0x6},
    {0x00,0x0000000f}, {0x04,0x00000000}, {0x08,0x0000000a}, {0x0c,0x04020402}, {0x10,0x00000032}, {0x14,0x140a010a}, {0x18,0x00000000}, {0x1c,0x00000001},
    {0x20,0x00002904}, {0x24,0x000001e0}, {0x28,0x00000320}, {0x2c,0xf801000f}, {0x30,0xf8010005}, {0x34,0x00000005}, {0x38,0x00000002}, {0x3c,0x00000fff},
    {0x40,0x80000000}, {0x44,0x00180018}, {0x48,0x00000fff}, {0x4c,0x04040402}, {0x50,0x00030002}, {0x54,0x00000e10}, {0x58,0x000a012c}, {0x5c,0x1b6db688},
    {0x60,0x20100804}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x0000012c}, {0x78,0x000a003c}, {0x7c,0x00000000},

    {0xf0,0x7},
    {0x00,0x01040007}, {0x04,0x03060209}, {0x08,0x0508040a}, {0x0c,0x07110610}, {0x10,0x09130812}, {0x14,0x00123456}, {0x18,0x00000000}, {0x1c,0x000a0078},
    {0x20,0x00001081}, {0x24,0xff080010}, {0x28,0xff080120}, {0x2c,0xff080140}, {0x30,0xff080160}, {0x34,0x0000012c}, {0x38,0x00000104}, {0x3c,0x000000dc},
    {0x40,0x00000000}, {0x44,0x012c012c}, {0x48,0x00000103}, {0x4c,0x320f0f03}, {0x50,0x00000000}, {0x54,0x00000001}, {0x58,0x000e0000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00070002}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00012345}, {0x7c,0x006789ab},

    {0xf0,0x8},
    {0x00,0x026f028f}, {0x04,0x02af02cf}, {0x08,0x02ef030f}, {0x0c,0x032f034f}, {0x10,0x01f301f4}, {0x14,0x01f501f6}, {0x18,0x01f701f8}, {0x1c,0x11f901fa},
    {0x20,0x01f401f5}, {0x24,0x01f601f7}, {0x28,0x01f801f9}, {0x2c,0x01fa0000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x01020304}, {0x44,0x05060708}, {0x48,0x090a0000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00040003}, {0x5c,0x00000008},
    {0x60,0x00000190}, {0x64,0x04030201}, {0x68,0x08080402}, {0x6c,0x006600cd}, {0x70,0x000000cd}, {0x74,0x0000008c}, {0x78,0x00000000}, {0x7c,0x0000000a},

    {0xf0,0x9},
    {0x00,0xff080094}, {0x04,0x00070011}, {0x08,0xff080090}, {0x0c,0x00040000}, {0x10,0xff080068}, {0x14,0x00030000}, {0x18,0xff080064}, {0x1c,0x01002582},
    {0x20,0xff080060}, {0x24,0x00000000}, {0x28,0xff08004c}, {0x2c,0x00197fff}, {0x30,0xfffffff0}, {0x34,0x00000000}, {0x38,0xfffffff0}, {0x3c,0x00000000},
    {0x40,0xfffffff0}, {0x44,0x00000000}, {0x48,0xfffffff0}, {0x4c,0x00000000}, {0x50,0xfffffff0}, {0x54,0x00000000}, {0x58,0xfffffff0}, {0x5c,0x00000000},
    {0x60,0xfffffff0}, {0x64,0x00000000}, {0x68,0xfffffff0}, {0x6c,0x00000000}, {0x70,0xfffffff0}, {0x74,0x00000000}, {0x78,0xfffffff0}, {0x7c,0x00000000},

    {0xf0,0x1e},
    {0x00,0x5a5a0f00}, {0x04,0x000028a0}, {0x08,0x00002914}, {0x0c,0x00002a48}, {0x10,0x00002b44}, {0x14,0x00002cf8}, {0x18,0x00002ec4}, {0x1c,0x00003074},
    {0x20,0x0000321c}, {0x24,0x00003320}, {0x28,0x000033d0}, {0x2c,0x00003608}, {0x30,0x00003710}, {0x34,0x0000397c}, {0x38,0x000039f4}, {0x3c,0x00003b8c},
    {0x40,0x00003e74}, {0x44,0x00004210}, {0x48,0x000042f8}, {0x4c,0x0000435c}, {0x50,0x000045bc}, {0x54,0x000046dc}, {0x58,0x000047c8}, {0x5c,0x00004814},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},

    {0xf0,0x1f},
    {0x00,0x00000000}, {0x04,0x00000000}, {0x08,0x00000000}, {0x0c,0x00000000}, {0x10,0x00000000}, {0x14,0x00000000}, {0x18,0x00000000}, {0x1c,0x00000000},
    {0x20,0x00000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x000024b0}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00002530}, {0x50,0x000025f0}, {0x54,0x00000000}, {0x58,0x000026a0}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x000027d0}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x000048cc}, {0x7c,0x5a5a0ffc},

    {0xf0,0x0},
    {0x00,0x01000000}, {0x04,0x01000000}, {0x08,0x01000000}, {0x0c,0x233fc0c0}, {0x10,0xa2146004}, {0x14,0xa4102000}, {0x18,0xe4244000}, {0x1c,0x233fc0c0},
    {0x20,0xa2146010}, {0x24,0x2500003f}, {0x28,0xa414a3ff}, {0x2c,0xe4244000}, {0x30,0x01000000}, {0x34,0x821020e0}, {0x38,0x81880001}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0xa410200f}, {0x48,0xe4a00040}, {0x4c,0x01000000}, {0x50,0xa0100000}, {0x54,0xa2100000}, {0x58,0xa4100000}, {0x5c,0xa6100000},
    {0x60,0xa8100000}, {0x64,0xaa100000}, {0x68,0xac100000}, {0x6c,0xae100000}, {0x70,0x90100000}, {0x74,0x92100000}, {0x78,0x94100000}, {0x7c,0x96100000},

    {0xf0,0x1},
    {0x00,0x98100000}, {0x04,0x9a100000}, {0x08,0x9c100000}, {0x0c,0x9e100000}, {0x10,0x84100000}, {0x14,0x01000000}, {0x18,0x01000000}, {0x1c,0x82100000},
    {0x20,0x81900001}, {0x24,0x82100000}, {0x28,0x81980001}, {0x2c,0x81800000}, {0x30,0x01000000}, {0x34,0x01000000}, {0x38,0x01000000}, {0x3c,0xbc102cfc},
    {0x40,0x9c102cf8}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0x27001040}, {0x50,0xa614e00f}, {0x54,0xe6a00040}, {0x58,0x01000000}, {0x5c,0x40000309},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x10bfffff}, {0x6c,0x01000000}, {0x70,0x03169684}, {0x74,0x82106053}, {0x78,0x03296945}, {0x7c,0x82106288},

    {0xf0,0x1a},
    {0x00,0x9de3bf98}, {0x04,0x40000010}, {0x08,0x01000000}, {0x0c,0x40000007}, {0x10,0x01000000}, {0x14,0x4000031b}, {0x18,0x01000000}, {0x1c,0x40000015},
    {0x20,0x01000000}, {0x24,0x30bffffe}, {0x28,0x82102001}, {0x2c,0x81904000}, {0x30,0x01000000}, {0x34,0x01000000}, {0x38,0x01000000}, {0x3c,0x81c3e008},
    {0x40,0x01000000}, {0x44,0x03000008}, {0x48,0x82106342}, {0x4c,0xa3804000}, {0x50,0x03000004}, {0x54,0x82106000}, {0x58,0x81984000}, {0x5c,0x01000000},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x81c3e008}, {0x6c,0x01000000}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x01000000}, {0x7c,0xa7800000},

    {0xf0,0x1b},
    {0x00,0x01000000}, {0x04,0x01000000}, {0x08,0x01000000}, {0x0c,0x81c3e008}, {0x10,0x01000000}, {0x14,0x80a22000}, {0x18,0x02800006}, {0x1c,0x01000000},
    {0x20,0x01000000}, {0x24,0x90823fff}, {0x28,0x12bffffe}, {0x2c,0x01000000}, {0x30,0x81c3e008}, {0x34,0x01000000}, {0x38,0x01000000}, {0x3c,0x05001040},
    {0x40,0x8410a00f}, {0x44,0xc4a00040}, {0x48,0x01000000}, {0x4c,0x81c3e008}, {0x50,0x01000000}, {0x54,0x9de3bf18}, {0x58,0xb2067fff}, {0x5c,0x96100018},
    {0x60,0xf227bf7c}, {0x64,0xc027bf78}, {0x68,0x98102000}, {0x6c,0xb007bff8}, {0x70,0xd4063f80}, {0x74,0xda063f84}, {0x78,0xb810000a}, {0x7c,0x98033fff},

    {0xf0,0x1c},
    {0x00,0xb0063ff8}, {0x04,0x80a2800d}, {0x08,0x16800031}, {0x0c,0xb610000d}, {0x10,0xbb2aa002}, {0x14,0x832b6002}, {0x18,0xb207400b}, {0x1c,0xb400400b},
    {0x20,0xde02c01d}, {0x24,0x80a7001b}, {0x28,0x1680001c}, {0x2c,0x01000000}, {0x30,0xc2068000}, {0x34,0x80a0400f}, {0x38,0x04800005}, {0x3c,0x80a7001b},
    {0x40,0xb606ffff}, {0x44,0x10bffff8}, {0x48,0xb406bffc}, {0x4c,0x16800013}, {0x50,0x80a7001b}, {0x54,0xc2068000}, {0x58,0xc2264000}, {0x5c,0xb8072001},
    {0x60,0x80a7001b}, {0x64,0x1680000d}, {0x68,0xb2066004}, {0x6c,0xc2064000}, {0x70,0x80a0400f}, {0x74,0x26bffffb}, {0x78,0xb8072001}, {0x7c,0x80a7001b},

    {0xf0,0x1d},
    {0x00,0x16800006}, {0x04,0x01000000}, {0x08,0xc2268000}, {0x0c,0xb606ffff}, {0x10,0xb406bffc}, {0x14,0x80a7001b}, {0x18,0x12bfffe4}, {0x1c,0x80a7001b},
    {0x20,0xb0062008}, {0x24,0x82073fff}, {0x28,0xc2263f84}, {0x2c,0xd4263f80}, {0x30,0x832f2002}, {0x34,0xb0062008}, {0x38,0xde22c001}, {0x3c,0xba072001},
    {0x40,0xfa263f80}, {0x44,0xda263f84}, {0x48,0x98032002}, {0x4c,0x80a33fff}, {0x50,0x34bfffc9}, {0x54,0xd4063f80}, {0x58,0x81c7e008}, {0x5c,0x81e80000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},

    {0xf0,0x20},
    {0x00,0x83580000}, {0x04,0x82086ff0}, {0x08,0x83306004}, {0x0c,0x80a06005}, {0x10,0x02800027}, {0x14,0x01000000}, {0x18,0x80a06006}, {0x1c,0x0280003c},
    {0x20,0x01000000}, {0x24,0x80a06015}, {0x28,0x02800054}, {0x2c,0x01000000}, {0x30,0x80a0602a}, {0x34,0x02800090}, {0x38,0x01000000}, {0x3c,0x80a06018},
    {0x40,0x02800085}, {0x44,0x01000000}, {0x48,0x073fc180}, {0x4c,0x8610e03c}, {0x50,0x05169680}, {0x54,0x84004002}, {0x58,0xc420c000}, {0x5c,0x073fc000},
    {0x60,0x8610e020}, {0x64,0x84102001}, {0x68,0xc420c000}, {0x6c,0x0500000c}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x8480bfff}, {0x7c,0x12bffffe},

    {0xf0,0x21},
    {0x00,0x01000000}, {0x04,0x01000000}, {0x08,0x073fc000}, {0x0c,0x8610e020}, {0x10,0x84102000}, {0x14,0xc420c000}, {0x18,0x01000000}, {0x1c,0x01000000},
    {0x20,0x81c44000}, {0x24,0x81cc8000}, {0x28,0x01000000}, {0x2c,0xa7500000}, {0x30,0xa92ce002}, {0x34,0xa734e001}, {0x38,0xa614c014}, {0x3c,0xa60ce007},
    {0x40,0x81900000}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0x81e00000}, {0x50,0xe03ba000}, {0x54,0xe43ba008}, {0x58,0xe83ba010}, {0x5c,0xec3ba018},
    {0x60,0xf03ba020}, {0x64,0xf43ba028}, {0x68,0xf83ba030}, {0x6c,0xfc3ba038}, {0x70,0x81e80000}, {0x74,0x8194c000}, {0x78,0x01000000}, {0x7c,0x01000000},

    {0xf0,0x22},
    {0x00,0x81c44000}, {0x04,0x81cc8000}, {0x08,0x01000000}, {0x0c,0xa7500000}, {0x10,0xa934e002}, {0x14,0xa72ce001}, {0x18,0xa614c014}, {0x1c,0xa60ce007},
    {0x20,0x81900000}, {0x24,0x01000000}, {0x28,0x01000000}, {0x2c,0x81e80000}, {0x30,0x81e80000}, {0x34,0xe01ba000}, {0x38,0xe41ba008}, {0x3c,0xe81ba010},
    {0x40,0xec1ba018}, {0x44,0xf01ba020}, {0x48,0xf41ba028}, {0x4c,0xf81ba030}, {0x50,0xfc1ba038}, {0x54,0x81e00000}, {0x58,0x81e00000}, {0x5c,0x8194c000},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x81c44000}, {0x6c,0x81cc8000}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x82102010}, {0x7c,0x273fc0c0},

    {0xf0,0x23},
    {0x00,0xa614e010}, {0x04,0xc224c000}, {0x08,0x01000000}, {0x0c,0x033fc0c0}, {0x10,0x82106004}, {0x14,0xa6102000}, {0x18,0xe6204000}, {0x1c,0x01000000},
    {0x20,0x01000000}, {0x24,0x01000000}, {0x28,0xa6102020}, {0x2c,0x83480000}, {0x30,0x82104013}, {0x34,0x81884000}, {0x38,0x01000000}, {0x3c,0x4000044d},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0xa7500000}, {0x50,0xa934e002}, {0x54,0xa72ce001}, {0x58,0xa614c014}, {0x5c,0xa60ce007},
    {0x60,0x81900000}, {0x64,0x01000000}, {0x68,0x81e80000}, {0x6c,0xe01ba000}, {0x70,0xe41ba008}, {0x74,0xe81ba010}, {0x78,0xec1ba018}, {0x7c,0xf01ba020},

    {0xf0,0x24},
    {0x00,0xf41ba028}, {0x04,0xf81ba030}, {0x08,0xfc1ba038}, {0x0c,0x81e00000}, {0x10,0x8194c000}, {0x14,0x01000000}, {0x18,0xa6102020}, {0x1c,0x83480000},
    {0x20,0x82284013}, {0x24,0x81884000}, {0x28,0x01000000}, {0x2c,0x033fc0c0}, {0x30,0x82106004}, {0x34,0xa6103fff}, {0x38,0xe6204000}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x81c44000}, {0x4c,0x81cc8000}, {0x50,0x01000000}, {0x54,0x82102080}, {0x58,0x273fc0c0}, {0x5c,0xa614e010},
    {0x60,0xc224c000}, {0x64,0x01000000}, {0x68,0x81c44000}, {0x6c,0x81cc8000}, {0x70,0x01000000}, {0x74,0x81c48000}, {0x78,0x81cca004}, {0x7c,0x01000000},

    {0xf0,0x25},
    {0x00,0x1b3fc140}, {0x04,0x82136048}, {0x08,0xc2104000}, {0x0c,0x82006003}, {0x10,0x82086003}, {0x14,0x83286002}, {0x18,0x9a136080}, {0x1c,0x81c3e008},
    {0x20,0xd000400d}, {0x24,0x94102000}, {0x28,0x96102000}, {0x2c,0x832ae002}, {0x30,0xd20063a4}, {0x34,0x98102000}, {0x38,0x832b2002}, {0x3c,0x9b2aa002},
    {0x40,0xda02000d}, {0x44,0x98032001}, {0x48,0xda224001}, {0x4c,0x80a32005}, {0x50,0x04bffffa}, {0x54,0x9402a001}, {0x58,0x9602e001}, {0x5c,0x80a2e003},
    {0x60,0x04bffff4}, {0x64,0x832ae002}, {0x68,0x81c3e008}, {0x6c,0x01000000}, {0x70,0x82020008}, {0x74,0x82004008}, {0x78,0x9b326002}, {0x7c,0x82004001},

    {0xf0,0x26},
    {0x00,0x8200400d}, {0x04,0x83286002}, {0x08,0x920a6003}, {0x0c,0x932a6003}, {0x10,0xd00065b0}, {0x14,0x91320009}, {0x18,0x81c3e008}, {0x1c,0x900a20ff},
    {0x20,0x9a020008}, {0x24,0x9a034008}, {0x28,0x97326002}, {0x2c,0x9a03400d}, {0x30,0x9a03400b}, {0x34,0x920a6003}, {0x38,0x9b2b6002}, {0x3c,0x932a6003},
    {0x40,0x821020ff}, {0x44,0xd80365b0}, {0x48,0x83284009}, {0x4c,0x822b0001}, {0x50,0x952a8009}, {0x54,0x8210400a}, {0x58,0xc22365b0}, {0x5c,0x912a2002},
    {0x60,0xda0223a4}, {0x64,0x972ae002}, {0x68,0x81c3e008}, {0x6c,0xc223400b}, {0x70,0x82102005}, {0x74,0x1b3fc200}, {0x78,0x82204008}, {0x7c,0x9a136008},

    {0xf0,0x27},
    {0x00,0x83286002}, {0x04,0xd0034000}, {0x08,0x91320001}, {0x0c,0x81c3e008}, {0x10,0x900a200f}, {0x14,0x9de3bf58}, {0x18,0xa12e2002}, {0x1c,0xda1423da},
    {0x20,0xd61423d8}, {0x24,0xc200247c}, {0x28,0xb022c00d}, {0x2c,0x825b4001}, {0x30,0xb21e6001}, {0x34,0x92862001}, {0x38,0x0280000c}, {0x3c,0x9a004019},
    {0x40,0xb0100009}, {0x44,0x9807bfb8}, {0x48,0x8203400d}, {0x4c,0xc2168001}, {0x50,0xc2230000}, {0x54,0xc200247c}, {0x58,0x9a034001}, {0x5c,0xb0863fff},
    {0x60,0x12bffffa}, {0x64,0x98032004}, {0x68,0x7ffffe7b}, {0x6c,0x9007bfb8}, {0x70,0xda1423ea}, {0x74,0xd61423e8}, {0x78,0x80a3400b}, {0x7c,0x92102000},

    {0xf0,0x28},
    {0x00,0x1880000b}, {0x04,0xb010000d}, {0x08,0x832b6002}, {0x0c,0x8200401e}, {0x10,0x98007fb8}, {0x14,0xc2030000}, {0x18,0xb0062001}, {0x1c,0x92024001},
    {0x20,0x80a6000b}, {0x24,0x08bffffc}, {0x28,0x98032004}, {0x2c,0xb022c00d}, {0x30,0xb0062001}, {0x34,0x81800000}, {0x38,0x01000000}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0xb0724018}, {0x48,0x81c7e008}, {0x4c,0x81e80000}, {0x50,0x033fc200}, {0x54,0x961060a0}, {0x58,0x98102000}, {0x5c,0x832b2002},
    {0x60,0x9a03000c}, {0x64,0xda136400}, {0x68,0x98032001}, {0x6c,0x80a3200f}, {0x70,0x04bffffb}, {0x74,0xda20400b}, {0x78,0x81c3e008}, {0x7c,0x01000000},

    {0xf0,0x29},
    {0x00,0x9de3bf98}, {0x04,0xc200247c}, {0x08,0x83306001}, {0x0c,0x80a60001}, {0x10,0x1a800006}, {0x14,0x90100018}, {0x18,0x7fffffb6}, {0x1c,0x01000000},
    {0x20,0x10800006}, {0x24,0xb0020008}, {0x28,0x7fffffb2}, {0x2c,0x90260001}, {0x30,0x90020008}, {0x34,0xb0022001}, {0x38,0x81c7e008}, {0x3c,0x81e80000},
    {0x40,0x9de3bf98}, {0x44,0xa8102000}, {0x48,0xc20023d4}, {0x4c,0x80a50001}, {0x50,0x1a800057}, {0x54,0xe2002348}, {0x58,0xa4102000}, {0x5c,0xc200247c},
    {0x60,0x80a48001}, {0x64,0x3a80004e}, {0x68,0xa8052001}, {0x6c,0x7fffffe5}, {0x70,0x90100012}, {0x74,0xaa100008}, {0x78,0x90100014}, {0x7c,0x7fffff7d},

    {0xf0,0x2a},
    {0x00,0x92100015}, {0x04,0x80a62000}, {0x08,0x12800004}, {0x0c,0xa0100008}, {0x10,0x10800016}, {0x14,0xa0102000}, {0x18,0x80a62009}, {0x1c,0x18800011},
    {0x20,0x80a62007}, {0x24,0x7fffff57}, {0x28,0x01000000}, {0x2c,0x94100008}, {0x30,0x90100014}, {0x34,0x7fffff98}, {0x38,0x92100012}, {0x3c,0x80a20011},
    {0x40,0x04800007}, {0x44,0xa6100008}, {0x48,0x9a102008}, {0x4c,0x9a234018}, {0x50,0x82102001}, {0x54,0x8328400d}, {0x58,0xa02c0001}, {0x5c,0x80a62007},
    {0x60,0x18800008}, {0x64,0x80a62008}, {0x68,0x9a102007}, {0x6c,0x9a234018}, {0x70,0x82102001}, {0x74,0x8328400d}, {0x78,0x10800023}, {0x7c,0xa0140001},

    {0xf0,0x2b},
    {0x00,0x1280000a}, {0x04,0x821e2009}, {0x08,0x80a420fe}, {0x0c,0x24800002}, {0x10,0xa0042001}, {0x14,0x03000019}, {0x18,0x9b2ca002}, {0x1c,0x82106080},
    {0x20,0x10800019}, {0x24,0xe6234001}, {0x28,0x80a00001}, {0x2c,0x9a603fff}, {0x30,0x80a420fe}, {0x34,0x04800003}, {0x38,0x82102001}, {0x3c,0x82102000},
    {0x40,0x808b4001}, {0x44,0x02800011}, {0x48,0x92100015}, {0x4c,0x03000019}, {0x50,0x9b2ca002}, {0x54,0x82106080}, {0x58,0xc2034001}, {0x5c,0x80a04011},
    {0x60,0x18800003}, {0x64,0x9a204011}, {0x68,0x9a244001}, {0x6c,0x80a4c011}, {0x70,0x14800003}, {0x74,0x8224c011}, {0x78,0x82244013}, {0x7c,0x80a34001},

    {0xf0,0x2c},
    {0x00,0xa0642000}, {0x04,0x92100015}, {0x08,0x94100010}, {0x0c,0x7fffff45}, {0x10,0x90100014}, {0x14,0x10bfffb2}, {0x18,0xa404a001}, {0x1c,0xc20023d4},
    {0x20,0x80a50001}, {0x24,0x0abfffae}, {0x28,0xa4102000}, {0x2c,0x81c7e008}, {0x30,0x81e80000}, {0x34,0x98102000}, {0x38,0x9610201e}, {0x3c,0x80a22000},
    {0x40,0x12800003}, {0x44,0x8210000c}, {0x48,0x8222c00c}, {0x4c,0x83286002}, {0x50,0xda006480}, {0x54,0x80a37ff0}, {0x58,0x02800006}, {0x5c,0x98032002},
    {0x60,0xc2006484}, {0x64,0x80a3201f}, {0x68,0x04bffff5}, {0x6c,0xc2234000}, {0x70,0x81c3e008}, {0x74,0x01000000}, {0x78,0x9de3bf98}, {0x7c,0xda002508},

    {0xf0,0x2d},
    {0x00,0x033fc000}, {0x04,0x9a0b4001}, {0x08,0x173fc180}, {0x0c,0x03202020}, {0x10,0x82106080}, {0x14,0x9812e038}, {0x18,0xc2230000}, {0x1c,0x03168000},
    {0x20,0x80a34001}, {0x24,0xc022c000}, {0x28,0xa6102000}, {0x2c,0x1280000b}, {0x30,0xa8102000}, {0x34,0x0300003f}, {0x38,0xda002548}, {0x3c,0x821063ff},
    {0x40,0x9a0b4001}, {0x44,0xd8002508}, {0x48,0x033fffc0}, {0x4c,0x980b0001}, {0x50,0x9a13400c}, {0x54,0xda202548}, {0x58,0x80a62000}, {0x5c,0x16800027},
    {0x60,0x1b296956}, {0x64,0x1b3fc040}, {0x68,0xc2002548}, {0x6c,0x8208400d}, {0x70,0x1b168040}, {0x74,0x80a0400d}, {0x78,0x22800002}, {0x7c,0xa6102001},

    {0xf0,0x2e},
    {0x00,0xda002654}, {0x04,0xc20021dc}, {0x08,0x80a34001}, {0x0c,0x1a80001b}, {0x10,0x1b296956}, {0x14,0x031696a9}, {0x18,0xda00250c}, {0x1c,0x821061a5},
    {0x20,0x80a34001}, {0x24,0x02800006}, {0x28,0x03296956}, {0x2c,0x8210625a}, {0x30,0x80a34001}, {0x34,0x12800011}, {0x38,0x1b296956}, {0x3c,0x11000018},
    {0x40,0x92122340}, {0x44,0x40000187}, {0x48,0x90122200}, {0x4c,0x03000019}, {0x50,0x96106200}, {0x54,0xa8102001}, {0x58,0x98102000}, {0x5c,0x9b2b2002},
    {0x60,0x98032001}, {0x64,0xc20365b0}, {0x68,0x80a32017}, {0x6c,0x08bffffc}, {0x70,0xc223400b}, {0x74,0x1b296956}, {0x78,0x80a00013}, {0x7c,0x82380018},

    {0xf0,0x2f},
    {0x00,0xa413625a}, {0x04,0x90402000}, {0x08,0x8330601f}, {0x0c,0x1b1696a9}, {0x10,0xa21361a5}, {0x14,0x9e104008}, {0x18,0xd200254c}, {0x1c,0xe0002548},
    {0x20,0x98102500}, {0x24,0xc200250c}, {0x28,0x96033a50}, {0x2c,0x941b250c}, {0x30,0x80a04012}, {0x34,0x02800007}, {0x38,0x9a184011}, {0x3c,0x80a0000d},
    {0x40,0x82603fff}, {0x44,0x80904008}, {0x48,0x0280000a}, {0x4c,0x80a3e000}, {0x50,0x80a2e0e0}, {0x54,0x9a402000}, {0x58,0x80a0000a}, {0x5c,0x82603fff},
    {0x60,0x80934001}, {0x64,0x3280000a}, {0x68,0x98032004}, {0x6c,0x80a3e000}, {0x70,0x02800005}, {0x74,0x82033970}, {0x78,0x80a0603f}, {0x7c,0x28800004},

    {0xf0,0x30},
    {0x00,0x98032004}, {0x04,0xc0230000}, {0x08,0x98032004}, {0x0c,0x80a32867}, {0x10,0x28bfffe6}, {0x14,0xc200250c}, {0x18,0x80a62000}, {0x1c,0x06800006},
    {0x20,0x0300003f}, {0x24,0x821063ff}, {0x28,0x820a4001}, {0x2c,0x10800005}, {0x30,0xc220254c}, {0x34,0xc21022d4}, {0x38,0x83286010}, {0x3c,0xc2202690},
    {0x40,0xc20023c8}, {0x44,0xda002548}, {0x48,0x83306012}, {0x4c,0x82086200}, {0x50,0x9a0b7dff}, {0x54,0x9a134001}, {0x58,0xd800254c}, {0x5c,0xd4002334},
    {0x60,0x033fc200}, {0x64,0x9402a001}, {0x68,0x92050014}, {0x6c,0x980b3ffd}, {0x70,0x82106054}, {0x74,0xc2004000}, {0x78,0x98130009}, {0x7c,0x9732a001},

    {0xf0,0x31},
    {0x00,0x9602800b}, {0x04,0xd820254c}, {0x08,0x80a06114}, {0x0c,0xd620258c}, {0x10,0xda202548}, {0x14,0xd4202554}, {0x18,0xd4202588}, {0x1c,0xc2002344},
    {0x20,0x1280000a}, {0x24,0x99342018}, {0x28,0x820860ff}, {0x2c,0x8258605a}, {0x30,0x82006040}, {0x34,0x83306007}, {0x38,0x9b286010}, {0x3c,0x9a034001},
    {0x40,0x10800003}, {0x44,0xda202570}, {0x48,0xc2202570}, {0x4c,0x173fc200}, {0x50,0xda002570}, {0x54,0x8212e030}, {0x58,0xda204000}, {0x5c,0x80a3205a},
    {0x60,0x1280000a}, {0x64,0xda20257c}, {0x68,0x1b00003f}, {0x6c,0xc2002548}, {0x70,0x9a1363ff}, {0x74,0x8208400d}, {0x78,0x1b3fffc0}, {0x7c,0x9a0c000d},

    {0xf0,0x32},
    {0x00,0x8210400d}, {0x04,0xc2202548}, {0x08,0x80a4e001}, {0x0c,0x1280000c}, {0x10,0x03296956}, {0x14,0xc2002548}, {0x18,0x83306012}, {0x1c,0x80886001},
    {0x20,0x32800010}, {0x24,0x82102029}, {0x28,0x9a12e074}, {0x2c,0xc2002330}, {0x30,0xc2234000}, {0x34,0x1080000b}, {0x38,0x82102029}, {0x3c,0xda00250c},
    {0x40,0x8210625a}, {0x44,0x80a34001}, {0x48,0x1280000b}, {0x4c,0x01000000}, {0x50,0xc200254c}, {0x54,0x80886002}, {0x58,0x12800007}, {0x5c,0x82102015},
    {0x60,0xc2202500}, {0x64,0x7ffffe50}, {0x68,0x901025b0}, {0x6c,0x7ffffeb9}, {0x70,0x81e80000}, {0x74,0x01000000}, {0x78,0x81c7e008}, {0x7c,0x81e80000},

    {0xf0,0x33},
    {0x00,0x9de3bf98}, {0x04,0x4000023f}, {0x08,0x90102000}, {0x0c,0x133fc200}, {0x10,0xe80021fc}, {0x14,0xc2002298}, {0x18,0x98126070}, {0x1c,0xc2230000},
    {0x20,0x033fc000}, {0x24,0xd8002338}, {0x28,0x82106030}, {0x2c,0xd8204000}, {0x30,0xc200232c}, {0x34,0x90126074}, {0x38,0xc2220000}, {0x3c,0x1b3fc140},
    {0x40,0x0300003f}, {0x44,0xa013608c}, {0x48,0x9e136058}, {0x4c,0xa213605c}, {0x50,0xa4136080}, {0x54,0xa6136084}, {0x58,0x821063ff}, {0x5c,0x15000017},
    {0x60,0xc223c000}, {0x64,0x9412a380}, {0x68,0x17000018}, {0x6c,0x9a136088}, {0x70,0xd4234000}, {0x74,0x9612e0c0}, {0x78,0x03200040}, {0x7c,0xd6240000},

    {0xf0,0x34},
    {0x00,0x82106101}, {0x04,0xc2224000}, {0x08,0x033fc0c0}, {0x0c,0x82106004}, {0x10,0xc0204000}, {0x14,0xc0244000}, {0x18,0xd4248000}, {0x1c,0xd624c000},
    {0x20,0x80a52000}, {0x24,0xc2002374}, {0x28,0xd600247c}, {0x2c,0x02800006}, {0x30,0xd40022f8}, {0x34,0x82584014}, {0x38,0x82006800}, {0x3c,0x10800003},
    {0x40,0xa130600c}, {0x44,0xa0100001}, {0x48,0x1b3fc140}, {0x4c,0x98136040}, {0x50,0xd0030000}, {0x54,0x9732e001}, {0x58,0x825ac00a}, {0x5c,0x900a3800},
    {0x60,0x90120001}, {0x64,0xd0230000}, {0x68,0x9a136004}, {0x6c,0xd0034000}, {0x70,0x900a3fe0}, {0x74,0x9012000b}, {0x78,0x193fc200}, {0x7c,0xd0234000},

    {0xf0,0x35},
    {0x00,0x94132080}, {0x04,0xd0028000}, {0x08,0x92132054}, {0x0c,0x900a3fe0}, {0x10,0xc2024000}, {0x14,0x9012000b}, {0x18,0x960861ff}, {0x1c,0x80a2e114},
    {0x20,0x1280000d}, {0x24,0xd0228000}, {0x28,0x82102006}, {0x2c,0xc2228000}, {0x30,0x03009051}, {0x34,0x9a132058}, {0x38,0x82106040}, {0x3c,0xc2234000},
    {0x40,0x80a52000}, {0x44,0x12800004}, {0x48,0xd6224000}, {0x4c,0x825c2349}, {0x50,0xa130600a}, {0x54,0x233fc200}, {0x58,0xda0023f8}, {0x5c,0x82146008},
    {0x60,0xda204000}, {0x64,0x9814600c}, {0x68,0x7ffffe5a}, {0x6c,0xda230000}, {0x70,0x9814607c}, {0x74,0xd0030000}, {0x78,0x033ff000}, {0x7c,0x902a0001},

    {0xf0,0x36},
    {0x00,0xd0230000}, {0x04,0x033fc1c0}, {0x08,0xd8002340}, {0x0c,0x82106064}, {0x10,0xd8204000}, {0x14,0x1b3fc140}, {0x18,0x0300007f}, {0x1c,0x94136010},
    {0x20,0x821063ff}, {0x24,0x92136030}, {0x28,0x1707ffc0}, {0x2c,0xc2228000}, {0x30,0x9a136014}, {0x34,0xd6224000}, {0x38,0x80a42000}, {0x3c,0x0280000f},
    {0x40,0xd6234000}, {0x44,0x932c2010}, {0x48,0x82042001}, {0x4c,0x9b2c2004}, {0x50,0x83286010}, {0x54,0x9a02400d}, {0x58,0x98146084}, {0x5c,0xda230000},
    {0x60,0x82004010}, {0x64,0x96146088}, {0x68,0x82006002}, {0x6c,0x9414608c}, {0x70,0xd222c000}, {0x74,0xc2228000}, {0x78,0xc2002174}, {0x7c,0x80a06000},

    {0xf0,0x37},
    {0x00,0x02800007}, {0x04,0x173fc080}, {0x08,0xc2082174}, {0x0c,0xda082177}, {0x10,0x9812e008}, {0x14,0xc2230000}, {0x18,0xda22c000}, {0x1c,0x7ffffeb7},
    {0x20,0x90103fff}, {0x24,0x7ffffea4}, {0x28,0x90102001}, {0x2c,0x1b3fc0c0}, {0x30,0x82103fff}, {0x34,0x9a136004}, {0x38,0xc2234000}, {0x3c,0x03200040},
    {0x40,0xc2244000}, {0x44,0x81c7e008}, {0x48,0x81e80000}, {0x4c,0x9de3bf98}, {0x50,0xc2002508}, {0x54,0x808860ff}, {0x58,0x02800015}, {0x5c,0x1b3fc180},
    {0x60,0x82102001}, {0x64,0x9a13603c}, {0x68,0xc2234000}, {0x6c,0xc2002508}, {0x70,0x820860ff}, {0x74,0x80a04018}, {0x78,0x1280000b}, {0x7c,0x033fc180},

    {0xf0,0x38},
    {0x00,0x7ffffc6e}, {0x04,0x01000000}, {0x08,0xda002508}, {0x0c,0x033fc040}, {0x10,0x9a0b60ff}, {0x14,0x8210600c}, {0x18,0xc0204000}, {0x1c,0x10bffff7},
    {0x20,0x80a34018}, {0x24,0x8210603c}, {0x28,0xc0204000}, {0x2c,0x81c7e008}, {0x30,0x81e80000}, {0x34,0x9a102000}, {0x38,0x832b6002}, {0x3c,0x9a036001},
    {0x40,0x80a3604f}, {0x44,0x08bffffd}, {0x48,0xc0220001}, {0x4c,0x81c3e008}, {0x50,0x01000000}, {0x54,0xc20022f8}, {0x58,0xda00247c}, {0x5c,0x8258400d},
    {0x60,0x83306001}, {0x64,0x96102000}, {0x68,0x80a2c001}, {0x6c,0x1a80003b}, {0x70,0x0300003f}, {0x74,0x941063ff}, {0x78,0x98102000}, {0x7c,0x80a26001},

    {0xf0,0x39},
    {0x00,0x0280002e}, {0x04,0x80a26004}, {0x08,0x12800008}, {0x0c,0x80a26008}, {0x10,0xc2030008}, {0x14,0x9a08400a}, {0x18,0x83306012}, {0x1c,0x83286010},
    {0x20,0x10800014}, {0x24,0x9b336002}, {0x28,0x28800015}, {0x2c,0xda02000c}, {0x30,0xda030008}, {0x34,0x83336010}, {0x38,0x82004001}, {0x3c,0x9a0b400a},
    {0x40,0x81800000}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0x01000000}, {0x50,0x82704009}, {0x54,0x9a03400d}, {0x58,0x83286010}, {0x5c,0x81800000},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x01000000}, {0x6c,0x9a734009}, {0x70,0x8200400d}, {0x74,0x10800011}, {0x78,0xc2230008}, {0x7c,0x83336010},

    {0xf0,0x3a},
    {0x00,0x81800000}, {0x04,0x01000000}, {0x08,0x01000000}, {0x0c,0x01000000}, {0x10,0x82704009}, {0x14,0x9a0b400a}, {0x18,0x83286010}, {0x1c,0x81800000},
    {0x20,0x01000000}, {0x24,0x01000000}, {0x28,0x01000000}, {0x2c,0x9a734009}, {0x30,0x8200400d}, {0x34,0xc222000c}, {0x38,0xc20022f8}, {0x3c,0xda00247c},
    {0x40,0x8258400d}, {0x44,0x9602e001}, {0x48,0x83306001}, {0x4c,0x80a2c001}, {0x50,0x0abfffcb}, {0x54,0x98032004}, {0x58,0x81c3e008}, {0x5c,0x01000000},
    {0x60,0x98102000}, {0x64,0x9b2b2002}, {0x68,0x98032001}, {0x6c,0xc202000d}, {0x70,0x80a3204f}, {0x74,0x04bffffc}, {0x78,0xc222400d}, {0x7c,0x81c3e008},

    {0xf0,0x3b},
    {0x00,0x01000000}, {0x04,0xd6020000}, {0x08,0xd8024000}, {0x0c,0x9132e010}, {0x10,0x95332010}, {0x14,0x900a2fff}, {0x18,0x940aafff}, {0x1c,0x03000007},
    {0x20,0x9a22000a}, {0x24,0x821063ff}, {0x28,0x940b0001}, {0x2c,0x900ac001}, {0x30,0x9022000a}, {0x34,0x9a5b400d}, {0x38,0x905a0008}, {0x3c,0x81c3e008},
    {0x40,0x90034008}, {0x44,0x9de3bf88}, {0x48,0x82064019}, {0x4c,0x82004019}, {0x50,0x83286002}, {0x54,0x82004018}, {0x58,0x3b000019}, {0x5c,0xba176080},
    {0x60,0x83286002}, {0x64,0xf400401d}, {0x68,0x033fc000}, {0x6c,0x82106001}, {0x70,0xc227bfe8}, {0x74,0x03000040}, {0x78,0xc227bfec}, {0x7c,0x033fc040},

    {0xf0,0x3c},
    {0x00,0x821061fe}, {0x04,0xc227bff0}, {0x08,0x0300bf80}, {0x0c,0x82106101}, {0x10,0xc227bff4}, {0x14,0x9e10001d}, {0x18,0xb6102000}, {0x1c,0x9a07bff8},
    {0x20,0xba06c00d}, {0x24,0xc24f7ff0}, {0x28,0xb0060001}, {0x2c,0xc24f7ff8}, {0x30,0xb2064001}, {0x34,0x82067fff}, {0x38,0x82164001}, {0x3c,0x80a62000},
    {0x40,0x0480001e}, {0x44,0xbb30601f}, {0x48,0xc2002308}, {0x4c,0x80a04018}, {0x50,0x82402000}, {0x54,0x8090401d}, {0x58,0x32800019}, {0x5c,0xb606e001},
    {0x60,0xc2002300}, {0x64,0x80a64001}, {0x68,0x18800014}, {0x6c,0xba064019}, {0x70,0x82074019}, {0x74,0x83286002}, {0x78,0x82004018}, {0x7c,0x80a6e003},

    {0xf0,0x3d},
    {0x00,0x14800008}, {0x04,0xb9286002}, {0x08,0xc207000f}, {0x0c,0x80a0401a}, {0x10,0x2680000b}, {0x14,0xb606e001}, {0x18,0x1080000d}, {0x1c,0xb0102000},
    {0x20,0xc207000f}, {0x24,0x80a0401a}, {0x28,0x24800005}, {0x2c,0xb606e001}, {0x30,0x10800007}, {0x34,0xb0102000}, {0x38,0xb606e001}, {0x3c,0x80a6e007},
    {0x40,0x04bfffd9}, {0x44,0xba06c00d}, {0x48,0xb0102001}, {0x4c,0x81c7e008}, {0x50,0x81e80000}, {0x54,0xc2002548}, {0x58,0x82087fbf}, {0x5c,0xc2202548},
    {0x60,0xc020255c}, {0x64,0xc0202514}, {0x68,0x9a102000}, {0x6c,0x832b6002}, {0x70,0x9a036001}, {0x74,0xc0206738}, {0x78,0x80a36009}, {0x7c,0x04bffffc},

    {0xf0,0x3e},
    {0x00,0xc0206710}, {0x04,0x81c3e008}, {0x08,0x01000000}, {0x0c,0x9de3bf88}, {0x10,0x82063fff}, {0x14,0xb8067fff}, {0x18,0x82160001}, {0x1c,0xba16401c},
    {0x20,0x80974001}, {0x24,0xa610001a}, {0x28,0x0680007d}, {0x2c,0xa410001b}, {0x30,0xc2002308}, {0x34,0x80a60001}, {0x38,0x3880007a}, {0x3c,0xb0102000},
    {0x40,0xc2002300}, {0x44,0x80a64001}, {0x48,0x38800076}, {0x4c,0xb0102000}, {0x50,0x94062001}, {0x54,0x96066001}, {0x58,0xa207bff8}, {0x5c,0x8207bfe8},
    {0x60,0xb0102003}, {0x64,0xc0204000}, {0x68,0xb0863fff}, {0x6c,0x1cbffffe}, {0x70,0x82006004}, {0x74,0xc2002160}, {0x78,0x80886004}, {0x7c,0x0280000f},

    {0xf0,0x3f},
    {0x00,0xb0102004}, {0x04,0x80a66001}, {0x08,0x0280000b}, {0x0c,0x821021ff}, {0x10,0xc2002308}, {0x14,0x80a60001}, {0x18,0x02800007}, {0x1c,0x821021ff},
    {0x20,0xc2002300}, {0x24,0x80a64001}, {0x28,0x12800005}, {0x2c,0x80a7000b}, {0x30,0x821021ff}, {0x34,0xc227bff4}, {0x38,0x80a7000b}, {0x3c,0x14800031},
    {0x40,0xb210001c}, {0x44,0x8207001c}, {0x48,0xba02c00b}, {0x4c,0x8200401c}, {0x50,0xba07400b}, {0x54,0xba20401d}, {0x58,0x39000019}, {0x5c,0xa0172080},
    {0x60,0x9827400a}, {0x64,0x9b286002}, {0x68,0xb002bffe}, {0x6c,0x82034018}, {0x70,0x83286002}, {0x74,0xba06000c}, {0x78,0xb4004010}, {0x7c,0xb6076008},

    {0xf0,0x40},
    {0x00,0x90102001}, {0x04,0x932a001b}, {0x08,0x9e102000}, {0x0c,0xb92be002}, {0x10,0xfa072520}, {0x14,0xba5f4012}, {0x18,0x833f601f}, {0x1c,0x83306018},
    {0x20,0xba074001}, {0x24,0xc2068000}, {0x28,0x82204013}, {0x2c,0xbb3f6008}, {0x30,0x9e03e001}, {0x34,0x80a0401d}, {0x38,0x04800005}, {0x3c,0xb8070011},
    {0x40,0xc2073ff0}, {0x44,0x82104009}, {0x48,0xc2273ff0}, {0x4c,0x80a3e003}, {0x50,0x08bffff0}, {0x54,0xb92be002}, {0x58,0xb0062001}, {0x5c,0xb606e001},
    {0x60,0x80a6000a}, {0x64,0x04bfffe8}, {0x68,0xb406a004}, {0x6c,0xb2066001}, {0x70,0x9a03600c}, {0x74,0x80a6400b}, {0x78,0x04bfffdc}, {0x7c,0x98032003},

    {0xf0,0x41},
    {0x00,0xc207bfe8}, {0x04,0x80886010}, {0x08,0x0280000a}, {0x0c,0xfa07bfec}, {0x10,0xc207bff4}, {0x14,0x80886082}, {0x18,0x02800007}, {0x1c,0x808f6010},
    {0x20,0x80886028}, {0x24,0x1280001f}, {0x28,0xb0102003}, {0x2c,0xfa07bfec}, {0x30,0x808f6010}, {0x34,0x02800012}, {0x38,0xc207bff0}, {0x3c,0x808f6082},
    {0x40,0x02800007}, {0x44,0x808f6028}, {0x48,0xc207bff4}, {0x4c,0x80886028}, {0x50,0x32800014}, {0x54,0xb0102002}, {0x58,0x808f6028}, {0x5c,0x02800008},
    {0x60,0xc207bff0}, {0x64,0xc207bff4}, {0x68,0x80886082}, {0x6c,0x02800004}, {0x70,0xc207bff0}, {0x74,0x1080000b}, {0x78,0xb0102002}, {0x7c,0x80886010},

    {0xf0,0x42},
    {0x00,0x02800008}, {0x04,0xb0102000}, {0x08,0x80886082}, {0x0c,0x02800005}, {0x10,0x80886028}, {0x14,0x12800003}, {0x18,0xb0102001}, {0x1c,0xb0102000},
    {0x20,0x81c7e008}, {0x24,0x81e80000}, {0x28,0x94102000}, {0x2c,0xc2002514}, {0x30,0x80a28001}, {0x34,0x96102000}, {0x38,0x1a80000e}, {0x3c,0x9b2ae002},
    {0x40,0xc2036710}, {0x44,0x9602e001}, {0x48,0x80a06000}, {0x4c,0x02800006}, {0x50,0x992aa002}, {0x54,0xc2232710}, {0x58,0xc2036738}, {0x5c,0x9402a001},
    {0x60,0xc2232738}, {0x64,0xc2002514}, {0x68,0x10bffff4}, {0x6c,0x80a2c001}, {0x70,0x81c3e008}, {0x74,0xd4202514}, {0x78,0xd4020000}, {0x7c,0x03000018},

    {0xf0,0x43},
    {0x00,0x9802800a}, {0x04,0x82106200}, {0x08,0xda530001}, {0x0c,0xc2002590}, {0x10,0xc250400c}, {0x14,0x96a0400d}, {0x18,0x02800016}, {0x1c,0x03000018},
    {0x20,0x80a2e000}, {0x24,0x04800009}, {0x28,0x82102001}, {0x2c,0xda022004}, {0x30,0x8328400d}, {0x34,0x80a2c001}, {0x38,0x3480000d}, {0x3c,0x973ac00d},
    {0x40,0x1080000b}, {0x44,0x96102001}, {0x48,0x1680000a}, {0x4c,0x03000018}, {0x50,0xda022008}, {0x54,0x82103fff}, {0x58,0x8328400d}, {0x5c,0x80a2c001},
    {0x60,0x36800003}, {0x64,0x96103fff}, {0x68,0x973ac00d}, {0x6c,0x03000018}, {0x70,0x92106200}, {0x74,0x8202800a}, {0x78,0xc2504009}, {0x7c,0x9602c001},

    {0xf0,0x44},
    {0x00,0x808aa001}, {0x04,0x0280000f}, {0x08,0x9b3aa01f}, {0x0c,0x9b33601f}, {0x10,0x9a02800d}, {0x14,0x9b3b6001}, {0x18,0x9b2b6002}, {0x1c,0xd8034009},
    {0x20,0x033fffc0}, {0x24,0x980b0001}, {0x28,0x0300003f}, {0x2c,0x821063ff}, {0x30,0x820ac001}, {0x34,0x98030001}, {0x38,0x1080000d}, {0x3c,0xd8234009},
    {0x40,0x9b33601f}, {0x44,0x9a02800d}, {0x48,0x9b3b6001}, {0x4c,0x9b2b6002}, {0x50,0x0300003f}, {0x54,0xd8034009}, {0x58,0x821063ff}, {0x5c,0x980b0001},
    {0x60,0x832ae010}, {0x64,0x8200400c}, {0x68,0xc2234009}, {0x6c,0xc2020000}, {0x70,0xda00247c}, {0x74,0x8200400d}, {0x78,0x81c3e008}, {0x7c,0xc2220000},

    {0xf0,0x45},
    {0x00,0x9de3bf98}, {0x04,0x833e201f}, {0x08,0xd0002320}, {0x0c,0x82204018}, {0x10,0x80a22000}, {0x14,0x02800015}, {0x18,0x9b30601f}, {0x1c,0x033fc000},
    {0x20,0xa0106020}, {0x24,0xc200231c}, {0x28,0x80a00001}, {0x2c,0x82402000}, {0x30,0x8088400d}, {0x34,0xc2002318}, {0x38,0x02800009}, {0x3c,0xb01e0001},
    {0x40,0x80a00001}, {0x44,0x82603fff}, {0x48,0x7ffffab3}, {0x4c,0xc2240000}, {0x50,0xc2002318}, {0x54,0x10800005}, {0x58,0xc2240000}, {0x5c,0x033fc000},
    {0x60,0x82106020}, {0x64,0xf0204000}, {0x68,0x81c7e008}, {0x6c,0x81e80000}, {0x70,0x9de3bf98}, {0x74,0x7ffffab1}, {0x78,0x01000000}, {0x7c,0xe0002500},

    {0xf0,0x46},
    {0x00,0x80a42028}, {0x04,0x08800013}, {0x08,0x80a42000}, {0x0c,0xc0202584}, {0x10,0xa2102000}, {0x14,0x832c6002}, {0x18,0xc2006f04}, {0x1c,0x80a06000},
    {0x20,0x02800053}, {0x24,0xa2046001}, {0x28,0x9fc04000}, {0x2c,0x01000000}, {0x30,0xc2002584}, {0x34,0x80a06000}, {0x38,0x1280004d}, {0x3c,0x80a4603b},
    {0x40,0x24bffff6}, {0x44,0x832c6002}, {0x48,0x1080004a}, {0x4c,0xc2002500}, {0x50,0x1280000b}, {0x54,0x80a42014}, {0x58,0xc2002fcc}, {0x5c,0x9fc04000},
    {0x60,0x01000000}, {0x64,0x7ffffcc5}, {0x68,0x90102000}, {0x6c,0x7ffffcb2}, {0x70,0x90102001}, {0x74,0x1080003f}, {0x78,0xc2002500}, {0x7c,0x1880000c},

    {0xf0,0x47},
    {0x00,0x80a42015}, {0x04,0x808c2001}, {0x08,0x3280003a}, {0x0c,0xc2002500}, {0x10,0x90043ffe}, {0x14,0x7ffffc4b}, {0x18,0x91322001}, {0x1c,0x7ffffc2d},
    {0x20,0x01000000}, {0x24,0x10800033}, {0x28,0xc2002500}, {0x2c,0x18800010}, {0x30,0x80a42018}, {0x34,0x033fc180}, {0x38,0xda0025b0}, {0x3c,0x82106038},
    {0x40,0xda204000}, {0x44,0x033fc200}, {0x48,0x82106074}, {0x4c,0xda00232c}, {0x50,0xda204000}, {0x54,0x7ffffc98}, {0x58,0x90102000}, {0x5c,0xc200265c},
    {0x60,0xc2202538}, {0x64,0x10800020}, {0x68,0xc2002fcc}, {0x6c,0x1880000c}, {0x70,0x80a42028}, {0x74,0x90102000}, {0x78,0x92102000}, {0x7c,0xc2002fc0},

    {0xf0,0x48},
    {0x00,0x9fc04000}, {0x04,0x94102000}, {0x08,0x11000018}, {0x0c,0x7ffffe0a}, {0x10,0x90122200}, {0x14,0x10800017}, {0x18,0xc2002500}, {0x1c,0x38800015},
    {0x20,0xc2002500}, {0x24,0x7ffffb97}, {0x28,0x23000018}, {0x2c,0x92146200}, {0x30,0xc2002fc0}, {0x34,0x9fc04000}, {0x38,0x94043fe8}, {0x3c,0x80a42028},
    {0x40,0x3280000c}, {0x44,0xc2002500}, {0x48,0x90146200}, {0x4c,0x7ffffe02}, {0x50,0x92102008}, {0x54,0xc2002fd8}, {0x58,0x80a06000}, {0x5c,0x22800005},
    {0x60,0xc2002500}, {0x64,0x9fc04000}, {0x68,0x01000000}, {0x6c,0xc2002500}, {0x70,0x80a40001}, {0x74,0x1280000b}, {0x78,0x031fffff}, {0x7c,0x821063f0},

    {0xf0,0x49},
    {0x00,0x80a40001}, {0x04,0x38800003}, {0x08,0x21040000}, {0x0c,0xa0042001}, {0x10,0x033fc180}, {0x14,0x82106034}, {0x18,0xe0204000}, {0x1c,0xe0202500},
    {0x20,0x81c7e008}, {0x24,0x81e80000}, {0x28,0x81c3e008}, {0x2c,0x01000000}, {0x30,0x9de3bf98}, {0x34,0x9e100018}, {0x38,0x80a66000}, {0x3c,0x0280001b},
    {0x40,0xb010001a}, {0x44,0x031fffdf}, {0x48,0xb41063ff}, {0x4c,0x82102000}, {0x50,0xbb286002}, {0x54,0x80a62009}, {0x58,0xb6006001}, {0x5c,0x12800006},
    {0x60,0xb810001d}, {0x64,0xc206401d}, {0x68,0x83306001}, {0x6c,0x8208401a}, {0x70,0xc226401d}, {0x74,0x80a62008}, {0x78,0x08800006}, {0x7c,0xc206401c},

    {0xf0,0x4a},
    {0x00,0xfa03c01c}, {0x04,0xbb376001}, {0x08,0x10800003}, {0x0c,0xba0f401a}, {0x10,0xfa03c01c}, {0x14,0x8200401d}, {0x18,0xc226401c}, {0x1c,0x80a6e04f},
    {0x20,0x08bfffec}, {0x24,0x8210001b}, {0x28,0x81c7e008}, {0x2c,0x81e80000}, {0x30,0x03169696}, {0x34,0xda002180}, {0x38,0x8210625a}, {0x3c,0x80a34001},
    {0x40,0x94102000}, {0x44,0x12800006}, {0x48,0x96102000}, {0x4c,0x033fc180}, {0x50,0x82106030}, {0x54,0x10800024}, {0x58,0xda204000}, {0x5c,0xc202c000},
    {0x60,0x9602e004}, {0x64,0x80a2e4ff}, {0x68,0x08bffffd}, {0x6c,0x94028001}, {0x70,0x96102d00}, {0x74,0xd2002ff8}, {0x78,0x03000019}, {0x7c,0x80a2c009},

    {0xf0,0x4b},
    {0x00,0x1a80000b}, {0x04,0x901063ff}, {0x08,0xd802c000}, {0x0c,0x9602e004}, {0x10,0x80a2c009}, {0x14,0x9a402000}, {0x18,0x80a2000b}, {0x1c,0x82603fff},
    {0x20,0x808b4001}, {0x24,0x12bffff9}, {0x28,0x9402800c}, {0x2c,0xc20021fc}, {0x30,0x94228001}, {0x34,0xc2002200}, {0x38,0x94228001}, {0x3c,0x03169696},
    {0x40,0x8210625a}, {0x44,0x80a28001}, {0x48,0x033fc180}, {0x4c,0x82106030}, {0x50,0x02800005}, {0x54,0xd4204000}, {0x58,0x03000008}, {0x5c,0x81c06280},
    {0x60,0x90102001}, {0x64,0x01000000}, {0x68,0x81c3e008}, {0x6c,0x01000000}, {0x70,0x9de3bf98}, {0x74,0x9e100018}, {0x78,0x03000019}, {0x7c,0xb0100019},

    {0xf0,0x4c},
    {0x00,0xba106080}, {0x04,0xb6102000}, {0x08,0x832ee002}, {0x0c,0xb606e001}, {0x10,0x80a6e0d7}, {0x14,0x08bffffd}, {0x18,0xc020401d}, {0x1c,0xb6102000},
    {0x20,0xc20022fc}, {0x24,0x80a6c001}, {0x28,0x1a80001c}, {0x2c,0x03000019}, {0x30,0xb21060b4}, {0x34,0xb4102000}, {0x38,0xc20022f8}, {0x3c,0x80a68001},
    {0x40,0x1a800011}, {0x44,0x832ee002}, {0x48,0xb8004019}, {0x4c,0xc200247c}, {0x50,0xfa0ee380}, {0x54,0x825e8001}, {0x58,0x8200401d}, {0x5c,0x82004001},
    {0x60,0xfa160001}, {0x64,0xc213c001}, {0x68,0x8220401d}, {0x6c,0xc2270000}, {0x70,0xb406a001}, {0x74,0xc20022f8}, {0x78,0x80a68001}, {0x7c,0x0abffff4},

    {0xf0,0x4d},
    {0x00,0xb8072030}, {0x04,0xb606e001}, {0x08,0xc20022fc}, {0x0c,0x80a6c001}, {0x10,0x0abfffea}, {0x14,0xb4102000}, {0x18,0x81c7e008}, {0x1c,0x81e80000},
    {0x20,0x9de3bf98}, {0x24,0x21000018}, {0x28,0xc2002fe4}, {0x2c,0x9fc04000}, {0x30,0x90142200}, {0x34,0xc200259c}, {0x38,0xd800254c}, {0x3c,0x808b2002},
    {0x40,0x02800033}, {0x44,0xc2202550}, {0x48,0xda002654}, {0x4c,0x8203400d}, {0x50,0x8200400d}, {0x54,0xda00259c}, {0x58,0xb2036140}, {0x5c,0x82004001},
    {0x60,0x9a102061}, {0x64,0x9a234001}, {0x68,0x832e6007}, {0x6c,0x9738601f}, {0x70,0x8182e000}, {0x74,0x01000000}, {0x78,0x01000000}, {0x7c,0x01000000},

    {0xf0,0x4e},
    {0x00,0xb278400d}, {0x04,0x980b3ffd}, {0x08,0xd820254c}, {0x0c,0xc2002fe4}, {0x10,0x9fc04000}, {0x14,0x90142340}, {0x18,0x11000017}, {0x1c,0xe000259c},
    {0x20,0xc2002fe4}, {0x24,0x9fc04000}, {0x28,0x90122240}, {0x2c,0x80a64010}, {0x30,0x9a603fff}, {0x34,0xc200259c}, {0x38,0x80a64001}, {0x3c,0x82603fff},
    {0x40,0x80934001}, {0x44,0x22800012}, {0x48,0xc020250c}, {0x4c,0x21000019}, {0x50,0x7ffffad5}, {0x54,0x90142200}, {0x58,0xa0142200}, {0x5c,0xb2102000},
    {0x60,0x9b2e6002}, {0x64,0xb2066001}, {0x68,0xc2034010}, {0x6c,0x80a66017}, {0x70,0x04bffffc}, {0x74,0xc22365b0}, {0x78,0x7ffffb36}, {0x7c,0x01000000},

    {0xf0,0x4f},
    {0x00,0x82102015}, {0x04,0x10800010}, {0x08,0xc2202500}, {0x0c,0xc200250c}, {0x10,0x80a06000}, {0x14,0x1280000c}, {0x18,0x031696a9}, {0x1c,0x821061a5},
    {0x20,0x31000018}, {0x24,0x33000017}, {0x28,0xc220250c}, {0x2c,0xc0202658}, {0x30,0xc0202654}, {0x34,0xb0162200}, {0x38,0xb2166240}, {0x3c,0x7ffffd69},
    {0x40,0x81e80000}, {0x44,0x01000000}, {0x48,0x81c7e008}, {0x4c,0x81e80000}, {0x50,0x9de3bf58}, {0x54,0x94100018}, {0x58,0x9a102000}, {0x5c,0x96102000},
    {0x60,0x98102000}, {0x64,0x9e102000}, {0x68,0x8203000f}, {0x6c,0xf6086441}, {0x70,0x80a6e000}, {0x74,0x02800024}, {0x78,0xf4086440}, {0x7c,0xc208217d},

    {0xf0,0x50},
    {0x00,0x80807fff}, {0x04,0xf40ea37f}, {0x08,0xf60ee37f}, {0x0c,0x0c80001a}, {0x10,0xb2102000}, {0x14,0xb007bff8}, {0x18,0xc200247c}, {0x1c,0x82584019},
    {0x20,0xba06c001}, {0x24,0x82068001}, {0x28,0x82004001}, {0x2c,0xf8528001}, {0x30,0xba07401d}, {0x34,0xc252801d}, {0x38,0xb8270001}, {0x3c,0x80a66000},
    {0x40,0x02800007}, {0x44,0xf8263fc0}, {0x48,0xc2063fbc}, {0x4c,0x82a70001}, {0x50,0x2c800003}, {0x54,0x9a234001}, {0x58,0x9a034001}, {0x5c,0xc208217d},
    {0x60,0xb2066001}, {0x64,0x82007fff}, {0x68,0x80a64001}, {0x6c,0x04bfffeb}, {0x70,0xb0062004}, {0x74,0x9e03e001}, {0x78,0x80a3e00a}, {0x7c,0x04bfffdc},

    {0xf0,0x51},
    {0x00,0x8203000f}, {0x04,0x9602e001}, {0x08,0x80a2e001}, {0x0c,0x04bfffd6}, {0x10,0x9803200c}, {0x14,0xda20259c}, {0x18,0x81c7e008}, {0x1c,0x81e80000},
    {0x20,0x9de3bf98}, {0x24,0xc2002540}, {0x28,0x82006001}, {0x2c,0xc2202540}, {0x30,0xc2002588}, {0x34,0x80a06000}, {0x38,0x02800014}, {0x3c,0x11000018},
    {0x40,0xc2002594}, {0x44,0x80a06000}, {0x48,0x12800004}, {0x4c,0x90122340}, {0x50,0x7ffffcd9}, {0x54,0x01000000}, {0x58,0xda002588}, {0x5c,0xc2002594},
    {0x60,0x82006001}, {0x64,0x9a037fff}, {0x68,0xc2202594}, {0x6c,0x7ffffa65}, {0x70,0xda202588}, {0x74,0x13000018}, {0x78,0x92126340}, {0x7c,0xc2002fc0},

    {0xf0,0x52},
    {0x00,0x9fc04000}, {0x04,0xd4002594}, {0x08,0x01000000}, {0x0c,0x81c7e008}, {0x10,0x81e80000}, {0x14,0x9de3bf98}, {0x18,0xc2002588}, {0x1c,0x80a06000},
    {0x20,0x02800010}, {0x24,0x11000018}, {0x28,0xd8002548}, {0x2c,0x83332010}, {0x30,0x80886001}, {0x34,0x22800010}, {0x38,0xc200258c}, {0x3c,0xc2002558},
    {0x40,0x80a06000}, {0x44,0x3280000c}, {0x48,0xc200258c}, {0x4c,0xc2002594}, {0x50,0x80a06001}, {0x54,0x32800008}, {0x58,0xc200258c}, {0x5c,0x11000018},
    {0x60,0x90122340}, {0x64,0xd0202590}, {0x68,0xc0202588}, {0x6c,0x1080001a}, {0x70,0xd2002594}, {0x74,0x80a06000}, {0x78,0x12800015}, {0x7c,0x82102001},

    {0xf0,0x53},
    {0x00,0xda002554}, {0x04,0xc2002598}, {0x08,0x80a0400d}, {0x0c,0x1a800007}, {0x10,0x03000017}, {0x14,0x82102001}, {0x18,0xda20258c}, {0x1c,0xc2202584},
    {0x20,0x10800028}, {0x24,0xc0202598}, {0x28,0x808b2400}, {0x2c,0x12800004}, {0x30,0x82106240}, {0x34,0x03000017}, {0x38,0x82106100}, {0x3c,0xc2202590},
    {0x40,0xd2002598}, {0x44,0x10800004}, {0x48,0xd0002590}, {0x4c,0x1080001d}, {0x50,0xc2202584}, {0x54,0x7ffffca0}, {0x58,0x01000000}, {0x5c,0x13000018},
    {0x60,0xc2002fd0}, {0x64,0x92126200}, {0x68,0x9fc04000}, {0x6c,0xd0002590}, {0x70,0xc2002690}, {0x74,0x82087dff}, {0x78,0xc2202690}, {0x7c,0x82102080},

    {0xf0,0x54},
    {0x00,0xc220256c}, {0x04,0xc2002200}, {0x08,0x80a06000}, {0x0c,0x32800008}, {0x10,0xc2002548}, {0x14,0xda002548}, {0x18,0x83336010}, {0x1c,0x80886001},
    {0x20,0x22800006}, {0x24,0x03000004}, {0x28,0xc2002548}, {0x2c,0x1b000004}, {0x30,0x10800003}, {0x34,0x8210400d}, {0x38,0x822b4001}, {0x3c,0xc2202548},
    {0x40,0x81c7e008}, {0x44,0x81e80000}, {0x48,0xda002508}, {0x4c,0x033fc000}, {0x50,0x960b4001}, {0x54,0x808b7f00}, {0x58,0x0280001a}, {0x5c,0x901020a5},
    {0x60,0x1500003f}, {0x64,0x8212a300}, {0x68,0x980b4001}, {0x6c,0x03294000}, {0x70,0x80a2c001}, {0x74,0x02800013}, {0x78,0x01000000}, {0x7c,0xc200254c},

    {0xf0,0x55},
    {0x00,0x8210400c}, {0x04,0xc220254c}, {0x08,0xc2002500}, {0x0c,0x80a06028}, {0x10,0xc0202508}, {0x14,0x0880000b}, {0x18,0x033fffc0}, {0x1c,0x9a0b4001},
    {0x20,0x03168000}, {0x24,0x80a2c001}, {0x28,0x12800006}, {0x2c,0x9812a3ff}, {0x30,0xc2002548}, {0x34,0x8208400c}, {0x38,0x8210400d}, {0x3c,0xc2202548},
    {0x40,0x03000006}, {0x44,0x81c063cc}, {0x48,0x01000000}, {0x4c,0x01000000}, {0x50,0xda00247c}, {0x54,0xc20022f8}, {0x58,0x8258400d}, {0x5c,0x83306001},
    {0x60,0x9a102000}, {0x64,0x80a34001}, {0x68,0x1a800015}, {0x6c,0x031fffdf}, {0x70,0x961063ff}, {0x74,0x98036001}, {0x78,0x80a26008}, {0x7c,0x04800006},

    {0xf0,0x56},
    {0x00,0x9b2b6002}, {0x04,0xc202000d}, {0x08,0x83306001}, {0x0c,0x10800003}, {0x10,0x8208400b}, {0x14,0xc202000d}, {0x18,0x82584009}, {0x1c,0xc222000d},
    {0x20,0xda00247c}, {0x24,0xc20022f8}, {0x28,0x8258400d}, {0x2c,0x83306001}, {0x30,0x80a30001}, {0x34,0x0abffff0}, {0x38,0x9a10000c}, {0x3c,0x81c3e008},
    {0x40,0x01000000}, {0x44,0x9de3bf98}, {0x48,0xc2102548}, {0x4c,0x80886001}, {0x50,0x12800068}, {0x54,0x01000000}, {0x58,0xe6002460}, {0x5c,0x80a4e000},
    {0x60,0x02800064}, {0x64,0x01000000}, {0x68,0xc2002588}, {0x6c,0xda002594}, {0x70,0x9800400d}, {0x74,0xc2002554}, {0x78,0x80a30001}, {0x7c,0x1280005d},

    {0xf0,0x57},
    {0x00,0x01000000}, {0x04,0xda002598}, {0x08,0xc200258c}, {0x0c,0x8200400d}, {0x10,0x80a0400c}, {0x14,0x02800004}, {0x18,0x80a36000}, {0x1c,0x12800055},
    {0x20,0x01000000}, {0x24,0xe208217e}, {0x28,0x808c60ff}, {0x2c,0x9e102000}, {0x30,0x0280002e}, {0x34,0x96102001}, {0x38,0xe408217f}, {0x3c,0x13000019},
    {0x40,0x94102001}, {0x44,0x80a28012}, {0x48,0x14800020}, {0x4c,0xa0102000}, {0x50,0x832ae002}, {0x54,0x98006030}, {0x58,0xac126080}, {0x5c,0xaa126050},
    {0x60,0xa81260b0}, {0x64,0xd008217f}, {0x68,0xda030016}, {0x6c,0x80a34013}, {0x70,0x26800013}, {0x74,0x9402a001}, {0x78,0x80a2a001}, {0x7c,0x22800007},

    {0xf0,0x58},
    {0x00,0xc208217f}, {0x04,0xc2030015}, {0x08,0x80a34001}, {0x0c,0x2480000c}, {0x10,0x9402a001}, {0x14,0xc208217f}, {0x18,0x80a28001}, {0x1c,0x22800007},
    {0x20,0xa0042001}, {0x24,0xc2030014}, {0x28,0x80a34001}, {0x2c,0x26800004}, {0x30,0x9402a001}, {0x34,0xa0042001}, {0x38,0x9402a001}, {0x3c,0x80a28008},
    {0x40,0x04bfffea}, {0x44,0x98032030}, {0x48,0x80a4000f}, {0x4c,0x34800002}, {0x50,0x9e100010}, {0x54,0x9602e001}, {0x58,0x820c60ff}, {0x5c,0x80a2c001},
    {0x60,0x24bfffd9}, {0x64,0x94102001}, {0x68,0x96102000}, {0x6c,0xc20ae464}, {0x70,0x80a06000}, {0x74,0x22800006}, {0x78,0x9602e001}, {0x7c,0x80a3c001},

    {0xf0,0x59},
    {0x00,0x34800007}, {0x04,0xc20ae278}, {0x08,0x9602e001}, {0x0c,0x80a2e003}, {0x10,0x24bffff8}, {0x14,0xc20ae464}, {0x18,0x30800016}, {0x1c,0xda00256c},
    {0x20,0x8258400d}, {0x24,0x83306007}, {0x28,0xc220256c}, {0x2c,0xe00ae468}, {0x30,0x80a42000}, {0x34,0x0280000f}, {0x38,0x82102001}, {0x3c,0xc2202584},
    {0x40,0xd0002590}, {0x44,0x7fffff83}, {0x48,0xd2002554}, {0x4c,0xc2002588}, {0x50,0x82004010}, {0x54,0xc2202588}, {0x58,0xda00258c}, {0x5c,0xc2002554},
    {0x60,0x80a34001}, {0x64,0x18800003}, {0x68,0x82034010}, {0x6c,0xc220258c}, {0x70,0x81c7e008}, {0x74,0x81e80000}, {0x78,0x9de3bf58}, {0x7c,0xc2002548},

    {0xf0,0x5a},
    {0x00,0x8330600c}, {0x04,0x80886001}, {0x08,0x1280006d}, {0x0c,0x01000000}, {0x10,0xfa002500}, {0x14,0xc20021f4}, {0x18,0x80a74001}, {0x1c,0x18800068},
    {0x20,0x01000000}, {0x24,0xc200254c}, {0x28,0x8330600b}, {0x2c,0x82086001}, {0x30,0x80a00001}, {0x34,0x9a603fff}, {0x38,0x9403400d}, {0x3c,0xc252a210},
    {0x40,0x80a06000}, {0x44,0x0280005e}, {0x48,0x8207bfb8}, {0x4c,0xb810200f}, {0x50,0xc0204000}, {0x54,0xb8873fff}, {0x58,0x1cbffffe}, {0x5c,0x82006004},
    {0x60,0xb0102001}, {0x64,0xc2002300}, {0x68,0x80a60001}, {0x6c,0x1880003b}, {0x70,0x03000019}, {0x74,0x82106080}, {0x78,0x96006030}, {0x7c,0xb4102001},

    {0xf0,0x5b},
    {0x00,0xc2002308}, {0x04,0x80a68001}, {0x08,0x38800030}, {0x0c,0xb0062001}, {0x10,0xb202e004}, {0x14,0xfa52a210}, {0x18,0xc2064000}, {0x1c,0x80a0401d},
    {0x20,0x36800025}, {0x24,0xb406a001}, {0x28,0x832ea018}, {0x2c,0xbb2e2010}, {0x30,0x8200401d}, {0x34,0x9e006001}, {0x38,0xb8102000}, {0x3c,0x98102001},
    {0x40,0xb607bfb8}, {0x44,0xbb2f2002}, {0x48,0xc20f6828}, {0x4c,0x80a68001}, {0x50,0x1280000f}, {0x54,0xb8072001}, {0x58,0xc20f6829}, {0x5c,0x80a60001},
    {0x60,0x3280000c}, {0x64,0xc2176828}, {0x68,0xd826c000}, {0x6c,0xc217682a}, {0x70,0x80a061fe}, {0x74,0x38800010}, {0x78,0xb406a001}, {0x7c,0xc2076828},

    {0xf0,0x5c},
    {0x00,0x82006001}, {0x04,0x1080000b}, {0x08,0xc2276828}, {0x0c,0xc2176828}, {0x10,0x80a06000}, {0x14,0x12800005}, {0x18,0x80a7200f}, {0x1c,0xd826c000},
    {0x20,0x10800004}, {0x24,0xde276828}, {0x28,0x08bfffe7}, {0x2c,0xb606e004}, {0x30,0xb406a001}, {0x34,0xc2002308}, {0x38,0x80a68001}, {0x3c,0x08bfffd6},
    {0x40,0xb2066004}, {0x44,0xb0062001}, {0x48,0xc2002300}, {0x4c,0x80a60001}, {0x50,0x08bfffcb}, {0x54,0x9602e030}, {0x58,0xb4102000}, {0x5c,0xb8102000},
    {0x60,0xb607bff8}, {0x64,0xbb2f2002}, {0x68,0x8207401b}, {0x6c,0xc2007fc0}, {0x70,0x80a06000}, {0x74,0x32800004}, {0x78,0xfa17682a}, {0x7c,0x10800005},

    {0xf0,0x5d},
    {0x00,0xc0276828}, {0x04,0xc20b63cc}, {0x08,0x80a74001}, {0x0c,0xb466bfff}, {0x10,0xb8072001}, {0x14,0x80a7200f}, {0x18,0x08bffff4}, {0x1c,0xbb2f2002},
    {0x20,0xc20b63ce}, {0x24,0x80a68001}, {0x28,0x08800005}, {0x2c,0x3b000200}, {0x30,0xc200254c}, {0x34,0x8210401d}, {0x38,0xc220254c}, {0x3c,0x81c7e008},
    {0x40,0x81e80000}, {0x44,0x9de3bf70}, {0x48,0x1b00003f}, {0x4c,0xc2002350}, {0x50,0x9a1363ff}, {0x54,0xae08400d}, {0x58,0xa6102001}, {0x5c,0xda002300},
    {0x60,0x80a4c00d}, {0x64,0x18800062}, {0x68,0xa3306010}, {0x6c,0xa410200c}, {0x70,0xac102000}, {0x74,0xaa102000}, {0x78,0xa8102000}, {0x7c,0x80a46000},

    {0xf0,0x5e},
    {0x00,0x0280002c}, {0x04,0xa0102000}, {0x08,0x03000019}, {0x0c,0x96106080}, {0x10,0x92102000}, {0x14,0x9807bfd0}, {0x18,0x82050009}, {0x1c,0xda086440},
    {0x20,0x8204800d}, {0x24,0x80a36000}, {0x28,0x02800008}, {0x2c,0x83286002}, {0x30,0xc200400b}, {0x34,0xc2230000}, {0x38,0x92026001}, {0x3c,0x80a2600b},
    {0x40,0x04bffff6}, {0x44,0x98032004}, {0x48,0x7ffff7a3}, {0x4c,0x9007bfd0}, {0x50,0x80a5c011}, {0x54,0x1480000b}, {0x58,0x92100017}, {0x5c,0x832de002},
    {0x60,0x8200401e}, {0x64,0x9a007fd0}, {0x68,0xc2034000}, {0x6c,0x92026001}, {0x70,0xa0040001}, {0x74,0x80a24011}, {0x78,0x04bffffc}, {0x7c,0x9a036004},

    {0xf0,0x5f},
    {0x00,0x82244017}, {0x04,0x82006001}, {0x08,0x9b3c201f}, {0x0c,0x81836000}, {0x10,0x01000000}, {0x14,0x01000000}, {0x18,0x01000000}, {0x1c,0xa07c0001},
    {0x20,0xc25021ae}, {0x24,0x80a40001}, {0x28,0x26800002}, {0x2c,0xa0100001}, {0x30,0x03000019}, {0x34,0x94106080}, {0x38,0x92102000}, {0x3c,0x82054009},
    {0x40,0xda086440}, {0x44,0x8204800d}, {0x48,0x99286002}, {0x4c,0x80a36000}, {0x50,0x0280001d}, {0x54,0x9610000c}, {0x58,0x80a42000}, {0x5c,0x2480000c},
    {0x60,0xc202c00a}, {0x64,0xc203000a}, {0x68,0x80a04010}, {0x6c,0x16800007}, {0x70,0x98102000}, {0x74,0x80a06000}, {0x78,0x24800006}, {0x7c,0x98100001},

    {0xf0,0x60},
    {0x00,0x10800005}, {0x04,0x8204800d}, {0x08,0xc202c00a}, {0x0c,0x98204010}, {0x10,0x8204800d}, {0x14,0x9b286002}, {0x18,0xc20023c8}, {0x1c,0x83306011},
    {0x20,0x80886001}, {0x24,0x02800005}, {0x28,0x92026001}, {0x2c,0xc20ce2af}, {0x30,0x825b0001}, {0x34,0x99386007}, {0x38,0x80a2600b}, {0x3c,0x04bfffe0},
    {0x40,0xd823400a}, {0x44,0xac05a001}, {0x48,0xa805200c}, {0x4c,0x80a5a001}, {0x50,0x04bfffab}, {0x54,0xaa05600c}, {0x58,0xa604e001}, {0x5c,0xc2002300},
    {0x60,0x80a4c001}, {0x64,0x08bfffa3}, {0x68,0xa404a00c}, {0x6c,0x81c7e008}, {0x70,0x81e80000}, {0x74,0x9de3bf58}, {0x78,0xe2502458}, {0x7c,0x80a46000},

    {0xf0,0x61},
    {0x00,0x02800065}, {0x04,0x01000000}, {0x08,0xc208217f}, {0x0c,0x80a44001}, {0x10,0x16800061}, {0x14,0xa0102001}, {0x18,0xc208217e}, {0x1c,0x80a40001},
    {0x20,0x1480005d}, {0x24,0xe450245a}, {0x28,0x96102001}, {0x2c,0xc208217f}, {0x30,0x80a2c001}, {0x34,0x1480000f}, {0x38,0x03000019}, {0x3c,0x9b2c2002},
    {0x40,0x82106080}, {0x44,0x9a034001}, {0x48,0x9a036030}, {0x4c,0x9807bfb8}, {0x50,0xc2034000}, {0x54,0xc2230000}, {0x58,0x9602e001}, {0x5c,0xc208217f},
    {0x60,0x80a2c001}, {0x64,0x9a036030}, {0x68,0x04bffffa}, {0x6c,0x98032004}, {0x70,0x9007bfb8}, {0x74,0x7ffff738}, {0x78,0xd208217f}, {0x7c,0x96100012},

    {0xf0,0x62},
    {0x00,0x80a48011}, {0x04,0x1480000b}, {0x08,0x94102000}, {0x0c,0x832ca002}, {0x10,0x8200401e}, {0x14,0x9a007fb8}, {0x18,0xc2034000}, {0x1c,0x9602e001},
    {0x20,0x94028001}, {0x24,0x80a2c011}, {0x28,0x04bffffc}, {0x2c,0x9a036004}, {0x30,0x82244012}, {0x34,0x82006001}, {0x38,0x9b3aa01f}, {0x3c,0x81836000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0x947a8001}, {0x50,0xc25021ac}, {0x54,0x80a28001}, {0x58,0x26800002}, {0x5c,0x94100001},
    {0x60,0x96102001}, {0x64,0xc208217f}, {0x68,0x80a2c001}, {0x6c,0x14800025}, {0x70,0x832c2002}, {0x74,0x11000019}, {0x78,0x98006030}, {0x7c,0x92122080},

    {0xf0,0x63},
    {0x00,0x80a2a000}, {0x04,0x0480000b}, {0x08,0x9a122080}, {0x0c,0xc203000d}, {0x10,0x80a0400a}, {0x14,0x36800008}, {0x18,0xc2030009}, {0x1c,0x80a06000},
    {0x20,0x34800007}, {0x24,0xc023000d}, {0x28,0x10800006}, {0x2c,0xc20023c8}, {0x30,0xc2030009}, {0x34,0x8220400a}, {0x38,0xc2230009}, {0x3c,0xc20023c8},
    {0x40,0x83306012}, {0x44,0x80886001}, {0x48,0x0280000a}, {0x4c,0x9602e001}, {0x50,0xda0c22af}, {0x54,0xc2030009}, {0x58,0x8258400d}, {0x5c,0x9b38601f},
    {0x60,0x9b336019}, {0x64,0x8200400d}, {0x68,0x83386007}, {0x6c,0xc2230009}, {0x70,0xc208217f}, {0x74,0x80a2c001}, {0x78,0x04bfffe2}, {0x7c,0x98032030},

    {0xf0,0x64},
    {0x00,0xa0042001}, {0x04,0xc208217e}, {0x08,0x80a40001}, {0x0c,0x04bfffa8}, {0x10,0x96102001}, {0x14,0x81c7e008}, {0x18,0x81e80000}, {0x1c,0x9de3bf98},
    {0x20,0xfa5023c6}, {0x24,0x80a76000}, {0x28,0x0280003c}, {0x2c,0xb0102001}, {0x30,0xc208217e}, {0x34,0x80a60001}, {0x38,0x14800038}, {0x3c,0x19000019},
    {0x40,0x82132080}, {0x44,0x9a10001d}, {0x48,0x9e006004}, {0x4c,0xfa08217f}, {0x50,0x80a76000}, {0x54,0xb2102000}, {0x58,0x0280002b}, {0x5c,0x82102000},
    {0x60,0xb810001d}, {0x64,0xf45023c4}, {0x68,0xba03e030}, {0x6c,0xf6074000}, {0x70,0x80a6c01a}, {0x74,0x06800004}, {0x78,0xba076030}, {0x7c,0xb206401b},

    {0xf0,0x65},
    {0x00,0x82006001}, {0x04,0xb8873fff}, {0x08,0x32bffffa}, {0x0c,0xf6074000}, {0x10,0x80a06000}, {0x14,0x2280001d}, {0x18,0xb0062001}, {0x1c,0x973e601f},
    {0x20,0x8182e000}, {0x24,0x01000000}, {0x28,0x01000000}, {0x2c,0x01000000}, {0x30,0x827e4001}, {0x34,0x8258400d}, {0x38,0xbb38601f}, {0x3c,0xbb376016},
    {0x40,0x8200401d}, {0x44,0xb8102001}, {0x48,0xfa08217f}, {0x4c,0x80a7001d}, {0x50,0x1480000d}, {0x54,0xb338600a}, {0x58,0x832e2002}, {0x5c,0xba006030},
    {0x60,0xb6132080}, {0x64,0xc207401b}, {0x68,0x82204019}, {0x6c,0xc227401b}, {0x70,0xb8072001}, {0x74,0xc208217f}, {0x78,0x80a70001}, {0x7c,0x04bffffa},

    {0xf0,0x66},
    {0x00,0xba076030}, {0x04,0xb0062001}, {0x08,0xc208217e}, {0x0c,0x80a60001}, {0x10,0x04bfffcf}, {0x14,0x9e03e004}, {0x18,0x81c7e008}, {0x1c,0x81e80000},
    {0x20,0xc2082573}, {0x24,0xda00256c}, {0x28,0x82006001}, {0x2c,0xd808257f}, {0x30,0x9a5b4001}, {0x34,0x98032001}, {0x38,0x81800000}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x9a73400c}, {0x4c,0xda20256c}, {0x50,0x96102000}, {0x54,0x832ae002}, {0x58,0xd800256c}, {0x5c,0xda1063b6},
    {0x60,0x9a5b400c}, {0x64,0x9b336007}, {0x68,0x9602e001}, {0x6c,0x80a2e003}, {0x70,0x04bffff9}, {0x74,0xda206520}, {0x78,0xc20023c0}, {0x7c,0x80a06000},

    {0xf0,0x67},
    {0x00,0x02800007}, {0x04,0x82103000}, {0x08,0xc2002200}, {0x0c,0x80a06000}, {0x10,0x22800005}, {0x14,0xc200255c}, {0x18,0x82103000}, {0x1c,0xc220252c},
    {0x20,0xc200255c}, {0x24,0x80a06000}, {0x28,0x02800005}, {0x2c,0xc2002288}, {0x30,0x9b306001}, {0x34,0x83306002}, {0x38,0x8200400d}, {0x3c,0xc220251c},
    {0x40,0x03000007}, {0x44,0x81c062d4}, {0x48,0x01000000}, {0x4c,0x01000000}, {0x50,0x9de3bf98}, {0x54,0xc2002548}, {0x58,0x80886100}, {0x5c,0x1280007e},
    {0x60,0x01000000}, {0x64,0xc20026e4}, {0x68,0x80a06000}, {0x6c,0x02800004}, {0x70,0xa4102001}, {0x74,0x82007fff}, {0x78,0xc22026e4}, {0x7c,0xc2002300},

    {0xf0,0x68},
    {0x00,0x80a48001}, {0x04,0x18800074}, {0x08,0xaa10200c}, {0x0c,0xa2102001}, {0x10,0xc2002308}, {0x14,0x80a44001}, {0x18,0x1880006a}, {0x1c,0x9b2d6002},
    {0x20,0x03000050}, {0x24,0xa8048001}, {0x28,0xa6036004}, {0x2c,0xc2002514}, {0x30,0x80a06009}, {0x34,0x18800068}, {0x38,0x03000019}, {0x3c,0x82106080},
    {0x40,0xda04c001}, {0x44,0xc2002474}, {0x48,0x80a34001}, {0x4c,0x06800057}, {0x50,0x03000040}, {0x54,0x90100011}, {0x58,0x7ffffa5b}, {0x5c,0x92100012},
    {0x60,0x80a22000}, {0x64,0x02800051}, {0x68,0x03000040}, {0x6c,0xc2002ff0}, {0x70,0x80a06000}, {0x74,0x2280000a}, {0x78,0xc2002fe0}, {0x7c,0x90100011},

    {0xf0,0x69},
    {0x00,0x9fc04000}, {0x04,0x92100012}, {0x08,0xc200259c}, {0x0c,0x80a06000}, {0x10,0x12800046}, {0x14,0x03000040}, {0x18,0xc2002fe0}, {0x1c,0x80a06000},
    {0x20,0x2280000b}, {0x24,0xc2002ff4}, {0x28,0x90100011}, {0x2c,0x92100012}, {0x30,0x9fc04000}, {0x34,0xd4002470}, {0x38,0xc200259c}, {0x3c,0x80a06000},
    {0x40,0x1280003a}, {0x44,0x03000040}, {0x48,0xc2002ff4}, {0x4c,0x80a06000}, {0x50,0x02800006}, {0x54,0xa0102000}, {0x58,0x90100011}, {0x5c,0x9fc04000},
    {0x60,0x92100012}, {0x64,0xe000259c}, {0x68,0x90100011}, {0x6c,0x92100012}, {0x70,0x94100010}, {0x74,0x7ffffa86}, {0x78,0xd6002470}, {0x7c,0x80a22000},

    {0xf0,0x6a},
    {0x00,0x0280002a}, {0x04,0x03000040}, {0x08,0xc2002fe0}, {0x0c,0x80a06000}, {0x10,0x02800007}, {0x14,0x90100011}, {0x18,0x92100012}, {0x1c,0x9fc04000},
    {0x20,0x94102100}, {0x24,0x10800004}, {0x28,0x94100010}, {0x2c,0xc020259c}, {0x30,0x94100010}, {0x34,0x90100011}, {0x38,0x92100012}, {0x3c,0x7ffffa74},
    {0x40,0x96102100}, {0x44,0x80a22000}, {0x48,0x22800013}, {0x4c,0xc2002514}, {0x50,0xc200259c}, {0x54,0x80a06000}, {0x58,0x3280000f}, {0x5c,0xc2002514},
    {0x60,0xc2002548}, {0x64,0x82106040}, {0x68,0xc2202548}, {0x6c,0xc2002280}, {0x70,0xd8002514}, {0x74,0xc22026e4}, {0x78,0x9b2c6010}, {0x7c,0x832b2002},

    {0xf0,0x6b},
    {0x00,0x9a034012}, {0x04,0xda206710}, {0x08,0x98032001}, {0x0c,0x10800006}, {0x10,0xd8202514}, {0x14,0x9b286002}, {0x18,0xe8236710}, {0x1c,0x82006001},
    {0x20,0xc2202514}, {0x24,0x03000040}, {0x28,0xa8050001}, {0x2c,0xa2046001}, {0x30,0xc2002308}, {0x34,0x80a44001}, {0x38,0x08bfff9d}, {0x3c,0xa604e004},
    {0x40,0xa404a001}, {0x44,0xc2002300}, {0x48,0x80a48001}, {0x4c,0x08bfff90}, {0x50,0xaa05600c}, {0x54,0x81c7e008}, {0x58,0x81e80000}, {0x5c,0x82220009},
    {0x60,0x9a58400a}, {0x64,0x833b601f}, {0x68,0x80a20009}, {0x6c,0x83306019}, {0x70,0x04800004}, {0x74,0x90102000}, {0x78,0x82034001}, {0x7c,0x91386007},

    {0xf0,0x6c},
    {0x00,0x81c3e008}, {0x04,0x01000000}, {0x08,0x9de3bf98}, {0x0c,0x7ffffac7}, {0x10,0xa8102001}, {0x14,0xc208217f}, {0x18,0x80a07fff}, {0x1c,0x0280001c},
    {0x20,0xa6102003}, {0x24,0x23000019}, {0x28,0xa12ce004}, {0x2c,0x82146088}, {0x30,0xa4146084}, {0x34,0xd2040001}, {0x38,0xd408228c}, {0x3c,0x7fffffe8},
    {0x40,0xd0040012}, {0x44,0x9a146080}, {0x48,0xd024000d}, {0x4c,0xc2002308}, {0x50,0xa12ce002}, {0x54,0xa0040001}, {0x58,0xa12c2002}, {0x5c,0xa214607c},
    {0x60,0xd004000d}, {0x64,0xd2040011}, {0x68,0x7fffffdd}, {0x6c,0xd408228d}, {0x70,0xd0240012}, {0x74,0xc208217f}, {0x78,0xa8052001}, {0x7c,0x82006001},

    {0xf0,0x6d},
    {0x00,0x80a50001}, {0x04,0x08bfffe8}, {0x08,0xa604e003}, {0x0c,0xa6102001}, {0x10,0xc2002308}, {0x14,0x80a4c001}, {0x18,0x1880001c}, {0x1c,0x23000019},
    {0x20,0xa12ce002}, {0x24,0x821460e0}, {0x28,0xa41460b0}, {0x2c,0xd2040001}, {0x30,0xd408228e}, {0x34,0x7fffffca}, {0x38,0xd0040012}, {0x3c,0x9a146080},
    {0x40,0xd024000d}, {0x44,0xc2002300}, {0x48,0xa0004001}, {0x4c,0xa0040001}, {0x50,0xa12c2002}, {0x54,0xa0040013}, {0x58,0xa12c2002}, {0x5c,0xa2146050},
    {0x60,0xd004000d}, {0x64,0xd2040011}, {0x68,0x7fffffbd}, {0x6c,0xd408228f}, {0x70,0xd0240012}, {0x74,0xa604e001}, {0x78,0xc2002308}, {0x7c,0x80a4c001},

    {0xf0,0x6e},
    {0x00,0x08bfffe8}, {0x04,0x23000019}, {0x08,0x81c7e008}, {0x0c,0x81e80000}, {0x10,0x9de3bf88}, {0x14,0xc2002fe4}, {0x18,0x9fc04000}, {0x1c,0xd0002590},
    {0x20,0xda002550}, {0x24,0x832b6004}, {0x28,0x8220400d}, {0x2c,0xe000259c}, {0x30,0x82040001}, {0x34,0x83306004}, {0x38,0xc2202550}, {0x3c,0x11000018},
    {0x40,0xc2002fe4}, {0x44,0x9fc04000}, {0x48,0x90122200}, {0x4c,0xc208217c}, {0x50,0xda08217d}, {0x54,0x9a5b4001}, {0x58,0xc200259c}, {0x5c,0x8200400d},
    {0x60,0xc220259c}, {0x64,0xda00259c}, {0x68,0xc2002550}, {0x6c,0x80a34001}, {0x70,0x28800011}, {0x74,0xc2002548}, {0x78,0xc200259c}, {0x7c,0x80a04010},

    {0xf0,0x6f},
    {0x00,0x0880000d}, {0x04,0xc2002548}, {0x08,0x80a42000}, {0x0c,0x0280000a}, {0x10,0x01000000}, {0x14,0x82087f7f}, {0x18,0xc2202548}, {0x1c,0xc2002658},
    {0x20,0x80a06000}, {0x24,0x26800006}, {0x28,0xc0202658}, {0x2c,0x10800005}, {0x30,0xc208254e}, {0x34,0x82106080}, {0x38,0xc2202548}, {0x3c,0xc208254e},
    {0x40,0x80a00001}, {0x44,0x82602000}, {0x48,0xa0087ffe}, {0x4c,0xd8002548}, {0x50,0x83332010}, {0x54,0x80886001}, {0x58,0x02800012}, {0x5c,0xa0042003},
    {0x60,0x03000180}, {0x64,0x820b0001}, {0x68,0x1b000080}, {0x6c,0x80a0400d}, {0x70,0x22800061}, {0x74,0xc02026d4}, {0x78,0xc2002160}, {0x7c,0x80886040},

    {0xf0,0x70},
    {0x00,0x0280002c}, {0x04,0xa0102000}, {0x08,0xc2002558}, {0x0c,0x80a06000}, {0x10,0x02800028}, {0x14,0xa0102002}, {0x18,0x10800026}, {0x1c,0xa0102000},
    {0x20,0x033fc200}, {0x24,0x82106030}, {0x28,0xda004000}, {0x2c,0xc2002570}, {0x30,0x80a34001}, {0x34,0x32800050}, {0x38,0xc02026d4}, {0x3c,0xc2002200},
    {0x40,0x80a06000}, {0x44,0x3280004c}, {0x48,0xc02026d4}, {0x4c,0xda002690}, {0x50,0x03000007}, {0x54,0x808b4001}, {0x58,0x32800047}, {0x5c,0xc02026d4},
    {0x60,0xda002654}, {0x64,0xc20021dc}, {0x68,0x80a34001}, {0x6c,0x2a800008}, {0x70,0xc2002514}, {0x74,0xc200254c}, {0x78,0x8330600e}, {0x7c,0x80886001},

    {0xf0,0x71},
    {0x00,0x1280000a}, {0x04,0x808b2040}, {0x08,0xc2002514}, {0x0c,0x80a06000}, {0x10,0x22800009}, {0x14,0xc20c2314}, {0x18,0x808b2080}, {0x1c,0x22800006},
    {0x20,0xc20c2314}, {0x24,0x808b2040}, {0x28,0x32800033}, {0x2c,0xc02026d4}, {0x30,0xc20c2314}, {0x34,0x80a06000}, {0x38,0x2280002f}, {0x3c,0xc02026d4},
    {0x40,0xc20026d4}, {0x44,0x98006001}, {0x48,0xc2002700}, {0x4c,0xda082169}, {0x50,0xd82026d4}, {0x54,0x80a0400d}, {0x58,0x0480000a}, {0x5c,0xc20c2314},
    {0x60,0x820860ff}, {0x64,0xda082168}, {0x68,0x8258400d}, {0x6c,0x80a30001}, {0x70,0x06800021}, {0x74,0x01000000}, {0x78,0x10800007}, {0x7c,0xc02026d4},

    {0xf0,0x72},
    {0x00,0x820860ff}, {0x04,0x80a30001}, {0x08,0x0680001b}, {0x0c,0x01000000}, {0x10,0xc02026d4}, {0x14,0xc20c234c}, {0x18,0xc227bfec}, {0x1c,0xc20c230c},
    {0x20,0xc227bff0}, {0x24,0xa0102000}, {0x28,0xc208217c}, {0x2c,0x80a40001}, {0x30,0x1a800011}, {0x34,0x01000000}, {0x38,0xc20c2380}, {0x3c,0xc227bfe8},
    {0x40,0xa2102000}, {0x44,0xc208217d}, {0x48,0x80a44001}, {0x4c,0x3a800006}, {0x50,0xa0042001}, {0x54,0x7ffffa09}, {0x58,0x9007bfe8}, {0x5c,0x10bffffa},
    {0x60,0xa2046001}, {0x64,0xc208217c}, {0x68,0x80a40001}, {0x6c,0x2abffff4}, {0x70,0xc20c2380}, {0x74,0x81c7e008}, {0x78,0x81e80000}, {0x7c,0x92102000},

    {0xf0,0x73},
    {0x00,0xc2002514}, {0x04,0x80a24001}, {0x08,0x1a800019}, {0x0c,0x912a6002}, {0x10,0xd4022710}, {0x14,0x820aafff}, {0x18,0x9732a010}, {0x1c,0x98006001},
    {0x20,0x960aefff}, {0x24,0x82007fff}, {0x28,0x83286018}, {0x2c,0x992b2010}, {0x30,0x9a02ffff}, {0x34,0x8200400c}, {0x38,0x9b2b6008}, {0x3c,0x8200400d},
    {0x40,0x8200400b}, {0x44,0x9532a00c}, {0x48,0x92026001}, {0x4c,0x808aa001}, {0x50,0x12800003}, {0x54,0x82006001}, {0x58,0xc2222738}, {0x5c,0xc2002514},
    {0x60,0x80a24001}, {0x64,0x0abfffeb}, {0x68,0x912a6002}, {0x6c,0x81c3e008}, {0x70,0x01000000}, {0x74,0x9de3bf98}, {0x78,0x92102000}, {0x7c,0xc2002514},

    {0xf0,0x74},
    {0x00,0x80a24001}, {0x04,0x1a80005f}, {0x08,0x94102000}, {0x0c,0xb72a6002}, {0x10,0xf20ee738}, {0x14,0xc20ee739}, {0x18,0x80a64001}, {0x1c,0x96102000},
    {0x20,0x9e102000}, {0x24,0x14800022}, {0x28,0x9a102000}, {0x2c,0x82064019}, {0x30,0x82004019}, {0x34,0x3b000019}, {0x38,0x90176080}, {0x3c,0xb9286002},
    {0x40,0xc20ee73b}, {0x44,0xf40ee73a}, {0x48,0x80a68001}, {0x4c,0x14800012}, {0x50,0x8207001a}, {0x54,0x83286002}, {0x58,0x82004008}, {0x5c,0xd80aa73b},
    {0x60,0xf6004000}, {0x64,0xba5ec01a}, {0x68,0x82006004}, {0x6c,0xb406a001}, {0x70,0x80a6e000}, {0x74,0x04800005}, {0x78,0xb05ec019}, {0x7c,0x9e03c018},

    {0xf0,0x75},
    {0x00,0x9a03401d}, {0x04,0x9602c01b}, {0x08,0x80a6800c}, {0x0c,0x24bffff6}, {0x10,0xf6004000}, {0x14,0xb72a6002}, {0x18,0xb2066001}, {0x1c,0xc20ee739},
    {0x20,0x80a64001}, {0x24,0x04bfffe7}, {0x28,0xb807200c}, {0x2c,0xbb2b6006}, {0x30,0xb92be006}, {0x34,0xc202a710}, {0x38,0x9b3f201f}, {0x3c,0x81836000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0xb87f000b}, {0x50,0x9b3f601f}, {0x54,0x81836000}, {0x58,0x01000000}, {0x5c,0x01000000},
    {0x60,0x01000000}, {0x64,0xba7f400b}, {0x68,0x8330600c}, {0x6c,0x80886001}, {0x70,0x9e073fe0}, {0x74,0x02800007}, {0x78,0x9a077fe0}, {0x7c,0xc200237c},

    {0xf0,0x76},
    {0x00,0x80a06000}, {0x04,0x22800003}, {0x08,0x9e073fa0}, {0x0c,0x9a077fa0}, {0x10,0x80a3e000}, {0x14,0x24800002}, {0x18,0x9e102001}, {0x1c,0x80a36000},
    {0x20,0x24800002}, {0x24,0x9a102001}, {0x28,0xc208217f}, {0x2c,0x83286006}, {0x30,0x80a3c001}, {0x34,0x36800002}, {0x38,0x9e007fff}, {0x3c,0xc208217e},
    {0x40,0x83286006}, {0x44,0x80a34001}, {0x48,0x36800002}, {0x4c,0x9a007fff}, {0x50,0xfa02a710}, {0x54,0x832be010}, {0x58,0x3900003c}, {0x5c,0x8200400d},
    {0x60,0xba0f401c}, {0x64,0x8200401d}, {0x68,0xc222a710}, {0x6c,0x92026001}, {0x70,0xc2002514}, {0x74,0x80a24001}, {0x78,0x0abfffa5}, {0x7c,0x9402a004},

    {0xf0,0x77},
    {0x00,0x7ffff813}, {0x04,0x91e8205a}, {0x08,0x01000000}, {0x0c,0x9de3bf98}, {0x10,0xd8002548}, {0x14,0x8333200c}, {0x18,0x80886001}, {0x1c,0x12800081},
    {0x20,0x01000000}, {0x24,0xc208254d}, {0x28,0x80a06000}, {0x2c,0x1280007d}, {0x30,0x01000000}, {0x34,0xc208254e}, {0x38,0x80a06000}, {0x3c,0x02800079},
    {0x40,0x01000000}, {0x44,0xc2002554}, {0x48,0x825860fa}, {0x4c,0xda002500}, {0x50,0x80a34001}, {0x54,0x0a800073}, {0x58,0x01000000}, {0x5c,0xd6002658},
    {0x60,0x80a2e000}, {0x64,0x1480001c}, {0x68,0x8202ffff}, {0x6c,0xc2002514}, {0x70,0x80a06001}, {0x74,0x3280006b}, {0x78,0xc0202658}, {0x7c,0xc2002710},

    {0xf0,0x78},
    {0x00,0x80a06000}, {0x04,0x12800004}, {0x08,0x808b2040}, {0x0c,0x10800065}, {0x10,0xc0202658}, {0x14,0x32800006}, {0x18,0xd800239c}, {0x1c,0x80a2e000},
    {0x20,0x0680000d}, {0x24,0x8202e001}, {0x28,0xd800239c}, {0x2c,0x0300003f}, {0x30,0x9b332010}, {0x34,0x821063ff}, {0x38,0x980b0001}, {0x3c,0x9a38000d},
    {0x40,0x9a23400c}, {0x44,0x80a2c00d}, {0x48,0x06800005}, {0x4c,0x03296956}, {0x50,0x8202ffff}, {0x54,0x10800053}, {0x58,0xc2202658}, {0x5c,0x9a10625a},
    {0x60,0xc200250c}, {0x64,0x80a0400d}, {0x68,0x02800004}, {0x6c,0xa0102000}, {0x70,0xda20250c}, {0x74,0xc0202654}, {0x78,0x832c2002}, {0x7c,0x92006610},

    {0xf0,0x79},
    {0x00,0xc2006610}, {0x04,0x90102710}, {0x08,0x80a06000}, {0x0c,0x0280000d}, {0x10,0xa0042001}, {0x14,0x7ffff83c}, {0x18,0x01000000}, {0x1c,0xc2002300},
    {0x20,0xda002308}, {0x24,0x82584001}, {0x28,0x9a5b400d}, {0x2c,0x8200400d}, {0x30,0x83286006}, {0x34,0x80a20001}, {0x38,0x0a80003a}, {0x3c,0x01000000},
    {0x40,0x80a4200f}, {0x44,0x08bfffee}, {0x48,0x832c2002}, {0x4c,0x030041eb}, {0x50,0xc2202658}, {0x54,0xa010200f}, {0x58,0x9b2c2002}, {0x5c,0xc203660c},
    {0x60,0xa0843fff}, {0x64,0x12bffffd}, {0x68,0xc2236610}, {0x6c,0xda002654}, {0x70,0xc2002710}, {0x74,0xc2202610}, {0x78,0x80a3600f}, {0x7c,0x14800003},

    {0xf0,0x7a},
    {0x00,0x9010200f}, {0x04,0x9010000d}, {0x08,0x03000017}, {0x0c,0x9e106240}, {0x10,0x82006400}, {0x14,0xa2106200}, {0x18,0x0300003f}, {0x1c,0xa61063ff},
    {0x20,0x92022001}, {0x24,0xa0102000}, {0x28,0xa410000f}, {0x2c,0x972c2002}, {0x30,0x8203c00b}, {0x34,0xda106002}, {0x38,0x9804400b}, {0x3c,0xd4132002},
    {0x40,0xc213c00b}, {0x44,0xd814400b}, {0x48,0x82584008}, {0x4c,0x9a5b4008}, {0x50,0x8200400c}, {0x54,0x9a03400a}, {0x58,0x81800000}, {0x5c,0x01000000},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x82704009}, {0x6c,0x81800000}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x01000000}, {0x7c,0x9a734009},

    {0xf0,0x7b},
    {0x00,0x83286010}, {0x04,0x9a0b4013}, {0x08,0x8200400d}, {0x0c,0xa0042001}, {0x10,0x80a4204f}, {0x14,0x08bfffe6}, {0x18,0xc222c012}, {0x1c,0xd2202654},
    {0x20,0x81c7e008}, {0x24,0x81e80000}, {0x28,0x9de3bf98}, {0x2c,0xa2102000}, {0x30,0xc2002514}, {0x34,0x80a44001}, {0x38,0xa0102000}, {0x3c,0x1a800012},
    {0x40,0x832c2002}, {0x44,0x92006710}, {0x48,0xc2006710}, {0x4c,0x80a06000}, {0x50,0x02800009}, {0x54,0x90100018}, {0x58,0x7ffff7eb}, {0x5c,0x01000000},
    {0x60,0x82102001}, {0x64,0x80a20019}, {0x68,0x1a800003}, {0x6c,0x83284010}, {0x70,0xa2144001}, {0x74,0xa0042001}, {0x78,0xc2002514}, {0x7c,0x10bffff0},

    {0xf0,0x7c},
    {0x00,0x80a40001}, {0x04,0x81c7e008}, {0x08,0x91e80011}, {0x0c,0x9de3bf90}, {0x10,0x03003fc0}, {0x14,0x82106001}, {0x18,0xc227bff0}, {0x1c,0xc027bff4},
    {0x20,0xa0102000}, {0x24,0x8207bff8}, {0x28,0x82040001}, {0x2c,0xd2487ff9}, {0x30,0xd0487ff8}, {0x34,0x90060008}, {0x38,0x92064009}, {0x3c,0x94102000},
    {0x40,0x7ffff833}, {0x44,0x9610001a}, {0x48,0x80a22000}, {0x4c,0x02800004}, {0x50,0xa0042001}, {0x54,0x10800006}, {0x58,0xb0102001}, {0x5c,0x80a42004},
    {0x60,0x04bffff2}, {0x64,0x8207bff8}, {0x68,0xb0102000}, {0x6c,0x81c7e008}, {0x70,0x81e80000}, {0x74,0x9de3bf98}, {0x78,0xc2102548}, {0x7c,0x80886001},

    {0xf0,0x7d},
    {0x00,0xb8102000}, {0x04,0xba102000}, {0x08,0x02800004}, {0x0c,0xae102000}, {0x10,0x10800026}, {0x14,0xec002354}, {0x18,0xc200254c}, {0x1c,0x80886200},
    {0x20,0x32800010}, {0x24,0xc2002690}, {0x28,0x03296956}, {0x2c,0xda00250c}, {0x30,0x8210625a}, {0x34,0x80a34001}, {0x38,0x12800006}, {0x3c,0xc20023c8},
    {0x40,0x80886800}, {0x44,0x32800007}, {0x48,0xc2002690}, {0x4c,0xc20023c8}, {0x50,0x80886400}, {0x54,0x22800015}, {0x58,0xec0022ac}, {0x5c,0xc2002690},
    {0x60,0x80886200}, {0x64,0x1280000a}, {0x68,0xb8102001}, {0x6c,0x13000017}, {0x70,0xc2002fd0}, {0x74,0x92126240}, {0x78,0x9fc04000}, {0x7c,0xd0002590},

    {0xf0,0x7e},
    {0x00,0xc2002690}, {0x04,0x82106200}, {0x08,0xc2202690}, {0x0c,0x10800007}, {0x10,0xec0023a0}, {0x14,0xc200254c}, {0x18,0x1b000080}, {0x1c,0x8210400d},
    {0x20,0x10800080}, {0x24,0xc220254c}, {0x28,0x7ffff729}, {0x2c,0x901020aa}, {0x30,0xda002500}, {0x34,0xc20022d0}, {0x38,0x80a34001}, {0x3c,0x3880000b},
    {0x40,0xba102001}, {0x44,0xc2002200}, {0x48,0x80a00001}, {0x4c,0x9a402000}, {0x50,0x80a00016}, {0x54,0x82603fff}, {0x58,0x80934001}, {0x5c,0x02800004},
    {0x60,0xaa102000}, {0x64,0xba102001}, {0x68,0xaa102000}, {0x6c,0xa52d6003}, {0x70,0xc204a760}, {0x74,0x80a06000}, {0x78,0x22800067}, {0x7c,0xaa056001},

    {0xf0,0x7f},
    {0x00,0x80a56000}, {0x04,0x12800008}, {0x08,0x80a76000}, {0x0c,0xc200255c}, {0x10,0x80a06000}, {0x14,0x1280000c}, {0x18,0x29200000}, {0x1c,0x1080000a},
    {0x20,0xa8102000}, {0x24,0x02800005}, {0x28,0x9004a760}, {0x2c,0xc024a764}, {0x30,0x10800058}, {0x34,0xc024a760}, {0x38,0x7fffff7c}, {0x3c,0x92100016},
    {0x40,0xa8100008}, {0x44,0x80a52000}, {0x48,0x02800050}, {0x4c,0xae15c014}, {0x50,0xc214a766}, {0x54,0x80a06000}, {0x58,0x0280000a}, {0x5c,0x80a56000},
    {0x60,0xc204a764}, {0x64,0x82007fff}, {0x68,0xc224a764}, {0x6c,0xc214a766}, {0x70,0x80a06000}, {0x74,0x32800048}, {0x78,0xaa056001}, {0x7c,0x80a56000},

    {0xf0,0x80},
    {0x00,0x12800006}, {0x04,0x1b004000}, {0x08,0xc200254c}, {0x0c,0x8210400d}, {0x10,0x10800040}, {0x14,0xc220254c}, {0x18,0xc2102548}, {0x1c,0x80886001},
    {0x20,0x02800006}, {0x24,0x82103fff}, {0x28,0xc024a764}, {0x2c,0xc024a760}, {0x30,0x10800038}, {0x34,0xc2202700}, {0x38,0x80a72000}, {0x3c,0x02bfffb6},
    {0x40,0xa6102000}, {0x44,0xc2002514}, {0x48,0x80a4c001}, {0x4c,0x1a80002c}, {0x50,0x83350013}, {0x54,0x80886001}, {0x58,0x02800027}, {0x5c,0x9b2ce002},
    {0x60,0xe0036710}, {0x64,0xa3342016}, {0x68,0xa00c2fff}, {0x6c,0xa13c2006}, {0x70,0xa20c603f}, {0x74,0xa0042001}, {0x78,0xa2046001}, {0x7c,0x92100011},

    {0xf0,0x81},
    {0x00,0xd410246e}, {0x04,0x7fffff62}, {0x08,0x90100010}, {0x0c,0x80a22000}, {0x10,0x92100011}, {0x14,0x1280000f}, {0x18,0x90100010}, {0x1c,0xc210246c},
    {0x20,0x94906000}, {0x24,0x2280000f}, {0x28,0xc200254c}, {0x2c,0xc2002548}, {0x30,0x80886080}, {0x34,0x2280000b}, {0x38,0xc200254c}, {0x3c,0x7fffff54},
    {0x40,0x01000000}, {0x44,0x80a22000}, {0x48,0x22800006}, {0x4c,0xc200254c}, {0x50,0x82102001}, {0x54,0x83284013}, {0x58,0x10800007}, {0x5c,0xa82d0001},
    {0x60,0x1b000080}, {0x64,0x8210400d}, {0x68,0xc220254c}, {0x6c,0x10800004}, {0x70,0xc02026e8}, {0x74,0x10bfffd4}, {0x78,0xa604e001}, {0x7c,0x80a52000},

    {0xf0,0x82},
    {0x00,0x32800005}, {0x04,0xaa056001}, {0x08,0xc024a764}, {0x0c,0xc024a760}, {0x10,0xaa056001}, {0x14,0x80a56013}, {0x18,0x04bfff96}, {0x1c,0xa52d6003},
    {0x20,0x80a5e000}, {0x24,0x22800002}, {0x28,0xc02026e8}, {0x2c,0xc200255c}, {0x30,0x80a06000}, {0x34,0x0280000a}, {0x38,0x80a76000}, {0x3c,0xc2002760},
    {0x40,0x80a06000}, {0x44,0x12800006}, {0x48,0x80a76000}, {0x4c,0x03000004}, {0x50,0xc2202760}, {0x54,0xc2002248}, {0x58,0xc2202764}, {0x5c,0x1280002b},
    {0x60,0xaa102000}, {0x64,0xc2002514}, {0x68,0x80a54001}, {0x6c,0x1a800027}, {0x70,0x96102001}, {0x74,0x992d6002}, {0x78,0xc2032710}, {0x7c,0x8330600c},

    {0xf0,0x83},
    {0x00,0x80886001}, {0x04,0x3280001d}, {0x08,0xaa056001}, {0x0c,0x832ac015}, {0x10,0x808dc001}, {0x14,0x32800019}, {0x18,0xaa056001}, {0x1c,0xa6102001},
    {0x20,0x9b2ce003}, {0x24,0xc2036760}, {0x28,0x80a06000}, {0x2c,0x1280000f}, {0x30,0xa604e001}, {0x34,0xc2032710}, {0x38,0xc2236760}, {0x3c,0xc2102548},
    {0x40,0x80886001}, {0x44,0x02800004}, {0x48,0x80a72000}, {0x4c,0x10800005}, {0x50,0xc2002358}, {0x54,0x22800003}, {0x58,0xc2002378}, {0x5c,0xc200239c},
    {0x60,0x10800005}, {0x64,0xc2236764}, {0x68,0x80a4e013}, {0x6c,0x04bfffee}, {0x70,0x9b2ce003}, {0x74,0xaa056001}, {0x78,0xc2002514}, {0x7c,0x80a54001},

    {0xf0,0x84},
    {0x00,0x0abfffde}, {0x04,0x992d6002}, {0x08,0x81c7e008}, {0x0c,0x81e80000}, {0x10,0x9de3bf98}, {0x14,0x7ffff7c5}, {0x18,0x01000000}, {0x1c,0xda002548},
    {0x20,0x83336010}, {0x24,0x80886001}, {0x28,0x12800032}, {0x2c,0x8333600c}, {0x30,0x80886001}, {0x34,0x1280002f}, {0x38,0x01000000}, {0x3c,0xc2002538},
    {0x40,0x80a06000}, {0x44,0x2280000b}, {0x48,0xc208254d}, {0x4c,0x82007fff}, {0x50,0x80a06000}, {0x54,0x12800006}, {0x58,0xc2202538}, {0x5c,0xc200254c},
    {0x60,0x1b002000}, {0x64,0x8210400d}, {0x68,0xc220254c}, {0x6c,0xc208254d}, {0x70,0x80a06000}, {0x74,0x0280001f}, {0x78,0x033fc180}, {0x7c,0xc0204000},

    {0xf0,0x85},
    {0x00,0xa0102002}, {0x04,0x7ffff7ff}, {0x08,0x90102001}, {0x0c,0x11000099}, {0x10,0x7ffff2c1}, {0x14,0x9012233c}, {0x18,0xa0843fff}, {0x1c,0x1cbffffa},
    {0x20,0x01000000}, {0x24,0x7ffff7f7}, {0x28,0x90102000}, {0x2c,0xda00254c}, {0x30,0x83336010}, {0x34,0x80886001}, {0x38,0x32800002}, {0x3c,0xc020250c},
    {0x40,0x83336017}, {0x44,0x80886001}, {0x48,0x32800005}, {0x4c,0xc2002538}, {0x50,0xc2002188}, {0x54,0xc2202538}, {0x58,0xc2002538}, {0x5c,0xc220265c},
    {0x60,0x7ffff4e6}, {0x64,0x90102015}, {0x68,0x82102001}, {0x6c,0xc2202584}, {0x70,0x81c7e008}, {0x74,0x81e80000}, {0x78,0xc2002588}, {0x7c,0x80a06000},

    {0xf0,0x86},
    {0x00,0x32800006}, {0x04,0xc200258c}, {0x08,0xc2002554}, {0x0c,0xc2202588}, {0x10,0xc0202594}, {0x14,0xc200258c}, {0x18,0x80a06000}, {0x1c,0x32800006},
    {0x20,0xc2102548}, {0x24,0xc2002554}, {0x28,0xc220258c}, {0x2c,0xc0202598}, {0x30,0xc2102548}, {0x34,0x80886001}, {0x38,0x02800007}, {0x3c,0x01000000},
    {0x40,0xc2002558}, {0x44,0x80a06001}, {0x48,0x18800003}, {0x4c,0x82102001}, {0x50,0xc2202584}, {0x54,0x81c3e008}, {0x58,0x01000000}, {0x5c,0x9de3bf98},
    {0x60,0xb0102000}, {0x64,0xc2002514}, {0x68,0x80a60001}, {0x6c,0x1a800053}, {0x70,0x9b2e2002}, {0x74,0xc2036710}, {0x78,0xbb30600c}, {0x7c,0xb9306010},

    {0xf0,0x87},
    {0x00,0xb80f2fff}, {0x04,0x9e8f6001}, {0x08,0x12800014}, {0x0c,0xb6086fff}, {0x10,0xc250229e}, {0x14,0xfa5022a2}, {0x18,0x8226c001}, {0x1c,0xba27001d},
    {0x20,0xf850229c}, {0x24,0xf65022a0}, {0x28,0x8258401c}, {0x2c,0xba5f401b}, {0x30,0x82006800}, {0x34,0xba076800}, {0x38,0xb938601f}, {0x3c,0xb73f601f},
    {0x40,0xb9372014}, {0x44,0xb736e014}, {0x48,0x8200401c}, {0x4c,0xba07401b}, {0x50,0xb738600c}, {0x54,0xb93f600c}, {0x58,0xf4002324}, {0x5c,0xf2002328},
    {0x60,0xfa002308}, {0x64,0xc2002300}, {0x68,0xb65ec01a}, {0x6c,0xbb2f6006}, {0x70,0xb85f0019}, {0x74,0x83286006}, {0x78,0x993ee01f}, {0x7c,0x81832000},

    {0xf0,0x88},
    {0x00,0x01000000}, {0x04,0x01000000}, {0x08,0x01000000}, {0x0c,0xb67ec01d}, {0x10,0x993f201f}, {0x14,0x81832000}, {0x18,0x01000000}, {0x1c,0x01000000},
    {0x20,0x01000000}, {0x24,0xb87f0001}, {0x28,0x80a3e000}, {0x2c,0x3280001c}, {0x30,0x832ee010}, {0x34,0x80a6e000}, {0x38,0x24800002}, {0x3c,0xb6102001},
    {0x40,0x80a6c01a}, {0x44,0x3a800002}, {0x48,0xb606bfff}, {0x4c,0x80a72000}, {0x50,0x24800002}, {0x54,0xb8102001}, {0x58,0x80a70019}, {0x5c,0x3a800002},
    {0x60,0xb8067fff}, {0x64,0xc20023c8}, {0x68,0x80886002}, {0x6c,0x32800002}, {0x70,0xb626801b}, {0x74,0x80886004}, {0x78,0x32800002}, {0x7c,0xb826401c},

    {0xf0,0x89},
    {0x00,0x80886008}, {0x04,0x02800006}, {0x08,0x832ee010}, {0x0c,0xb61ec01c}, {0x10,0xb81f001b}, {0x14,0xb61ec01c}, {0x18,0x832ee010}, {0x1c,0x8200401c},
    {0x20,0xc2236710}, {0x24,0xb0062001}, {0x28,0xc2002514}, {0x2c,0x80a60001}, {0x30,0x0abfffb1}, {0x34,0x9b2e2002}, {0x38,0x81c7e008}, {0x3c,0x81e80000},
    {0x40,0x031fffff}, {0x44,0x9002200c}, {0x48,0x821063ff}, {0x4c,0x9a102063}, {0x50,0xc2220000}, {0x54,0x9a837fff}, {0x58,0x1cbffffe}, {0x5c,0x90022004},
    {0x60,0x81c3e008}, {0x64,0x01000000}, {0x68,0x031fffff}, {0x6c,0x821063ff}, {0x70,0xc2222008}, {0x74,0x92102000}, {0x78,0x96100008}, {0x7c,0x94102000},

    {0xf0,0x8a},
    {0x00,0x9a02e00c}, {0x04,0xd8034000}, {0x08,0xc2022008}, {0x0c,0x80a30001}, {0x10,0x16800005}, {0x14,0x9a036004}, {0x18,0xd8222008}, {0x1c,0xd4220000},
    {0x20,0xd2222004}, {0x24,0x9402a001}, {0x28,0x80a2a009}, {0x2c,0x24bffff7}, {0x30,0xd8034000}, {0x34,0x92026001}, {0x38,0x80a26009}, {0x3c,0x04bffff0},
    {0x40,0x9602e028}, {0x44,0xda022008}, {0x48,0x03200000}, {0x4c,0x8238400d}, {0x50,0x80a00001}, {0x54,0x81c3e008}, {0x58,0x90402000}, {0x5c,0xc2022004},
    {0x60,0x9b286002}, {0x64,0x9a034001}, {0x68,0x031fffff}, {0x6c,0x9b2b6003}, {0x70,0x9a034008}, {0x74,0x981063ff}, {0x78,0x9a03600c}, {0x7c,0x82102009},

    {0xf0,0x8b},
    {0x00,0xd8234000}, {0x04,0x82807fff}, {0x08,0x1cbffffe}, {0x0c,0x9a036004}, {0x10,0xc2020000}, {0x14,0x83286002}, {0x18,0x82004008}, {0x1c,0x8200600c},
    {0x20,0x9a102009}, {0x24,0xd8204000}, {0x28,0x9a837fff}, {0x2c,0x1cbffffe}, {0x30,0x82006028}, {0x34,0x81c3e008}, {0x38,0x01000000}, {0x3c,0x9de3bdf8},
    {0x40,0xac07be58}, {0x44,0x7fffffbf}, {0x48,0x90100016}, {0x4c,0xaa102000}, {0x50,0xa8102000}, {0x54,0xc2052800}, {0x58,0x80a06000}, {0x5c,0x22800016},
    {0x60,0xaa056001}, {0x64,0xa4102000}, {0x68,0xc2002514}, {0x6c,0x80a48001}, {0x70,0x3a800011}, {0x74,0xaa056001}, {0x78,0xa2100014}, {0x7c,0xa6102710},

    {0xf0,0x8c},
    {0x00,0x90100013}, {0x04,0xa007bff8}, {0x08,0x7ffff5df}, {0x0c,0x92052800}, {0x10,0xa0044010}, {0x14,0xd0243e6c}, {0x18,0xa404a001}, {0x1c,0xc2002514},
    {0x20,0x80a48001}, {0x24,0xa604e004}, {0x28,0x0abffff6}, {0x2c,0xa2046028}, {0x30,0xaa056001}, {0x34,0x80a56009}, {0x38,0x04bfffe7}, {0x3c,0xa8052004},
    {0x40,0xa4102000}, {0x44,0xc2002514}, {0x48,0x80a48001}, {0x4c,0x1a800022}, {0x50,0x01000000}, {0x54,0x7fffffa5}, {0x58,0x90100016}, {0x5c,0x80a22000},
    {0x60,0xa004a001}, {0x64,0x0280001c}, {0x68,0x90100016}, {0x6c,0xc207be60}, {0x70,0xda002230}, {0x74,0x80a0400d}, {0x78,0xc207be5c}, {0x7c,0x0a800007},

    {0xf0,0x8d},
    {0x00,0x97286002}, {0x04,0xda02e710}, {0x08,0x033c0000}, {0x0c,0x822b4001}, {0x10,0x1080000d}, {0x14,0xc222e710}, {0x18,0xc207be58}, {0x1c,0x83286002},
    {0x20,0xda006800}, {0x24,0xd802e710}, {0x28,0x9b33601c}, {0x2c,0x033c0000}, {0x30,0x822b0001}, {0x34,0x9b2b601c}, {0x38,0x8210400d}, {0x3c,0x7fffffa8},
    {0x40,0xc222e710}, {0x44,0xc2002514}, {0x48,0x80a40001}, {0x4c,0x0abfffe2}, {0x50,0xa4100010}, {0x54,0x81c7e008}, {0x58,0x81e80000}, {0x5c,0x9de3bf98},
    {0x60,0xb2102000}, {0x64,0xc2002514}, {0x68,0x80a64001}, {0x6c,0x1a800034}, {0x70,0xb0102000}, {0x74,0x0303ffff}, {0x78,0x9e1063ff}, {0x7c,0xb4102000},

    {0xf0,0x8e},
    {0x00,0xc206a710}, {0x04,0x8330601c}, {0x08,0x80a06000}, {0x0c,0x12800024}, {0x10,0xbb2e6002}, {0x14,0xba102001}, {0x18,0xb6102000}, {0x1c,0xb92ee002},
    {0x20,0xc2072710}, {0x24,0x8330601c}, {0x28,0x80a0401d}, {0x2c,0x0280000c}, {0x30,0x80a6e00a}, {0x34,0xc2072800}, {0x38,0x8330601c}, {0x3c,0x80a0401d},
    {0x40,0x02800007}, {0x44,0x80a6e00a}, {0x48,0xb606e001}, {0x4c,0x80a6e009}, {0x50,0x08bffff4}, {0x54,0xb92ee002}, {0x58,0x80a6e00a}, {0x5c,0x22800007},
    {0x60,0xf806a710}, {0x64,0xba076001}, {0x68,0x80a7600a}, {0x6c,0x08bfffec}, {0x70,0xb6102000}, {0x74,0xf806a710}, {0x78,0x8337200e}, {0x7c,0x80886001},

    {0xf0,0x8f},
    {0x00,0x3280000b}, {0x04,0xb0062001}, {0x08,0xbb2f601c}, {0x0c,0x820f000f}, {0x10,0x8210401d}, {0x14,0xc226a710}, {0x18,0xbb2e6002}, {0x1c,0xc206a710},
    {0x20,0xc2276710}, {0x24,0xb2066001}, {0x28,0xb0062001}, {0x2c,0xc2002514}, {0x30,0x80a60001}, {0x34,0x0abfffd3}, {0x38,0xb406a004}, {0x3c,0xf2202514},
    {0x40,0x81c7e008}, {0x44,0x81e80000}, {0x48,0x033fc180}, {0x4c,0x96106004}, {0x50,0x98102000}, {0x54,0x9b2b2002}, {0x58,0xc0236800}, {0x5c,0xc2002514},
    {0x60,0x80a30001}, {0x64,0x1a800005}, {0x68,0x98032001}, {0x6c,0xc2036710}, {0x70,0xc2236800}, {0x74,0xc223400b}, {0x78,0x80a32009}, {0x7c,0x08bffff7},

    {0xf0,0x90},
    {0x00,0x9b2b2002}, {0x04,0xda002514}, {0x08,0x033fc180}, {0x0c,0x81c3e008}, {0x10,0xda204000}, {0x14,0x9de3bf98}, {0x18,0xd8002548}, {0x1c,0x8333200e},
    {0x20,0x80886001}, {0x24,0x22800006}, {0x28,0xc2102516}, {0x2c,0x03000010}, {0x30,0x822b0001}, {0x34,0x1080001e}, {0x38,0xc2202548}, {0x3c,0x80a06000},
    {0x40,0x02800006}, {0x44,0x9b332003}, {0x48,0x808b2004}, {0x4c,0x2280000a}, {0x50,0xc200231c}, {0x54,0x9b332003}, {0x58,0x83332002}, {0x5c,0x82086001},
    {0x60,0x9a0b6001}, {0x64,0x80a0400d}, {0x68,0x2280000a}, {0x6c,0xc2002560}, {0x70,0xc200231c}, {0x74,0x80a06000}, {0x78,0x22800003}, {0x7c,0xc2082360},

    {0xf0,0x91},
    {0x00,0x82102005}, {0x04,0xc2202560}, {0x08,0x10800007}, {0x0c,0x90102001}, {0x10,0x80a06000}, {0x14,0x02800004}, {0x18,0x90102000}, {0x1c,0x10bffffa},
    {0x20,0x82007fff}, {0x24,0x7ffff677}, {0x28,0x01000000}, {0x2c,0xc2002548}, {0x30,0x9a004001}, {0x34,0x9a0b6008}, {0x38,0x82087ff7}, {0x3c,0x8210400d},
    {0x40,0xc2202548}, {0x44,0x81c7e008}, {0x48,0x81e80000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},

    {0xf0,0x92},
    {0x00,0x00001688}, {0x04,0xa5010503}, {0x08,0xa5000000}, {0x0c,0x00000000}, {0x10,0x4c494348}, {0x14,0x49444449}, {0x18,0x88ecbdae}, {0x1c,0x39cf8648},
    {0x20,0xbe71c654}, {0x24,0x80cb80d1}, {0x28,0xc471cce5}, {0x2c,0xc9f2c400}, {0x30,0xbc0dbc43}, {0x34,0xc643a060}, {0x38,0xbd130000}, {0x3c,0x00000000},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x41756720}, {0x6c,0x31322032}, {0x70,0x30313500}, {0x74,0x00000000}, {0x78,0x31383a31}, {0x7c,0x393a3338},
};

#elif GSL1680_FW == GSL1680_FW_2

const struct fw_data GSLX680_FW[] = {
    //GSL_1680E+1688E+2681B+2682B_V1.3.4_2013.02.28

    {0xf0,0x3},
    {0x00,0xa5a5ffc0}, {0x04,0x00000000}, {0x08,0xe810c4e1}, {0x0c,0xd3dd7f4d}, {0x10,0xd7c56634}, {0x14,0xe3505a2a}, {0x18,0x514d494f}, {0x1c,0xafebf471},
    {0x20,0x00000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00001000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x00000001}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x01020304}, {0x58,0x05060708}, {0x5c,0x090a0b0c},
    {0x60,0x0d0e0e0f}, {0x64,0x10111213}, {0x68,0x14151617}, {0x6c,0x18191a1b}, {0x70,0x1b1c1e1f}, {0x74,0x00000000}, {0x78,0x00010000}, {0x7c,0x8c846af3},

    {0xf0,0x4},
    {0x00,0x00000000}, {0x04,0x00000000}, {0x08,0x00000000}, {0x0c,0x00000000}, {0x10,0xffffff38}, {0x14,0x00000000}, {0x18,0x00000000}, {0x1c,0x00000000},
    {0x20,0x00000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00002400}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00010203}, {0x58,0x03040506}, {0x5c,0x06070808},
    {0x60,0x090a0b0c}, {0x64,0x0d0e0f10}, {0x68,0x10111314}, {0x6c,0x15161819}, {0x70,0x1a1b1d1f}, {0x74,0x00000000}, {0x78,0x8080a680}, {0x7c,0x8c846af3},

    {0xf0,0x5},
    {0x00,0xf3b18989}, {0x04,0x00000005}, {0x08,0x0000012c}, {0x0c,0x80808080}, {0x10,0x00000000}, {0x14,0x00000000}, {0x18,0x00010fff}, {0x1c,0x10000000},
    {0x20,0x10000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000400}, {0x30,0x00808080}, {0x34,0x80808080}, {0x38,0x80808080}, {0x3c,0x80808080},
    {0x40,0x80808080}, {0x44,0x80808080}, {0x48,0x80808080}, {0x4c,0x80808080}, {0x50,0x00000000}, {0x54,0x00010202}, {0x58,0x03040505}, {0x5c,0x06070808},
    {0x60,0x090a0b0c}, {0x64,0x0d0e0f10}, {0x68,0x11121314}, {0x6c,0x15161819}, {0x70,0x1a1b1d1e}, {0x74,0x00000001}, {0x78,0x0000000f}, {0x7c,0x0000000a},

    {0xf0,0x6},
    {0x00,0x0000000f}, {0x04,0x00000000}, {0x08,0x0000000a}, {0x0c,0x00000000}, {0x10,0x00000032}, {0x14,0x00000014}, {0x18,0x00000000}, {0x1c,0x00000001},
    {0x20,0x00002904}, {0x24,0x000001e0}, {0x28,0x00000320}, {0x2c,0xf8010009}, {0x30,0xf8010009}, {0x34,0x00000004}, {0x38,0x00000003}, {0x3c,0x00010fff},
    {0x40,0x80000000}, {0x44,0x00160016}, {0x48,0x00000fff}, {0x4c,0x00000003}, {0x50,0x00020001}, {0x54,0x00000064}, {0x58,0x00001000}, {0x5c,0x09249248},
    {0x60,0x00000000}, {0x64,0x000007d0}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x000001c2}, {0x78,0x00000064}, {0x7c,0x00000000},

    {0xf0,0x7},
    {0x00,0x04010700}, {0x04,0x06030902}, {0x08,0x0805040a}, {0x0c,0x07110610}, {0x10,0x09130812}, {0x14,0x00543216}, {0x18,0x007890ab}, {0x1c,0x00321094},
    {0x20,0x005678ab}, {0x24,0xff080010}, {0x28,0xff080120}, {0x2c,0xff080140}, {0x30,0xff080160}, {0x34,0x000000fa}, {0x38,0x000000d8}, {0x3c,0x000000b7},
    {0x40,0x00000014}, {0x44,0x00000100}, {0x48,0x00000000}, {0x4c,0x00000004}, {0x50,0x00000000}, {0x54,0x00000001}, {0x58,0x000e0000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00080002}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00432105}, {0x7c,0x006789ab},

    {0xf0,0x8},
    {0x00,0x026f028f}, {0x04,0x02af02cf}, {0x08,0x02ef030f}, {0x0c,0x032f034f}, {0x10,0x01f301f4}, {0x14,0x01f501f6}, {0x18,0x01f701f8}, {0x1c,0x11f901fa},
    {0x20,0x022f024f}, {0x24,0x036f01f0}, {0x28,0x01f101f2}, {0x2c,0x020f0000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x000043ef},
    {0x40,0x02040608}, {0x44,0x0a000000}, {0x48,0x00000000}, {0x4c,0x01030507}, {0x50,0x09000000}, {0x54,0x00000000}, {0x58,0x00c800aa}, {0x5c,0x00000008},
    {0x60,0x00000118}, {0x64,0x00000201}, {0x68,0x00000804}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x0000000a},

    {0xf0,0x9},
    {0x00,0xff080094}, {0x04,0x00070011}, {0x08,0xff080090}, {0x0c,0x00040000}, {0x10,0xfffffff0}, {0x14,0x00000000}, {0x18,0xfffffff0}, {0x1c,0x00000000},
    {0x20,0xfffffff0}, {0x24,0x00000000}, {0x28,0xfffffff0}, {0x2c,0x00000000}, {0x30,0xfffffff0}, {0x34,0x00000000}, {0x38,0xfffffff0}, {0x3c,0x00000000},
    {0x40,0xfffffff0}, {0x44,0x00000000}, {0x48,0xfffffff0}, {0x4c,0x00000000}, {0x50,0xfffffff0}, {0x54,0x00000000}, {0x58,0xfffffff0}, {0x5c,0x00000000},
    {0x60,0xfffffff0}, {0x64,0x00000000}, {0x68,0xfffffff0}, {0x6c,0x00000000}, {0x70,0xfffffff0}, {0x74,0x00000000}, {0x78,0xfffffff0}, {0x7c,0x00000000},

    {0xf0,0xe0},
    {0x00,0x006e002b}, {0x04,0x00000075}, {0x08,0x005c0088}, {0x0c,0x009a0011}, {0x10,0x00ad0007}, {0x14,0x0024000c}, {0x18,0x001500e9}, {0x1c,0x003f0084},
    {0x20,0x00bc0021}, {0x24,0x003c0079}, {0x28,0x007d0064}, {0x2c,0x006200b6}, {0x30,0x00d30001}, {0x34,0x0000011e}, {0x38,0x0135003c}, {0x3c,0x00730086},
    {0x40,0x006401f4}, {0x44,0x00640064}, {0x48,0x01900064}, {0x4c,0x00500190}, {0x50,0x00500050}, {0x54,0x012c0050}, {0x58,0x012c012c}, {0x5c,0x0032012c},
    {0x60,0x00640000}, {0x64,0x00640064}, {0x68,0x00000032}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},

    {0xf0,0xe1},
    {0x00,0x00810028}, {0x04,0x00000068}, {0x08,0x00590071}, {0x0c,0x00a80014}, {0x10,0x00aa0000}, {0x14,0x0029000a}, {0x18,0x002000bc}, {0x1c,0x003e0079},
    {0x20,0x00a70025}, {0x24,0x00330071}, {0x28,0x00720062}, {0x2c,0x008300ae}, {0x30,0x00b50000}, {0x34,0x00000110}, {0x38,0x012c0034}, {0x3c,0x005d0090},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},

    {0xf0,0x0},
    {0x00,0x01000000}, {0x04,0x01000000}, {0x08,0x01000000}, {0x0c,0x233fc0c0}, {0x10,0xa2146004}, {0x14,0xa4102000}, {0x18,0xe4244000}, {0x1c,0x233fc0c0},
    {0x20,0xa2146010}, {0x24,0x2500003f}, {0x28,0xa414a3ff}, {0x2c,0xe4244000}, {0x30,0x01000000}, {0x34,0x821020e0}, {0x38,0x81880001}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x270010c0}, {0x4c,0xa614e00f}, {0x50,0xe6a00040}, {0x54,0x01000000}, {0x58,0xa410200f}, {0x5c,0xe4a00040},
    {0x60,0x01000000}, {0x64,0xa0100000}, {0x68,0xa2100000}, {0x6c,0xa4100000}, {0x70,0xa6100000}, {0x74,0xa8100000}, {0x78,0xaa100000}, {0x7c,0xac100000},

    {0xf0,0x1},
    {0x00,0xae100000}, {0x04,0x90100000}, {0x08,0x92100000}, {0x0c,0x94100000}, {0x10,0x96100000}, {0x14,0x98100000}, {0x18,0x9a100000}, {0x1c,0x9c100000},
    {0x20,0x9e100000}, {0x24,0x84100000}, {0x28,0x86100000}, {0x2c,0x88100000}, {0x30,0x8a100000}, {0x34,0x8c100000}, {0x38,0x8e100000}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x82100000}, {0x4c,0x81900001}, {0x50,0x82100000}, {0x54,0x81980001}, {0x58,0x81800000}, {0x5c,0x01000000},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0xbc102cf8}, {0x6c,0x9c102c78}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x01000000}, {0x7c,0x01000000},

    {0xf0,0x2},
    {0x00,0x270010c0}, {0x04,0xa614e00f}, {0x08,0xe6a00040}, {0x0c,0x01000000}, {0x10,0x40000451}, {0x14,0x01000000}, {0x18,0x01000000}, {0x1c,0x10bfffff},
    {0x20,0x01000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},

    {0xf0,0x1a},
    {0x00,0x0000000e}, {0x04,0xfffffe65}, {0x08,0x000003fc}, {0x0c,0x00000af6}, {0x10,0x000003d4}, {0x14,0xfffffe64}, {0x18,0x00000008}, {0x1c,0xfffffe66},
    {0x20,0x00000425}, {0x24,0x00000af5}, {0x28,0x000003ac}, {0x2c,0xfffffe65}, {0x30,0x00000003}, {0x34,0xfffffe67}, {0x38,0x0000044e}, {0x3c,0x00000af3},
    {0x40,0x00000384}, {0x44,0xfffffe65}, {0x48,0xfffffffd}, {0x4c,0xfffffe69}, {0x50,0x00000476}, {0x54,0x00000aef}, {0x58,0x0000035c}, {0x5c,0xfffffe67},
    {0x60,0xfffffff7}, {0x64,0xfffffe6c}, {0x68,0x0000049f}, {0x6c,0x00000aea}, {0x70,0x00000335}, {0x74,0xfffffe68}, {0x78,0xfffffff1}, {0x7c,0xfffffe6f},

    {0xf0,0x1b},
    {0x00,0x000004c9}, {0x04,0x00000ae5}, {0x08,0x0000030e}, {0x0c,0xfffffe6a}, {0x10,0xffffffeb}, {0x14,0xfffffe73}, {0x18,0x000004f2}, {0x1c,0x00000ade},
    {0x20,0x000002e7}, {0x24,0xfffffe6d}, {0x28,0xffffffe4}, {0x2c,0xfffffe78}, {0x30,0x0000051b}, {0x34,0x00000ad5}, {0x38,0x000002c1}, {0x3c,0xfffffe70},
    {0x40,0xffffffde}, {0x44,0xfffffe7d}, {0x48,0x00000544}, {0x4c,0x00000acc}, {0x50,0x0000029c}, {0x54,0xfffffe74}, {0x58,0xffffffd7}, {0x5c,0xfffffe83},
    {0x60,0x0000056d}, {0x64,0x00000ac2}, {0x68,0x00000276}, {0x6c,0xfffffe78}, {0x70,0xffffffd0}, {0x74,0xfffffe89}, {0x78,0x00000597}, {0x7c,0x00000ab6},

    {0xf0,0x1c},
    {0x00,0x00000251}, {0x04,0xfffffe7c}, {0x08,0xffffffc8}, {0x0c,0xfffffe91}, {0x10,0x000005c0}, {0x14,0x00000aa9}, {0x18,0x0000022d}, {0x1c,0xfffffe81},
    {0x20,0xffffffc1}, {0x24,0xfffffe99}, {0x28,0x000005e9}, {0x2c,0x00000a9b}, {0x30,0x00000209}, {0x34,0xfffffe86}, {0x38,0xffffffb9}, {0x3c,0xfffffea1},
    {0x40,0x00000611}, {0x44,0x00000a8d}, {0x48,0x000001e5}, {0x4c,0xfffffe8b}, {0x50,0xffffffb2}, {0x54,0xfffffeab}, {0x58,0x0000063a}, {0x5c,0x00000a7d},
    {0x60,0x000001c3}, {0x64,0xfffffe91}, {0x68,0xffffffaa}, {0x6c,0xfffffeb5}, {0x70,0x00000663}, {0x74,0x00000a6b}, {0x78,0x000001a0}, {0x7c,0xfffffe97},

    {0xf0,0x1d},
    {0x00,0xffffffa2}, {0x04,0xfffffebf}, {0x08,0x0000068b}, {0x0c,0x00000a59}, {0x10,0x0000017e}, {0x14,0xfffffe9d}, {0x18,0xffffff9a}, {0x1c,0xfffffecb},
    {0x20,0x000006b3}, {0x24,0x00000a46}, {0x28,0x0000015d}, {0x2c,0xfffffea4}, {0x30,0xffffff91}, {0x34,0xfffffed7}, {0x38,0x000006da}, {0x3c,0x00000a32},
    {0x40,0x0000013d}, {0x44,0xfffffeab}, {0x48,0xffffff89}, {0x4c,0xfffffee4}, {0x50,0x00000702}, {0x54,0x00000a1d}, {0x58,0x0000011d}, {0x5c,0xfffffeb2},
    {0x60,0xffffff80}, {0x64,0xfffffef2}, {0x68,0x00000729}, {0x6c,0x00000a06}, {0x70,0x000000fd}, {0x74,0xfffffeba}, {0x78,0xffffff78}, {0x7c,0xffffff00},

    {0xf0,0x1e},
    {0x00,0x0000074f}, {0x04,0x000009ef}, {0x08,0x000000df}, {0x0c,0xfffffec1}, {0x10,0xffffff6f}, {0x14,0xffffff10}, {0x18,0x00000776}, {0x1c,0x000009d7},
    {0x20,0x000000c1}, {0x24,0xfffffec9}, {0x28,0xffffff66}, {0x2c,0xffffff20}, {0x30,0x0000079b}, {0x34,0x000009be}, {0x38,0x000000a3}, {0x3c,0xfffffed1},
    {0x40,0xffffff5e}, {0x44,0xffffff30}, {0x48,0x000007c1}, {0x4c,0x000009a4}, {0x50,0x00000087}, {0x54,0xfffffed9}, {0x58,0xffffff55}, {0x5c,0xffffff42},
    {0x60,0x000007e5}, {0x64,0x00000989}, {0x68,0x0000006b}, {0x6c,0xfffffee2}, {0x70,0xffffff4c}, {0x74,0xffffff54}, {0x78,0x0000080a}, {0x7c,0x0000096d},

    {0xf0,0x1f},
    {0x00,0x0000004f}, {0x04,0xfffffeea}, {0x08,0xffffff43}, {0x0c,0xffffff67}, {0x10,0x0000082d}, {0x14,0x00000951}, {0x18,0x00000035}, {0x1c,0xfffffef3},
    {0x20,0xffffff3a}, {0x24,0xffffff7b}, {0x28,0x00000850}, {0x2c,0x00000933}, {0x30,0x0000001b}, {0x34,0xfffffefb}, {0x38,0xffffff31}, {0x3c,0xffffff90},
    {0x40,0x00000873}, {0x44,0x00000915}, {0x48,0x00000002}, {0x4c,0xffffff04}, {0x50,0xffffff28}, {0x54,0xffffffa5}, {0x58,0x00000895}, {0x5c,0x000008f6},
    {0x60,0xffffffea}, {0x64,0xffffff0d}, {0x68,0xffffff1f}, {0x6c,0xffffffbb}, {0x70,0x000008b6}, {0x74,0x000008d6}, {0x78,0xffffffd2}, {0x7c,0xffffff16},

    {0xf0,0x20},
    {0x00,0x83580000}, {0x04,0x82086ff0}, {0x08,0x83306004}, {0x0c,0x80a06005}, {0x10,0x02800024}, {0x14,0x01000000}, {0x18,0x80a06006}, {0x1c,0x02800039},
    {0x20,0x01000000}, {0x24,0x80a06015}, {0x28,0x02800051}, {0x2c,0x01000000}, {0x30,0x80a0602a}, {0x34,0x02800085}, {0x38,0x01000000}, {0x3c,0x073fc180},
    {0x40,0x8610e03c}, {0x44,0x05169680}, {0x48,0x84004002}, {0x4c,0xc420c000}, {0x50,0x073fc000}, {0x54,0x8610e020}, {0x58,0x84102001}, {0x5c,0xc420c000},
    {0x60,0x0500000c}, {0x64,0x01000000}, {0x68,0x01000000}, {0x6c,0x8480bfff}, {0x70,0x12bffffe}, {0x74,0x01000000}, {0x78,0x01000000}, {0x7c,0x073fc000},

    {0xf0,0x21},
    {0x00,0x8610e020}, {0x04,0x84102000}, {0x08,0xc420c000}, {0x0c,0x01000000}, {0x10,0x01000000}, {0x14,0x81c44000}, {0x18,0x81cc8000}, {0x1c,0x01000000},
    {0x20,0xa7500000}, {0x24,0xa92ce002}, {0x28,0xa734e001}, {0x2c,0xa614c014}, {0x30,0xa60ce007}, {0x34,0x81900000}, {0x38,0x01000000}, {0x3c,0x01000000},
    {0x40,0x81e00000}, {0x44,0xe03ba000}, {0x48,0xe43ba008}, {0x4c,0xe83ba010}, {0x50,0xec3ba018}, {0x54,0xf03ba020}, {0x58,0xf43ba028}, {0x5c,0xf83ba030},
    {0x60,0xfc3ba038}, {0x64,0x81e80000}, {0x68,0x8194c000}, {0x6c,0x01000000}, {0x70,0x01000000}, {0x74,0x81c44000}, {0x78,0x81cc8000}, {0x7c,0x01000000},

    {0xf0,0x22},
    {0x00,0xa7500000}, {0x04,0xa934e002}, {0x08,0xa72ce001}, {0x0c,0xa614c014}, {0x10,0xa60ce007}, {0x14,0x81900000}, {0x18,0x01000000}, {0x1c,0x01000000},
    {0x20,0x81e80000}, {0x24,0x81e80000}, {0x28,0xe01ba000}, {0x2c,0xe41ba008}, {0x30,0xe81ba010}, {0x34,0xec1ba018}, {0x38,0xf01ba020}, {0x3c,0xf41ba028},
    {0x40,0xf81ba030}, {0x44,0xfc1ba038}, {0x48,0x81e00000}, {0x4c,0x81e00000}, {0x50,0x8194c000}, {0x54,0x01000000}, {0x58,0x01000000}, {0x5c,0x81c44000},
    {0x60,0x81cc8000}, {0x64,0x01000000}, {0x68,0x01000000}, {0x6c,0x82102010}, {0x70,0x273fc0c0}, {0x74,0xa614e010}, {0x78,0xc224c000}, {0x7c,0x01000000},

    {0xf0,0x23},
    {0x00,0x033fc0c0}, {0x04,0x82106004}, {0x08,0xa6102000}, {0x0c,0xe6204000}, {0x10,0x01000000}, {0x14,0x01000000}, {0x18,0x01000000}, {0x1c,0xa6102020},
    {0x20,0x83480000}, {0x24,0x82104013}, {0x28,0x81884000}, {0x2c,0x01000000}, {0x30,0x400011a1}, {0x34,0x01000000}, {0x38,0x01000000}, {0x3c,0x01000000},
    {0x40,0xa7500000}, {0x44,0xa934e002}, {0x48,0xa72ce001}, {0x4c,0xa614c014}, {0x50,0xa60ce007}, {0x54,0x81900000}, {0x58,0x01000000}, {0x5c,0x81e80000},
    {0x60,0xe01ba000}, {0x64,0xe41ba008}, {0x68,0xe81ba010}, {0x6c,0xec1ba018}, {0x70,0xf01ba020}, {0x74,0xf41ba028}, {0x78,0xf81ba030}, {0x7c,0xfc1ba038},

    {0xf0,0x24},
    {0x00,0x81e00000}, {0x04,0x8194c000}, {0x08,0x01000000}, {0x0c,0xa6102020}, {0x10,0x83480000}, {0x14,0x82284013}, {0x18,0x81884000}, {0x1c,0x01000000},
    {0x20,0x033fc0c0}, {0x24,0x82106004}, {0x28,0xa6103fff}, {0x2c,0xe6204000}, {0x30,0x01000000}, {0x34,0x01000000}, {0x38,0x01000000}, {0x3c,0x81c44000},
    {0x40,0x81cc8000}, {0x44,0x01000000}, {0x48,0x81c48000}, {0x4c,0x81cca004}, {0x50,0x01000000}, {0x54,0x9de3bf98}, {0x58,0x4000001b}, {0x5c,0x01000000},
    {0x60,0x40000012}, {0x64,0x01000000}, {0x68,0x400000ee}, {0x6c,0x01000000}, {0x70,0x40000040}, {0x74,0x01000000}, {0x78,0x400000a4}, {0x7c,0x01000000},

    {0xf0,0x25},
    {0x00,0x30bffffe}, {0x04,0x80a22000}, {0x08,0x02800006}, {0x0c,0x01000000}, {0x10,0x01000000}, {0x14,0x90823fff}, {0x18,0x12bffffe}, {0x1c,0x01000000},
    {0x20,0x81c3e008}, {0x24,0x01000000}, {0x28,0x82102001}, {0x2c,0x81904000}, {0x30,0x01000000}, {0x34,0x01000000}, {0x38,0x01000000}, {0x3c,0x81c3e008},
    {0x40,0x01000000}, {0x44,0x03000008}, {0x48,0x82106342}, {0x4c,0xa3804000}, {0x50,0x03000004}, {0x54,0x82106000}, {0x58,0x81984000}, {0x5c,0x01000000},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x81c3e008}, {0x6c,0x01000000}, {0x70,0x98102000}, {0x74,0x832b2002}, {0x78,0xda006480}, {0x7c,0x80a37ff0},

    {0xf0,0x26},
    {0x00,0x02800006}, {0x04,0x98032002}, {0x08,0xc2006484}, {0x0c,0x80a3201f}, {0x10,0x04bffff9}, {0x14,0xc2234000}, {0x18,0x81c3e008}, {0x1c,0x01000000},
    {0x20,0x03004040}, {0x24,0x94106101}, {0x28,0x98102000}, {0x2c,0x832b2002}, {0x30,0xd60063a4}, {0x34,0x9a102000}, {0x38,0x832b6002}, {0x3c,0x9a036001},
    {0x40,0x80a36004}, {0x44,0x04bffffd}, {0x48,0xd422c001}, {0x4c,0x98032001}, {0x50,0x80a32003}, {0x54,0x04bffff7}, {0x58,0x832b2002}, {0x5c,0x033fc200},
    {0x60,0xda002330}, {0x64,0x82106074}, {0x68,0x81c3e008}, {0x6c,0xda204000}, {0x70,0x9de3bf98}, {0x74,0x40000f98}, {0x78,0x90102000}, {0x7c,0x213fc140},

    {0xf0,0x27},
    {0x00,0xda00247c}, {0x04,0x98142040}, {0x08,0xea030000}, {0x0c,0xc20022f8}, {0x10,0x9b336001}, {0x14,0x825b4001}, {0x18,0xaa0d7c00}, {0x1c,0xaa154001},
    {0x20,0xea230000}, {0x24,0x82142004}, {0x28,0xea004000}, {0x2c,0xaa0d7ff0}, {0x30,0xaa15400d}, {0x34,0xea204000}, {0x38,0x2d3fc200}, {0x3c,0x8215a080},
    {0x40,0xea004000}, {0x44,0xaa0d7ff0}, {0x48,0xaa15400d}, {0x4c,0xea204000}, {0x50,0xc200233c}, {0x54,0x9a15a070}, {0x58,0xc2234000}, {0x5c,0x19000016},
    {0x60,0x033fc000}, {0x64,0xda002338}, {0x68,0xa21323a8}, {0x6c,0x82106030}, {0x70,0xda204000}, {0x74,0x98132180}, {0x78,0x96142088}, {0x7c,0xd822c000},

    {0xf0,0x28},
    {0x00,0x9414208c}, {0x04,0x0300003f}, {0x08,0xe2228000}, {0x0c,0x92142058}, {0x10,0x821063ff}, {0x14,0xc2224000}, {0x18,0xc20023f8}, {0x1c,0x9015a00c},
    {0x20,0xc2220000}, {0x24,0xc20023fc}, {0x28,0x9e15a008}, {0x2c,0xc223c000}, {0x30,0xa6142080}, {0x34,0xd824c000}, {0x38,0xa8142084}, {0x3c,0xa414205c},
    {0x40,0xe2250000}, {0x44,0x7fffffb7}, {0x48,0xc0248000}, {0x4c,0x400001fb}, {0x50,0xa415a030}, {0x54,0x9a15a07c}, {0x58,0xea034000}, {0x5c,0x033ff000},
    {0x60,0xd8002374}, {0x64,0xaa2d4001}, {0x68,0xea234000}, {0x6c,0x033fc1c0}, {0x70,0xda002340}, {0x74,0x82106064}, {0x78,0xda204000}, {0x7c,0x0300007f},

    {0xf0,0x29},
    {0x00,0x92142010}, {0x04,0x821063ff}, {0x08,0x1507ffc0}, {0x0c,0xc2224000}, {0x10,0x9e142030}, {0x14,0x96032001}, {0x18,0xd423c000}, {0x1c,0x972ae010},
    {0x20,0xa0142014}, {0x24,0x9602c00c}, {0x28,0xa32b2010}, {0x2c,0x912b2004}, {0x30,0xd4240000}, {0x34,0x80a32000}, {0x38,0x82044008}, {0x3c,0x9602e002},
    {0x40,0x9a15a084}, {0x44,0x9815a088}, {0x48,0x02800005}, {0x4c,0x9415a08c}, {0x50,0xc2234000}, {0x54,0xe2230000}, {0x58,0xd6228000}, {0x5c,0xc2002344},
    {0x60,0xc2248000}, {0x64,0x033fc0c0}, {0x68,0x82106004}, {0x6c,0x9a103fff}, {0x70,0x7fffff80}, {0x74,0xda204000}, {0x78,0x03200040}, {0x7c,0xc2258000},

    {0xf0,0x2a},
    {0x00,0x81c7e008}, {0x04,0x81e80000}, {0x08,0x01000000}, {0x0c,0x01000000}, {0x10,0x01000000}, {0x14,0xa7800000}, {0x18,0x01000000}, {0x1c,0x01000000},
    {0x20,0x01000000}, {0x24,0x81c3e008}, {0x28,0x01000000}, {0x2c,0x9de3bf98}, {0x30,0xb6102000}, {0x34,0xb0102000}, {0x38,0xb8102000}, {0x3c,0xc2070000},
    {0x40,0xb8072004}, {0x44,0x80a724ff}, {0x48,0x08bffffd}, {0x4c,0xb606c001}, {0x50,0x03000016}, {0x54,0x821061e0}, {0x58,0x82087f80}, {0x5c,0xb8102d00},
    {0x60,0x80a70001}, {0x64,0x3a80001e}, {0x68,0xfa002180}, {0x6c,0xb4100001}, {0x70,0x9a102001}, {0x74,0x9e100001}, {0x78,0xc2070000}, {0x7c,0xb8072004},

    {0xf0,0x2b},
    {0x00,0xb21f001a}, {0x04,0xbb37200c}, {0x08,0x808f2fff}, {0x0c,0x02800005}, {0x10,0xb606c001}, {0x14,0x80a7001a}, {0x18,0x1280000e}, {0x1c,0x80a7000f},
    {0x20,0x80a00019}, {0x24,0xba677fff}, {0x28,0x832f6002}, {0x2c,0xc2006180}, {0x30,0xb606c001}, {0x34,0xba077fff}, {0x38,0x80a6e000}, {0x3c,0x832b401d},
    {0x40,0x12800003}, {0x44,0xb6102000}, {0x48,0xb0160001}, {0x4c,0x80a7000f}, {0x50,0x2abfffeb}, {0x54,0xc2070000}, {0x58,0xfa002180}, {0x5c,0xb816001d},
    {0x60,0x821e001d}, {0x64,0x80a70001}, {0x68,0x32800009}, {0x6c,0xba16001d}, {0x70,0x0329697f}, {0x74,0x821063ff}, {0x78,0x80a70001}, {0x7c,0x32800004},

    {0xf0,0x2c},
    {0x00,0xba16001d}, {0x04,0x3b169696}, {0x08,0xba17625a}, {0x0c,0x033fc180}, {0x10,0x82106030}, {0x14,0xfa204000}, {0x18,0x81c7e008}, {0x1c,0x91e82001},
    {0x20,0x033fc180}, {0x24,0xc0204000}, {0x28,0x82102500}, {0x2c,0xc0204000}, {0x30,0x82006004}, {0x34,0x80a0687c}, {0x38,0x28bffffe}, {0x3c,0xc0204000},
    {0x40,0x033fc200}, {0x44,0x82106030}, {0x48,0xda004000}, {0x4c,0x82102010}, {0x50,0xc2202574}, {0x54,0x82102001}, {0x58,0xc2202540}, {0x5c,0x8210200f},
    {0x60,0xc2202548}, {0x64,0x81c3e008}, {0x68,0xda20257c}, {0x6c,0x9de3bf98}, {0x70,0x82102000}, {0x74,0x80a04019}, {0x78,0x16800015}, {0x7c,0x9e100019},

    {0xf0,0x2d},
    {0x00,0xb6006001}, {0x04,0x80a6c00f}, {0x08,0x1680000f}, {0x0c,0xba10001b}, {0x10,0xb3286002}, {0x14,0xb52f6002}, {0x18,0xf8060019}, {0x1c,0xc206001a},
    {0x20,0x80a70001}, {0x24,0x04800004}, {0x28,0xba076001}, {0x2c,0xc2260019}, {0x30,0xf826001a}, {0x34,0x80a7400f}, {0x38,0x06bffff8}, {0x3c,0xb52f6002},
    {0x40,0x80a6c00f}, {0x44,0x06bfffef}, {0x48,0x8210001b}, {0x4c,0x81c7e008}, {0x50,0x81e80000}, {0x54,0x033fc140}, {0x58,0x82106048}, {0x5c,0xda004000},
    {0x60,0x03000040}, {0x64,0x808b4001}, {0x68,0x03000016}, {0x6c,0x12800003}, {0x70,0x90106180}, {0x74,0x901063a8}, {0x78,0x81c3e008}, {0x7c,0x01000000},

    {0xf0,0x2e},
    {0x00,0x9de3bf38}, {0x04,0xa12e2002}, {0x08,0x1b00003f}, {0x0c,0xc20423d8}, {0x10,0x9a1363ff}, {0x14,0xb008400d}, {0x18,0x97306010}, {0x1c,0xc200247c},
    {0x20,0x9a22c018}, {0x24,0x825e0001}, {0x28,0x92836001}, {0x2c,0x0280000c}, {0x30,0xb0004019}, {0x34,0x9a100009}, {0x38,0x9807bf98}, {0x3c,0x82060018},
    {0x40,0xc2168001}, {0x44,0xc2230000}, {0x48,0xc200247c}, {0x4c,0xb0060001}, {0x50,0x9a837fff}, {0x54,0x12bffffa}, {0x58,0x98032004}, {0x5c,0x7fffffc4},
    {0x60,0x9007bf98}, {0x64,0x0300003f}, {0x68,0xda0423e8}, {0x6c,0x821063ff}, {0x70,0xb00b4001}, {0x74,0x97336010}, {0x78,0x80a6000b}, {0x7c,0x92102000},

    {0xf0,0x2f},
    {0x00,0x1880000b}, {0x04,0x9a100018}, {0x08,0x832e2002}, {0x0c,0x8200401e}, {0x10,0x98007f98}, {0x14,0xc2030000}, {0x18,0x9a036001}, {0x1c,0x92024001},
    {0x20,0x80a3400b}, {0x24,0x08bffffc}, {0x28,0x98032004}, {0x2c,0xb022c018}, {0x30,0xb0062001}, {0x34,0x81800000}, {0x38,0x01000000}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0xb0724018}, {0x48,0x81c7e008}, {0x4c,0x81e80000}, {0x50,0x832a2002}, {0x54,0x82004008}, {0x58,0x9b326002}, {0x5c,0x8200400d},
    {0x60,0x83286002}, {0x64,0x920a6003}, {0x68,0x932a6003}, {0x6c,0xd00065b0}, {0x70,0x91320009}, {0x74,0x81c3e008}, {0x78,0x900a20ff}, {0x7c,0x972a2002},

    {0xf0,0x30},
    {0x00,0x99326002}, {0x04,0x9002c008}, {0x08,0x9002000c}, {0x0c,0x920a6003}, {0x10,0x932a6003}, {0x14,0x912a2002}, {0x18,0x821020ff}, {0x1c,0xda0225b0},
    {0x20,0x83284009}, {0x24,0x822b4001}, {0x28,0x952a8009}, {0x2c,0x8210400a}, {0x30,0xc22225b0}, {0x34,0xda02e3a4}, {0x38,0x992b2002}, {0x3c,0x81c3e008},
    {0x40,0xc223400c}, {0x44,0x9de3bf98}, {0x48,0xda002310}, {0x4c,0x80a36000}, {0x50,0x02800049}, {0x54,0xb0102000}, {0x58,0xc2002594}, {0x5c,0x82006001},
    {0x60,0x80a0400d}, {0x64,0x0a800044}, {0x68,0xc2202594}, {0x6c,0xa4102000}, {0x70,0xc20023d4}, {0x74,0x80a48001}, {0x78,0xc0202594}, {0x7c,0xa2102000},

    {0xf0,0x31},
    {0x00,0x1a800028}, {0x04,0xa72c6002}, {0x08,0xc204e364}, {0x0c,0x80a06000}, {0x10,0x02800020}, {0x14,0xa0102000}, {0x18,0xc20022fc}, {0x1c,0x80a40001},
    {0x20,0x1a80001c}, {0x24,0x15000017}, {0x28,0xc200255c}, {0x2c,0xf00c2380}, {0x30,0x9412a1d0}, {0x34,0x90100011}, {0x38,0x80a06000}, {0x3c,0x02800007},
    {0x40,0x920e20ff}, {0x44,0x7fffff84}, {0x48,0x01000000}, {0x4c,0x94100008}, {0x50,0x90100011}, {0x54,0x920e20ff}, {0x58,0x7fffff8a}, {0x5c,0xa0042001},
    {0x60,0xc204e364}, {0x64,0xda002348}, {0x68,0x98020001}, {0x6c,0x82034001}, {0x70,0x80a20001}, {0x74,0x38bfffe9}, {0x78,0xa404a001}, {0x7c,0x80a3000d},

    {0xf0,0x32},
    {0x00,0x3abfffe7}, {0x04,0xc20022fc}, {0x08,0x10bfffe4}, {0x0c,0xa404a001}, {0x10,0xa2046001}, {0x14,0xc20023d4}, {0x18,0x10bfffda}, {0x1c,0x80a44001},
    {0x20,0xd800258c}, {0x24,0x80a0000c}, {0x28,0x9a603fff}, {0x2c,0x80a00012}, {0x30,0x82603fff}, {0x34,0x808b4001}, {0x38,0x02800007}, {0x3c,0x80a4a000},
    {0x40,0xc200255c}, {0x44,0x80a00001}, {0x48,0x82603fff}, {0x4c,0xc220255c}, {0x50,0x80a4a000}, {0x54,0x12800004}, {0x58,0x82032001}, {0x5c,0x10800003},
    {0x60,0xc020258c}, {0x64,0xc220258c}, {0x68,0xc200258c}, {0x6c,0x80a06003}, {0x70,0xb0603fff}, {0x74,0x81c7e008}, {0x78,0x81e80000}, {0x7c,0x9de3bf98},

    {0xf0,0x33},
    {0x00,0xc2002540}, {0x04,0x80a06000}, {0x08,0x0280002a}, {0x0c,0xb0102000}, {0x10,0xda002210}, {0x14,0x80a36000}, {0x18,0x02800026}, {0x1c,0xb4102001},
    {0x20,0xde0022f8}, {0x24,0x80a6800f}, {0x28,0x18800018}, {0x2c,0x03000018}, {0x30,0x98106220}, {0x34,0xf20022fc}, {0x38,0xb6102007}, {0x3c,0xb8102001},
    {0x40,0x80a70019}, {0x44,0x1880000d}, {0x48,0x832ee003}, {0x4c,0x8200400c}, {0x50,0xba006004}, {0x54,0xc2074000}, {0x58,0xb8072001}, {0x5c,0x80a0400d},
    {0x60,0x14800003}, {0x64,0xba076004}, {0x68,0xb0062001}, {0x6c,0x80a70019}, {0x70,0x28bffffa}, {0x74,0xc2074000}, {0x78,0xb406a001}, {0x7c,0x80a6800f},

    {0xf0,0x34},
    {0x00,0x08bfffef}, {0x04,0xb606e007}, {0x08,0xc21023ce}, {0x0c,0x80a60001}, {0x10,0x24800007}, {0x14,0xc0202598}, {0x18,0xc2002598}, {0x1c,0x82006001},
    {0x20,0xc2202598}, {0x24,0x10800003}, {0x28,0xb0102001}, {0x2c,0xb0102000}, {0x30,0x81c7e008}, {0x34,0x81e80000}, {0x38,0x9a102005}, {0x3c,0x8210200b},
    {0x40,0x9a234008}, {0x44,0x82204008}, {0x48,0x9b2b6002}, {0x4c,0x80a22005}, {0x50,0x14800007}, {0x54,0x99286002}, {0x58,0x033fc200}, {0x5c,0x8210600c},
    {0x60,0xc2004000}, {0x64,0x10800006}, {0x68,0x8330400d}, {0x6c,0x033fc200}, {0x70,0x82106008}, {0x74,0xc2004000}, {0x78,0x8330400c}, {0x7c,0x81c3e008},

    {0xf0,0x35},
    {0x00,0x9008600f}, {0x04,0x9de3bf98}, {0x08,0xc200247c}, {0x0c,0x83306001}, {0x10,0x80a60001}, {0x14,0x1a800006}, {0x18,0x90100018}, {0x1c,0x7fffffe7},
    {0x20,0x01000000}, {0x24,0x10800006}, {0x28,0xb0020008}, {0x2c,0x7fffffe3}, {0x30,0x90260001}, {0x34,0x90020008}, {0x38,0xb0022001}, {0x3c,0x81c7e008},
    {0x40,0x81e80000}, {0x44,0x9de3bf98}, {0x48,0xa8102000}, {0x4c,0xc20023d4}, {0x50,0x80a50001}, {0x54,0x1a800057}, {0x58,0xe2002348}, {0x5c,0xa4102000},
    {0x60,0xc200247c}, {0x64,0x80a48001}, {0x68,0x3a80004e}, {0x6c,0xa8052001}, {0x70,0x7fffffe5}, {0x74,0x90100012}, {0x78,0x92100008}, {0x7c,0x7fffff35},

    {0xf0,0x36},
    {0x00,0x90100014}, {0x04,0x80a62000}, {0x08,0x12800004}, {0x0c,0xa0100008}, {0x10,0x10800016}, {0x14,0xa0102000}, {0x18,0x80a62008}, {0x1c,0x18800011},
    {0x20,0x80a62007}, {0x24,0x7ffffeec}, {0x28,0x01000000}, {0x2c,0x94100008}, {0x30,0x90100014}, {0x34,0x7ffffef3}, {0x38,0x921ca001}, {0x3c,0x80a20011},
    {0x40,0x04800007}, {0x44,0xa6100008}, {0x48,0x9a102008}, {0x4c,0x9a234018}, {0x50,0x82102001}, {0x54,0x8328400d}, {0x58,0xa02c0001}, {0x5c,0x80a62007},
    {0x60,0x18800008}, {0x64,0x80a62008}, {0x68,0x9a102007}, {0x6c,0x9a234018}, {0x70,0x82102001}, {0x74,0x8328400d}, {0x78,0x10800022}, {0x7c,0xa0140001},

    {0xf0,0x37},
    {0x00,0x1280000a}, {0x04,0x821e2009}, {0x08,0x80a420fe}, {0x0c,0x24800002}, {0x10,0xa0042001}, {0x14,0x03000018}, {0x18,0x9b2ca002}, {0x1c,0x82106220},
    {0x20,0x10800018}, {0x24,0xe6234001}, {0x28,0x80a00001}, {0x2c,0x9a603fff}, {0x30,0x80a420fe}, {0x34,0x04800003}, {0x38,0x82102001}, {0x3c,0x82102000},
    {0x40,0x808b4001}, {0x44,0x0280000f}, {0x48,0x03000018}, {0x4c,0x9b2ca002}, {0x50,0x82106220}, {0x54,0xc2034001}, {0x58,0x80a04011}, {0x5c,0x18800003},
    {0x60,0x9a204011}, {0x64,0x9a244001}, {0x68,0x80a4c011}, {0x6c,0x14800003}, {0x70,0x8224c011}, {0x74,0x82244013}, {0x78,0x80a34001}, {0x7c,0xa0642000},

    {0xf0,0x38},
    {0x00,0x7fffffa1}, {0x04,0x90100012}, {0x08,0x92100008}, {0x0c,0x90100014}, {0x10,0x7ffffefb}, {0x14,0x94100010}, {0x18,0x10bfffb2}, {0x1c,0xa404a001},
    {0x20,0xc20023d4}, {0x24,0x80a50001}, {0x28,0x0abfffae}, {0x2c,0xa4102000}, {0x30,0x81c7e008}, {0x34,0x81e80000}, {0x38,0x033fc200}, {0x3c,0x961060a0},
    {0x40,0x98102000}, {0x44,0x832b2002}, {0x48,0x9a03000c}, {0x4c,0xda136400}, {0x50,0x98032001}, {0x54,0x80a32016}, {0x58,0x04bffffb}, {0x5c,0xda20400b},
    {0x60,0x81c3e008}, {0x64,0x01000000}, {0x68,0x9de3bf98}, {0x6c,0xc2002544}, {0x70,0x82006001}, {0x74,0xc2202544}, {0x78,0x03000017}, {0x7c,0xb41063f8},

    {0xf0,0x39},
    {0x00,0x9e100018}, {0x04,0x031fffdf}, {0x08,0xb01063ff}, {0x0c,0xba102000}, {0x10,0xb72f6002}, {0x14,0xc2002544}, {0x18,0x80a06009}, {0x1c,0xb2076001},
    {0x20,0x12800007}, {0x24,0xb810001b}, {0x28,0xc206c01a}, {0x2c,0x83306001}, {0x30,0x82084018}, {0x34,0xc226c01a}, {0x38,0xc2002544}, {0x3c,0x80a06008},
    {0x40,0x08800006}, {0x44,0xc207001a}, {0x48,0xfa03c01c}, {0x4c,0xbb376001}, {0x50,0x10800003}, {0x54,0xba0f4018}, {0x58,0xfa03c01c}, {0x5c,0x8200401d},
    {0x60,0xc227001a}, {0x64,0x80a66089}, {0x68,0x08bfffea}, {0x6c,0xba100019}, {0x70,0x81c7e008}, {0x74,0x81e80000}, {0x78,0x9de3bf98}, {0x7c,0x9e102001},

    {0xf0,0x3a},
    {0x00,0xc20022fc}, {0x04,0x80a3c001}, {0x08,0x1880002a}, {0x0c,0x03000018}, {0x10,0x82106220}, {0x14,0x9a006004}, {0x18,0x19000017}, {0x1c,0xc20022f8},
    {0x20,0xb6102001}, {0x24,0x80a6c001}, {0x28,0xb21323f8}, {0x2c,0xb41321d0}, {0x30,0x1880001b}, {0x34,0xc20be37f}, {0x38,0xb0004001}, {0x3c,0xb8036038},
    {0x40,0xc2002544}, {0x44,0xb606e001}, {0x48,0x80a06008}, {0x4c,0x08800003}, {0x50,0xfa164018}, {0x54,0xba07401d}, {0x58,0x81800000}, {0x5c,0xc2002548},
    {0x60,0x01000000}, {0x64,0x01000000}, {0x68,0x82774001}, {0x6c,0xba100001}, {0x70,0xc2168018}, {0x74,0xba274001}, {0x78,0xfa270000}, {0x7c,0xc200247c},

    {0xf0,0x3b},
    {0x00,0x82004001}, {0x04,0xfa0022f8}, {0x08,0xb4068001}, {0x0c,0x80a6c01d}, {0x10,0xb2064001}, {0x14,0x08bfffeb}, {0x18,0xb8072038}, {0x1c,0x9e03e001},
    {0x20,0xc20022fc}, {0x24,0x80a3c001}, {0x28,0x08bfffdd}, {0x2c,0x9a036004}, {0x30,0x81c7e008}, {0x34,0x81e80000}, {0x38,0xc2002540}, {0x3c,0x80a06000},
    {0x40,0x0280000f}, {0x44,0x1b3fc200}, {0x48,0xc2002298}, {0x4c,0x9a136070}, {0x50,0xc2234000}, {0x54,0x03000017}, {0x58,0xc0202540}, {0x5c,0xc0202544},
    {0x60,0x981063f8}, {0x64,0x9a102000}, {0x68,0x832b6002}, {0x6c,0x9a036001}, {0x70,0x80a36089}, {0x74,0x08bffffd}, {0x78,0xc020400c}, {0x7c,0x81c3e008},

    {0xf0,0x3c},
    {0x00,0x01000000}, {0x04,0xc200247c}, {0x08,0xda0022f8}, {0x0c,0x8258400d}, {0x10,0x97306001}, {0x14,0x98102000}, {0x18,0x80a3000b}, {0x1c,0x1680000e},
    {0x20,0x1b000017}, {0x24,0x0307ffc7}, {0x28,0x901363f8}, {0x2c,0x921063ff}, {0x30,0x941361d0}, {0x34,0x9b2b2002}, {0x38,0xc2034008}, {0x3c,0x83306003},
    {0x40,0x82084009}, {0x44,0x98032001}, {0x48,0x80a3000b}, {0x4c,0x06bffffa}, {0x50,0xc223400a}, {0x54,0x03000018}, {0x58,0x9a106220}, {0x5c,0x98102000},
    {0x60,0x832b2002}, {0x64,0x98032001}, {0x68,0x80a322d5}, {0x6c,0x04bffffd}, {0x70,0xc020400d}, {0x74,0x81c3e008}, {0x78,0x01000000}, {0x7c,0x00000000},

    {0xf0,0x3d},
    {0x00,0x82102020}, {0x04,0x82204009}, {0x08,0x80a06040}, {0x0c,0x04800003}, {0x10,0x9a100008}, {0x14,0x90023fff}, {0x18,0x80a06080}, {0x1c,0x34800002},
    {0x20,0x90037ffe}, {0x24,0x80a06000}, {0x28,0x24800002}, {0x2c,0x90036001}, {0x30,0x80a07fc0}, {0x34,0x24800002}, {0x38,0x90036002}, {0x3c,0x81c3e008},
    {0x40,0x01000000}, {0x44,0x900221ff}, {0x48,0x833a201f}, {0x4c,0x8330601a}, {0x50,0x82020001}, {0x54,0x82087fc0}, {0x58,0x90220001}, {0x5c,0x81c3e008},
    {0x60,0x90022001}, {0x64,0x9de3bf80}, {0x68,0x90102020}, {0x6c,0x7ffffff6}, {0x70,0x90220018}, {0x74,0x82102041}, {0x78,0x82204008}, {0x7c,0x9b2ea003},

    {0xf0,0x3e},
    {0x00,0x98004001}, {0x04,0x9a23401a}, {0x08,0x98030001}, {0x0c,0x9a03400d}, {0x10,0x9a03401b}, {0x14,0x03000018}, {0x18,0x82106220}, {0x1c,0x9b2b6002},
    {0x20,0x9a034001}, {0x24,0xc2002300}, {0x28,0x96020008}, {0x2c,0x9602c008}, {0x30,0xaa006001}, {0x34,0xc2002308}, {0x38,0xa52ae003}, {0x3c,0xa8006001},
    {0x40,0xa72b2003}, {0x44,0x96037ff8}, {0x48,0xa0103ffe}, {0x4c,0xb0102000}, {0x50,0x94103ffe}, {0x54,0xa206c010}, {0x58,0x9804ecfc}, {0x5c,0x9e04ace8},
    {0x60,0x9202ff90}, {0x64,0x8206800a}, {0x68,0x80a54001}, {0x6c,0x9a603fff}, {0x70,0x80a50011}, {0x74,0x82603fff}, {0x78,0x808b4001}, {0x7c,0x02800003},

    {0xf0,0x3f},
    {0x00,0x9a102000}, {0x04,0xda024000}, {0x08,0x80a22020}, {0x0c,0x34800003}, {0x10,0xc2030000}, {0x14,0xc203c000}, {0x18,0x825b4001}, {0x1c,0x9402a001},
    {0x20,0xb0060001}, {0x24,0x92026038}, {0x28,0x9e03e004}, {0x2c,0x80a2a003}, {0x30,0x04bfffed}, {0x34,0x98033ffc}, {0x38,0x832c2002}, {0x3c,0x8200401e},
    {0x40,0xa0042001}, {0x44,0xf0207fe8}, {0x48,0x80a42003}, {0x4c,0x04bfffe0}, {0x50,0x9602e004}, {0x54,0xd207bfe0}, {0x58,0xd407bfe4}, {0x5c,0xd607bfe8},
    {0x60,0xd807bfec}, {0x64,0xda07bff0}, {0x68,0xc207bff4}, {0x6c,0x933a6008}, {0x70,0x953aa008}, {0x74,0x973ae008}, {0x78,0x993b2008}, {0x7c,0x9b3b6008},

    {0xf0,0x40},
    {0x00,0x83386008}, {0x04,0x90102020}, {0x08,0xd227bfe0}, {0x0c,0xd427bfe4}, {0x10,0xd627bfe8}, {0x14,0xd827bfec}, {0x18,0xda27bff0}, {0x1c,0xc227bff4},
    {0x20,0x7fffffa9}, {0x24,0x90220019}, {0x28,0x80a22020}, {0x2c,0x14800011}, {0x30,0xb0102000}, {0x34,0x82020008}, {0x38,0x82004008}, {0x3c,0x83286003},
    {0x40,0x90006ce8}, {0x44,0x9807bfe0}, {0x48,0x94102005}, {0x4c,0xc2030000}, {0x50,0xda020000}, {0x54,0x8258400d}, {0x58,0xb0060001}, {0x5c,0x98032004},
    {0x60,0x9482bfff}, {0x64,0x1cbffffa}, {0x68,0x90022004}, {0x6c,0x30800011}, {0x70,0x82102041}, {0x74,0x90204008}, {0x78,0x82020008}, {0x7c,0x82004008},

    {0xf0,0x41},
    {0x00,0x83286003}, {0x04,0x90006cfc}, {0x08,0x9807bfe0}, {0x0c,0x94102005}, {0x10,0xc2030000}, {0x14,0xda020000}, {0x18,0x8258400d}, {0x1c,0xb0060001},
    {0x20,0x98032004}, {0x24,0x9482bfff}, {0x28,0x1cbffffa}, {0x2c,0x90023ffc}, {0x30,0x81c7e008}, {0x34,0x81e80000}, {0x38,0x9de3bf98}, {0x3c,0x9010001a},
    {0x40,0x7fffff70}, {0x44,0x92100018}, {0x48,0xb4100008}, {0x4c,0x9010001b}, {0x50,0x7fffff6c}, {0x54,0x92100019}, {0x58,0x7fffff83}, {0x5c,0x97e80008},
    {0x60,0x01000000}, {0x64,0x9de3bf90}, {0x68,0xa8102000}, {0x6c,0xf427a04c}, {0x70,0xaa102000}, {0x74,0xac102000}, {0x78,0xae102010}, {0x7c,0xe827bff4},

    {0xf0,0x42},
    {0x00,0xb4250017}, {0x04,0x9210001a}, {0x08,0x94100018}, {0x0c,0x96100019}, {0x10,0x7fffffea}, {0x14,0x90100015}, {0x18,0xa6100008}, {0x1c,0xb6254017},
    {0x20,0x92100014}, {0x24,0x94100018}, {0x28,0x96100019}, {0x2c,0x7fffffe3}, {0x30,0x9010001b}, {0x34,0xa4100008}, {0x38,0xb8050017}, {0x3c,0x9210001c},
    {0x40,0x94100018}, {0x44,0x96100019}, {0x48,0x7fffffdc}, {0x4c,0x90100015}, {0x50,0xa2100008}, {0x54,0xba054017}, {0x58,0x92100014}, {0x5c,0x94100018},
    {0x60,0x96100019}, {0x64,0x7fffffd5}, {0x68,0x9010001d}, {0x6c,0xa0100008}, {0x70,0x90100015}, {0x74,0x92100014}, {0x78,0x94100018}, {0x7c,0x7fffffcf},

    {0xf0,0x43},
    {0x00,0x96100019}, {0x04,0xa624c008}, {0x08,0xa0240008}, {0x0c,0xa4248008}, {0x10,0xa2244008}, {0x14,0x80a4e000}, {0x18,0x04800004}, {0x1c,0x82102000},
    {0x20,0x82100013}, {0x24,0xac102001}, {0x28,0x80a48001}, {0x2c,0x04800005}, {0x30,0x80a44001}, {0x34,0x82100012}, {0x38,0xac102003}, {0x3c,0x80a44001},
    {0x40,0x04800005}, {0x44,0x80a40001}, {0x48,0x82100011}, {0x4c,0xac102005}, {0x50,0x80a40001}, {0x54,0x04800005}, {0x58,0x80a06000}, {0x5c,0x82100010},
    {0x60,0xac102007}, {0x64,0x80a06000}, {0x68,0x14800017}, {0x6c,0x80a5a001}, {0x70,0x80a5e020}, {0x74,0x12800004}, {0x78,0x80a5e010}, {0x7c,0x10800020},

    {0xf0,0x44},
    {0x00,0xae102010}, {0x04,0x12800004}, {0x08,0x80a5e008}, {0x0c,0x1080001c}, {0x10,0xae102008}, {0x14,0x12800004}, {0x18,0x80a5e004}, {0x1c,0x10800018},
    {0x20,0xae102004}, {0x24,0x12800004}, {0x28,0x80a5e002}, {0x2c,0x10800014}, {0x30,0xae102002}, {0x34,0x12800018}, {0x38,0x832e2006}, {0x3c,0x10800010},
    {0x40,0xae102001}, {0x44,0x12800004}, {0x48,0x80a5a003}, {0x4c,0x1080000c}, {0x50,0xa810001a}, {0x54,0x12800004}, {0x58,0x80a5a005}, {0x5c,0x10800008},
    {0x60,0xaa10001b}, {0x64,0x12800004}, {0x68,0x80a5a007}, {0x6c,0x10800004}, {0x70,0xa810001c}, {0x74,0x22800002}, {0x78,0xaa10001d}, {0x7c,0xc207bff4},

    {0xf0,0x45},
    {0x00,0x82006001}, {0x04,0x80a0607f}, {0x08,0x04bfff9e}, {0x0c,0xc227bff4}, {0x10,0x832e2006}, {0x14,0xaa054001}, {0x18,0x82380015}, {0x1c,0x8338601f},
    {0x20,0xaa0d4001}, {0x24,0x9b2e6006}, {0x28,0xc2002308}, {0x2c,0xa885000d}, {0x30,0x1c800004}, {0x34,0x83286006}, {0x38,0x10800005}, {0x3c,0xa8102000},
    {0x40,0x80a50001}, {0x44,0x38800002}, {0x48,0xa8100001}, {0x4c,0x9a0d2fff}, {0x50,0x832d6010}, {0x54,0x8210400d}, {0x58,0xd807a04c}, {0x5c,0x9b2b2002},
    {0x60,0xc2236768}, {0x64,0x81c7e008}, {0x68,0x81e80000}, {0x6c,0x9de3bf98}, {0x70,0xfa50245a}, {0x74,0x80a76000}, {0x78,0x0280003d}, {0x7c,0x9e102001},

    {0xf0,0x46},
    {0x00,0xc20022fc}, {0x04,0x80a3c001}, {0x08,0x18800039}, {0x0c,0x17000018}, {0x10,0x8212e220}, {0x14,0x9810001d}, {0x18,0x9a006004}, {0x1c,0xb6102001},
    {0x20,0xf20022f8}, {0x24,0x80a6c019}, {0x28,0xb4102000}, {0x2c,0x1880002b}, {0x30,0x82102000}, {0x34,0xf0502458}, {0x38,0xba036038}, {0x3c,0xf8074000},
    {0x40,0xb606e001}, {0x44,0x80a70018}, {0x48,0x06800004}, {0x4c,0xba076038}, {0x50,0xb406801c}, {0x54,0x82006001}, {0x58,0x80a6c019}, {0x5c,0x28bffff9},
    {0x60,0xf8074000}, {0x64,0x80a06000}, {0x68,0x2280001d}, {0x6c,0x9e03e001}, {0x70,0x953ea01f}, {0x74,0x8182a000}, {0x78,0x01000000}, {0x7c,0x01000000},

    {0xf0,0x47},
    {0x00,0x01000000}, {0x04,0x827e8001}, {0x08,0x8258400c}, {0x0c,0xbb38601f}, {0x10,0xbb376016}, {0x14,0x8200401d}, {0x18,0xb6102001}, {0x1c,0xfa0022f8},
    {0x20,0x80a6c01d}, {0x24,0x1880000d}, {0x28,0xb538600a}, {0x2c,0x832be002}, {0x30,0xba006038}, {0x34,0xb812e220}, {0x38,0xc207401c}, {0x3c,0x8220401a},
    {0x40,0xc227401c}, {0x44,0xb606e001}, {0x48,0xc20022f8}, {0x4c,0x80a6c001}, {0x50,0x08bffffa}, {0x54,0xba076038}, {0x58,0x9e03e001}, {0x5c,0xc20022fc},
    {0x60,0x80a3c001}, {0x64,0x08bfffce}, {0x68,0x9a036004}, {0x6c,0x81c7e008}, {0x70,0x81e80000}, {0x74,0x9de3bf48}, {0x78,0x1b00003f}, {0x7c,0xc2002350},

    {0xf0,0x48},
    {0x00,0x9a1363ff}, {0x04,0xba08400d}, {0x08,0xa4102001}, {0x0c,0xda0022f8}, {0x10,0x80a4800d}, {0x14,0x18800063}, {0x18,0xa3306010}, {0x1c,0xae10200e},
    {0x20,0xac10200e}, {0x24,0xaa102000}, {0x28,0xa8102000}, {0x2c,0xa6102000}, {0x30,0x80a46000}, {0x34,0x02800033}, {0x38,0xa0102000}, {0x3c,0x03000018},
    {0x40,0x96106220}, {0x44,0x92102000}, {0x48,0x9807bfa8}, {0x4c,0x8204c009}, {0x50,0xda086440}, {0x54,0x8205800d}, {0x58,0x80a36000}, {0x5c,0x02800007},
    {0x60,0x83286002}, {0x64,0xc200400b}, {0x68,0xc2230000}, {0x6c,0x92026001}, {0x70,0x10bffff7}, {0x74,0x98032004}, {0x78,0x7ffffc7d}, {0x7c,0x9007bfa8},

    {0xf0,0x49},
    {0x00,0x80a74011}, {0x04,0x1480000b}, {0x08,0x9210001d}, {0x0c,0x832f6002}, {0x10,0x8200401e}, {0x14,0x9a007fa8}, {0x18,0xc2034000}, {0x1c,0x92026001},
    {0x20,0xa0040001}, {0x24,0x80a24011}, {0x28,0x04bffffc}, {0x2c,0x9a036004}, {0x30,0x8224401d}, {0x34,0x82006001}, {0x38,0x9b3c201f}, {0x3c,0x81836000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0xa0fc0001}, {0x50,0x36800007}, {0x54,0xda0023c4}, {0x58,0xc20023c8}, {0x5c,0x80886020},
    {0x60,0x22800026}, {0x64,0xaa056001}, {0x68,0xda0023c4}, {0x6c,0x9a5c000d}, {0x70,0x833b601f}, {0x74,0x83306018}, {0x78,0x9a034001}, {0x7c,0xa13b6008},

    {0xf0,0x4a},
    {0x00,0x92102000}, {0x04,0x11000018}, {0x08,0x82050009}, {0x0c,0xda086440}, {0x10,0x8205c00d}, {0x14,0x94122220}, {0x18,0x97286002}, {0x1c,0x80a36000},
    {0x20,0x02800015}, {0x24,0x92026001}, {0x28,0xc202c00a}, {0x2c,0x98204010}, {0x30,0xda0822b0}, {0x34,0x833b201f}, {0x38,0x80a0000d}, {0x3c,0x8220400c},
    {0x40,0x9a402000}, {0x44,0x8330601f}, {0x48,0x808b4001}, {0x4c,0x22bfffef}, {0x50,0xd822c00a}, {0x54,0xda0ca2b0}, {0x58,0x9a5b000d}, {0x5c,0x833b601f},
    {0x60,0x83306019}, {0x64,0x9a034001}, {0x68,0x993b6007}, {0x6c,0x10bfffe7}, {0x70,0xd822c00a}, {0x74,0xaa056001}, {0x78,0xa604e00c}, {0x7c,0x80a56001},

    {0xf0,0x4b},
    {0x00,0x04bfffac}, {0x04,0xa805200c}, {0x08,0xa404a001}, {0x0c,0xc20022f8}, {0x10,0x80a48001}, {0x14,0xac05a00e}, {0x18,0x08bfffa3}, {0x1c,0xae05e00e},
    {0x20,0x81c7e008}, {0x24,0x81e80000}, {0x28,0x9de3bf98}, {0x2c,0xc21023b6}, {0x30,0xf81023be}, {0x34,0x96102001}, {0x38,0xfa0022f8}, {0x3c,0x80a2c01d},
    {0x40,0xa8004001}, {0x44,0xa407001c}, {0x48,0x18800088}, {0x4c,0xe6002214}, {0x50,0x90102038}, {0x54,0x92102038}, {0x58,0x9810200e}, {0x5c,0x15000018},
    {0x60,0xb8102001}, {0x64,0xc20022fc}, {0x68,0x80a70001}, {0x6c,0x38800079}, {0x70,0x9602e001}, {0x74,0x2f000018}, {0x78,0xac12a220}, {0x7c,0xaa12a224},

    {0xf0,0x4c},
    {0x00,0x8203001c}, {0x04,0xb7286002}, {0x08,0xfa06c016}, {0x0c,0x80a74013}, {0x10,0x2480006b}, {0x14,0xb8072001}, {0x18,0x80a74014}, {0x1c,0x16800014},
    {0x20,0x83286002}, {0x24,0x80a74012}, {0x28,0x06800007}, {0x2c,0x8215e21c}, {0x30,0xc206c015}, {0x34,0x80a04012}, {0x38,0x1680000c}, {0x3c,0x8203001c},
    {0x40,0x8215e21c}, {0x44,0xc206c001}, {0x48,0x80a74001}, {0x4c,0x2680005c}, {0x50,0xb8072001}, {0x54,0xc206c015}, {0x58,0x80a74001}, {0x5c,0x24800058},
    {0x60,0xb8072001}, {0x64,0x8203001c}, {0x68,0x83286002}, {0x6c,0xfa0023c8}, {0x70,0x808f6040}, {0x74,0xf0004016}, {0x78,0x0280000b}, {0x7c,0xa2072001},

    {0xf0,0x4d},
    {0x00,0xfa0022fc}, {0x04,0x83376001}, {0x08,0x80a70001}, {0x0c,0x28800007}, {0x10,0x9a102000}, {0x14,0x8227401c}, {0x18,0xb8006001}, {0x1c,0x10800003},
    {0x20,0x9a102001}, {0x24,0x9a102000}, {0x28,0xfa00221c}, {0x2c,0xc2002220}, {0x30,0xba5f401c}, {0x34,0xba074001}, {0x38,0xba5e001d}, {0x3c,0x833f601f},
    {0x40,0x83306016}, {0x44,0xba074001}, {0x48,0xc2002224}, {0x4c,0x8258401c}, {0x50,0xbb3f600a}, {0x54,0xba074001}, {0x58,0xc2002240}, {0x5c,0xb0074001},
    {0x60,0xc2002218}, {0x64,0xb6070001}, {0x68,0xa012a220}, {0x6c,0xb92ee002}, {0x70,0xba10001c}, {0x74,0xb2024010}, {0x78,0x9e020010}, {0x7c,0xc20023c8},

    {0xf0,0x4e},
    {0x00,0x80886040}, {0x04,0xb806401c}, {0x08,0x02800007}, {0x0c,0xb403c01d}, {0x10,0xc20022fc}, {0x14,0x83306001}, {0x18,0x80a6c001}, {0x1c,0x38800027},
    {0x20,0xb8100011}, {0x24,0xfa0022fc}, {0x28,0x8227401b}, {0x2c,0x83286002}, {0x30,0x80a6c01d}, {0x34,0x18800020}, {0x38,0x82064001}, {0x3c,0x80a36000},
    {0x40,0x32800002}, {0x44,0xb8006004}, {0x48,0xc2070000}, {0x4c,0x82204018}, {0x50,0xc2270000}, {0x54,0xfa002228}, {0x58,0x8226c01d}, {0x5c,0x80a6c01d},
    {0x60,0x04800013}, {0x64,0xb85e0001}, {0x68,0x80a36000}, {0x6c,0x22800008}, {0x70,0xc200222c}, {0x74,0xc20022fc}, {0x78,0x8220401b}, {0x7c,0x83286002},

    {0xf0,0x4f},
    {0x00,0x8203c001}, {0x04,0xb4006004}, {0x08,0xc200222c}, {0x0c,0x825f0001}, {0x10,0xbb38601f}, {0x14,0xbb376018}, {0x18,0x8200401d}, {0x1c,0xfa068000},
    {0x20,0x83386008}, {0x24,0xba274001}, {0x28,0xfa268000}, {0x2c,0x10bfffd0}, {0x30,0xb606e001}, {0x34,0xb8100011}, {0x38,0xb8072001}, {0x3c,0xc20022fc},
    {0x40,0x80a70001}, {0x44,0x08bfff90}, {0x48,0x8203001c}, {0x4c,0x9602e001}, {0x50,0xc20022f8}, {0x54,0x80a2c001}, {0x58,0x9803200e}, {0x5c,0x92026038},
    {0x60,0x08bfff80}, {0x64,0x90022038}, {0x68,0x81c7e008}, {0x6c,0x81e80000}, {0x70,0x9de3bf98}, {0x74,0xc21023b6}, {0x78,0xf81023be}, {0x7c,0x96102001},

    {0xf0,0x50},
    {0x00,0xfa0022fc}, {0x04,0x80a2c01d}, {0x08,0xa0004001}, {0x0c,0x9207001c}, {0x10,0x1880005e}, {0x14,0xd0002214}, {0x18,0x15000018}, {0x1c,0x9a102001},
    {0x20,0xc20022f8}, {0x24,0x80a34001}, {0x28,0x18800053}, {0x2c,0x832ae002}, {0x30,0xb2006038}, {0x34,0x27000018}, {0x38,0xa412a220}, {0x3c,0xa212a258},
    {0x40,0xfa064012}, {0x44,0x80a74008}, {0x48,0x24800047}, {0x4c,0x9a036001}, {0x50,0x80a74010}, {0x54,0x36800013}, {0x58,0xfa00221c}, {0x5c,0x80a74009},
    {0x60,0x06800007}, {0x64,0x8214e1e8}, {0x68,0xc2064011}, {0x6c,0x80a04009}, {0x70,0x3680000c}, {0x74,0xfa00221c}, {0x78,0x8214e1e8}, {0x7c,0xc2064001},

    {0xf0,0x51},
    {0x00,0x80a74001}, {0x04,0x26800038}, {0x08,0x9a036001}, {0x0c,0xc2064011}, {0x10,0x80a74001}, {0x14,0x24800034}, {0x18,0x9a036001}, {0x1c,0xfa00221c},
    {0x20,0xc2002220}, {0x24,0xba5f400d}, {0x28,0xba074001}, {0x2c,0xf8064012}, {0x30,0xba5f001d}, {0x34,0x833f601f}, {0x38,0x83306016}, {0x3c,0xba074001},
    {0x40,0xc2002224}, {0x44,0x8258400d}, {0x48,0xbb3f600a}, {0x4c,0xba074001}, {0x50,0xc2002218}, {0x54,0xb6034001}, {0x58,0xc2002240}, {0x5c,0xb8074001},
    {0x60,0xc20022f8}, {0x64,0x80a6c001}, {0x68,0x1880001c}, {0x6c,0x832ee003}, {0x70,0x8220401b}, {0x74,0x82004001}, {0x78,0x8200400b}, {0x7c,0xb5286002},

    {0xf0,0x52},
    {0x00,0x9812a220}, {0x04,0xc206800c}, {0x08,0x9e20401c}, {0x0c,0xde26800c}, {0x10,0xfa002228}, {0x14,0x8226c01d}, {0x18,0x80a6c01d}, {0x1c,0xb05f0001},
    {0x20,0x0480000a}, {0x24,0xb606e001}, {0x28,0xc200222c}, {0x2c,0x825e0001}, {0x30,0xbb38601f}, {0x34,0xbb376018}, {0x38,0x8200401d}, {0x3c,0x83386008},
    {0x40,0x8223c001}, {0x44,0xc226800c}, {0x48,0xc20022f8}, {0x4c,0x80a6c001}, {0x50,0x08bfffed}, {0x54,0xb406a038}, {0x58,0x9a036001}, {0x5c,0xb2066038},
    {0x60,0x9a036001}, {0x64,0xc20022f8}, {0x68,0x80a34001}, {0x6c,0x08bfffb5}, {0x70,0xb2066038}, {0x74,0x9602e001}, {0x78,0xc20022fc}, {0x7c,0x80a2c001},

    {0xf0,0x53},
    {0x00,0x08bfffa8}, {0x04,0x9a102001}, {0x08,0x81c7e008}, {0x0c,0x81e80000}, {0x10,0xc2002214}, {0x14,0x80a06000}, {0x18,0x0280000c}, {0x1c,0x01000000},
    {0x20,0xc20023c8}, {0x24,0x80886010}, {0x28,0x02800005}, {0x2c,0x01000000}, {0x30,0x03000009}, {0x34,0x81c061a8}, {0x38,0x01000000}, {0x3c,0x03000009},
    {0x40,0x81c063f0}, {0x44,0x01000000}, {0x48,0x01000000}, {0x4c,0x81c3e008}, {0x50,0x01000000}, {0x54,0x9de3bf98}, {0x58,0xb0102001}, {0x5c,0xda002200},
    {0x60,0x80a6000d}, {0x64,0x1880001d}, {0x68,0xc0202504}, {0x6c,0x03000018}, {0x70,0x98106220}, {0x74,0xde0022fc}, {0x78,0xb2102007}, {0x7c,0xb6102001},

    {0xf0,0x54},
    {0x00,0x80a6c00f}, {0x04,0x18800011}, {0x08,0x832e6003}, {0x0c,0x8200400c}, {0x10,0xba006004}, {0x14,0xf4002238}, {0x18,0xc2074000}, {0x1c,0xb606e001},
    {0x20,0xba076004}, {0x24,0x80a0401a}, {0x28,0x08800005}, {0x2c,0xb820401a}, {0x30,0xc2002504}, {0x34,0x8200401c}, {0x38,0xc2202504}, {0x3c,0x80a6c00f},
    {0x40,0x28bffff7}, {0x44,0xc2074000}, {0x48,0xb0062001}, {0x4c,0x80a6000d}, {0x50,0x08bfffeb}, {0x54,0xb2066007}, {0x58,0xfa002504}, {0x5c,0xc200223c},
    {0x60,0x80a74001}, {0x64,0x28800004}, {0x68,0xc0202568}, {0x6c,0x82102001}, {0x70,0xc2202568}, {0x74,0x033fc180}, {0x78,0xfa002568}, {0x7c,0x8210602c},

    {0xf0,0x55},
    {0x00,0xfa204000}, {0x04,0xfa202570}, {0x08,0x81c7e008}, {0x0c,0x81e80000}, {0x10,0x9de3bf70}, {0x14,0x92102001}, {0x18,0xd0002300}, {0x1c,0x80a24008},
    {0x20,0x1880001c}, {0x24,0x9e102000}, {0x28,0x03000018}, {0x2c,0xa2106220}, {0x30,0xd4002308}, {0x34,0x98102007}, {0x38,0x96102001}, {0x3c,0x80a2c00a},
    {0x40,0x38800011}, {0x44,0x92026001}, {0x48,0x832b2003}, {0x4c,0x82004011}, {0x50,0x82006004}, {0x54,0xda004000}, {0x58,0x80a3400f}, {0x5c,0x04800005},
    {0x60,0x82006004}, {0x64,0x9e10000d}, {0x68,0xa0100009}, {0x6c,0xa410000b}, {0x70,0x9602e001}, {0x74,0x80a2c00a}, {0x78,0x28bffff8}, {0x7c,0xda004000},

    {0xf0,0x56},
    {0x00,0x92026001}, {0x04,0x80a24008}, {0x08,0x08bfffec}, {0x0c,0x98032007}, {0x10,0xa2042001}, {0x14,0x92043fff}, {0x18,0x80a24011}, {0x1c,0x1480002e},
    {0x20,0x9e102000}, {0x24,0x832a6003}, {0x28,0x90204009}, {0x2c,0x03000018}, {0x30,0xa6106220}, {0x34,0xa004a001}, {0x38,0x9604bfff}, {0x3c,0x80a2c010},
    {0x40,0x14800021}, {0x44,0x82020008}, {0x48,0x8200400b}, {0x4c,0x9b2be002}, {0x50,0x83286002}, {0x54,0x9a03401e}, {0x58,0x94004013}, {0x5c,0x9a037fd0},
    {0x60,0x833ae01f}, {0x64,0x8220400b}, {0x68,0x80a26000}, {0x6c,0x0480000f}, {0x70,0x9930601f}, {0x74,0xc2002300}, {0x78,0x80a04009}, {0x7c,0x82603fff},

    {0xf0,0x57},
    {0x00,0x8088400c}, {0x04,0x2280000a}, {0x08,0xc0234000}, {0x0c,0xc2002308}, {0x10,0x80a2c001}, {0x14,0x38800006}, {0x18,0xc0234000}, {0x1c,0xc2028000},
    {0x20,0x10800003}, {0x24,0xc2234000}, {0x28,0xc0234000}, {0x2c,0x9602e001}, {0x30,0x9e03e001}, {0x34,0x9a036004}, {0x38,0x80a2c010}, {0x3c,0x04bfffe9},
    {0x40,0x9402a004}, {0x44,0x92026001}, {0x48,0x80a24011}, {0x4c,0x04bfffdb}, {0x50,0x90022007}, {0x54,0x9007bfd0}, {0x58,0x7ffffaa5}, {0x5c,0x92102009},
    {0x60,0xda07bfec}, {0x64,0xc207bfe8}, {0x68,0x8200400d}, {0x6c,0xda07bff0}, {0x70,0x8200400d}, {0x74,0x9b30601f}, {0x78,0x8200400d}, {0x7c,0xd6082347},

    {0xf0,0x58},
    {0x00,0x9602e001}, {0x04,0xda00256c}, {0x08,0xd808257f}, {0x0c,0x9a5b400b}, {0x10,0x98032001}, {0x14,0x81800000}, {0x18,0x01000000}, {0x1c,0x01000000},
    {0x20,0x01000000}, {0x24,0x9a73400c}, {0x28,0x83386001}, {0x2c,0xc2202590}, {0x30,0xda20256c}, {0x34,0x96102000}, {0x38,0x94102c18}, {0x3c,0x992ae002},
    {0x40,0xc20323b4}, {0x44,0x80a06000}, {0x48,0x12800009}, {0x4c,0x80a2e002}, {0x50,0xc2002520}, {0x54,0x14800004}, {0x58,0x9a200001}, {0x5c,0x10800014},
    {0x60,0xc2232520}, {0x64,0x10800012}, {0x68,0xda232520}, {0x6c,0xda1323b4}, {0x70,0xc2002590}, {0x74,0x8258400d}, {0x78,0x9b38601f}, {0x7c,0x9b336018},

    {0xf0,0x59},
    {0x00,0x8200400d}, {0x04,0xda1323b6}, {0x08,0x83386008}, {0x0c,0x8200400d}, {0x10,0xda00256c}, {0x14,0x8258400d}, {0x18,0x83306007}, {0x1c,0x80a06c18},
    {0x20,0x04800003}, {0x24,0xc2232520}, {0x28,0xd4232520}, {0x2c,0x9602e001}, {0x30,0x80a2e003}, {0x34,0x04bfffe3}, {0x38,0x992ae002}, {0x3c,0xda102472},
    {0x40,0xc2002288}, {0x44,0x80a36000}, {0x48,0x02800004}, {0x4c,0xc220251c}, {0x50,0x10800005}, {0x54,0xda202530}, {0x58,0x0300001f}, {0x5c,0x821063ff},
    {0x60,0xc2202530}, {0x64,0x81c7e008}, {0x68,0x81e80000}, {0x6c,0x9de3bf80}, {0x70,0x832e6003}, {0x74,0x82204019}, {0x78,0x82004001}, {0x7c,0x82004018},

    {0xf0,0x5a},
    {0x00,0x3b000018}, {0x04,0x83286002}, {0x08,0xc020254c}, {0x0c,0xba176220}, {0x10,0xea00401d}, {0x14,0x9e100019}, {0x18,0xb2100018}, {0x1c,0xc2002528},
    {0x20,0x80a54001}, {0x24,0x9810001a}, {0x28,0x068000c9}, {0x2c,0xb0102000}, {0x30,0xa006401a}, {0x34,0xa403c01a}, {0x38,0x8207bfe0}, {0x3c,0xb2102004},
    {0x40,0xc0204000}, {0x44,0xb2867fff}, {0x48,0x1cbffffe}, {0x4c,0x82006004}, {0x50,0x9e23c00c}, {0x54,0x80a3c012}, {0x58,0x14800061}, {0x5c,0xb92be003},
    {0x60,0xba03c00f}, {0x64,0x82048012}, {0x68,0xb827000f}, {0x6c,0xba07400f}, {0x70,0x82004012}, {0x74,0xba274001}, {0x78,0x9607001c}, {0x7c,0x92274010},

    {0xf0,0x5b},
    {0x00,0x9410000b}, {0x04,0x2d000018}, {0x08,0x8203000c}, {0x0c,0xb2240001}, {0x10,0x80a64010}, {0x14,0x1480004c}, {0x18,0xbb3be01f}, {0x1c,0x82028019},
    {0x20,0xba27400f}, {0x24,0x83286002}, {0x28,0xb815a220}, {0x2c,0xb6064009}, {0x30,0x9a00401c}, {0x34,0xa937601f}, {0x38,0xb406e008}, {0x3c,0x80a32001},
    {0x40,0x0280000c}, {0x44,0x80a6600e}, {0x48,0x18800003}, {0x4c,0xba102001}, {0x50,0xba102000}, {0x54,0x80a3e019}, {0x58,0x18800003}, {0x5c,0x82102001},
    {0x60,0x82102000}, {0x64,0x80974001}, {0x68,0x32800033}, {0x6c,0xb2066001}, {0x70,0xc2034000}, {0x74,0x80a04015}, {0x78,0x14800003}, {0x7c,0xba102001},

    {0xf0,0x5c},
    {0x00,0xba102000}, {0x04,0x833e601f}, {0x08,0x82204019}, {0x0c,0x8330601f}, {0x10,0x808f4001}, {0x14,0x0280000c}, {0x18,0x80a32001}, {0x1c,0xc2002308},
    {0x20,0x80a04019}, {0x24,0x82603fff}, {0x28,0x80884014}, {0x2c,0x02800006}, {0x30,0x80a32001}, {0x34,0xc2002300}, {0x38,0x80a3c001}, {0x3c,0x08800083},
    {0x40,0x80a32001}, {0x44,0x3280001c}, {0x48,0xb2066001}, {0x4c,0x8202c019}, {0x50,0xa3286002}, {0x54,0x912b001a}, {0x58,0xb6102000}, {0x5c,0xa615a220},
    {0x60,0xb92ee002}, {0x64,0xc2072520}, {0x68,0xfa044013}, {0x6c,0x80a74001}, {0x70,0x0480000c}, {0x74,0x8207bff8}, {0x78,0x80a6e003}, {0x7c,0x14800006},

    {0xf0,0x5d},
    {0x00,0xb0070001}, {0x04,0xc2063fe8}, {0x08,0x82104008}, {0x0c,0x10800005}, {0x10,0xc2263fe8}, {0x14,0xc2063fe8}, {0x18,0x82006001}, {0x1c,0xc2263fe8},
    {0x20,0xb606e001}, {0x24,0x80a6e004}, {0x28,0x08bfffef}, {0x2c,0xb92ee002}, {0x30,0xb2066001}, {0x34,0x9a036004}, {0x38,0x80a64010}, {0x3c,0x04bfffc0},
    {0x40,0xb406a001}, {0x44,0x9e03e001}, {0x48,0x92026003}, {0x4c,0x9402a00e}, {0x50,0x80a3c012}, {0x54,0x04bfffad}, {0x58,0x9602e00e}, {0x5c,0xfa102470},
    {0x60,0xc207bff0}, {0x64,0x80a0401d}, {0x68,0x14800003}, {0x6c,0xba102001}, {0x70,0xba102000}, {0x74,0x821b2002}, {0x78,0x80a00001}, {0x7c,0x82603fff},

    {0xf0,0x5e},
    {0x00,0x80974001}, {0x04,0x12800052}, {0x08,0xb0103fff}, {0x0c,0xc207bfe0}, {0x10,0x80886010}, {0x14,0x0280000a}, {0x18,0xfa07bfe4}, {0x1c,0xc207bfec},
    {0x20,0x80886082}, {0x24,0x02800007}, {0x28,0x808f6082}, {0x2c,0x80886028}, {0x30,0x12800047}, {0x34,0xb0102003}, {0x38,0xfa07bfe4}, {0x3c,0x808f6082},
    {0x40,0x02800007}, {0x44,0x808f6028}, {0x48,0xc207bfec}, {0x4c,0x80886028}, {0x50,0x3280003f}, {0x54,0xb0102002}, {0x58,0x808f6028}, {0x5c,0x02800008},
    {0x60,0xf807bfe8}, {0x64,0xc207bfec}, {0x68,0x80886082}, {0x6c,0x02800005}, {0x70,0x820f200a}, {0x74,0x10800036}, {0x78,0xb0102002}, {0x7c,0x820f200a},

    {0xf0,0x5f},
    {0x00,0x8218600a}, {0x04,0x80a00001}, {0x08,0xb2043fff}, {0x0c,0xba603fff}, {0x10,0x821e6001}, {0x14,0x80a00001}, {0x18,0xb6402000}, {0x1c,0x808f401b},
    {0x20,0x02800005}, {0x24,0x9e04bfff}, {0x28,0x80a3e001}, {0x2c,0x32800028}, {0x30,0xb0102001}, {0x34,0x820f2022}, {0x38,0x80a06022}, {0x3c,0x1280000d},
    {0x40,0x820f2088}, {0x44,0xc2002308}, {0x48,0x821e4001}, {0x4c,0x80a00001}, {0x50,0xba402000}, {0x54,0x821be001}, {0x58,0x80a00001}, {0x5c,0x82402000},
    {0x60,0x808f4001}, {0x64,0x3280001a}, {0x68,0xb0102001}, {0x6c,0x820f2088}, {0x70,0x82186088}, {0x74,0x80a00001}, {0x78,0x82603fff}, {0x7c,0x8088401b},

    {0xf0,0x60},
    {0x00,0x02800007}, {0x04,0x820f20a0}, {0x08,0xc2002300}, {0x0c,0x80a3c001}, {0x10,0x3280000f}, {0x14,0xb0102001}, {0x18,0x820f20a0}, {0x1c,0x80a060a0},
    {0x20,0x1280000b}, {0x24,0xb0102000}, {0x28,0xc2002308}, {0x2c,0x80a64001}, {0x30,0x02800007}, {0x34,0x01000000}, {0x38,0xc2002300}, {0x3c,0x80a3c001},
    {0x40,0x12800003}, {0x44,0xb0102001}, {0x48,0xb0102000}, {0x4c,0x81c7e008}, {0x50,0x81e80000}, {0x54,0x9de3bf98}, {0x58,0x832e2003}, {0x5c,0x82204018},
    {0x60,0xb2100018}, {0x64,0xbb286003}, {0x68,0x31000018}, {0x6c,0x82162224}, {0x70,0xb6102002}, {0x74,0xf40022fc}, {0x78,0xf8074001}, {0x7c,0x80a6c01a},

    {0xf0,0x61},
    {0x00,0x1880000f}, {0x04,0x9e102001}, {0x08,0x82162220}, {0x0c,0x82074001}, {0x10,0x82006008}, {0x14,0xfa004000}, {0x18,0x80a7401c}, {0x1c,0x16800004},
    {0x20,0x82006004}, {0x24,0xb810001d}, {0x28,0x9e10001b}, {0x2c,0xb606e001}, {0x30,0x80a6c01a}, {0x34,0x28bffff9}, {0x38,0xfa004000}, {0x3c,0x80a72000},
    {0x40,0x16800017}, {0x44,0xb0102000}, {0x48,0x832e6003}, {0x4c,0x82204019}, {0x50,0x82004001}, {0x54,0x39000018}, {0x58,0x8200400f}, {0x5c,0x83286002},
    {0x60,0xba17221c}, {0x64,0xb6172220}, {0x68,0xfa00401d}, {0x6c,0xf600401b}, {0x70,0xb8172224}, {0x74,0xc200401c}, {0x78,0xba07401b}, {0x7c,0xba074001},

    {0xf0,0x62},
    {0x00,0xc200220c}, {0x04,0xba20001d}, {0x08,0xba5f4001}, {0x0c,0x833f601f}, {0x10,0x83306018}, {0x14,0xba074001}, {0x18,0xb13f6008}, {0x1c,0x81c7e008},
    {0x20,0x81e80000}, {0x24,0x9de3bee8}, {0x28,0xa0102000}, {0x2c,0xc20022f8}, {0x30,0x80a40001}, {0x34,0x1a80000a}, {0x38,0xa2042001}, {0x3c,0x8207bff8},
    {0x40,0xa12c2002}, {0x44,0xa0040001}, {0x48,0x7fffffc3}, {0x4c,0x90100011}, {0x50,0xd0243fa0}, {0x54,0x10bffff6}, {0x58,0xa0100011}, {0x5c,0xc0202514},
    {0x60,0xb607bff8}, {0x64,0x8207bf48}, {0x68,0xa2102013}, {0x6c,0xc0204000}, {0x70,0xa2847fff}, {0x74,0x1cbffffe}, {0x78,0x82006004}, {0x7c,0xa2102000},

    {0xf0,0x63},
    {0x00,0x832c6002}, {0x04,0xa2046001}, {0x08,0x80a46009}, {0x0c,0x04bffffd}, {0x10,0xc0206768}, {0x14,0xa0102001}, {0x18,0xc20022f8}, {0x1c,0x80a40001},
    {0x20,0x18800086}, {0x24,0xb810201c}, {0x28,0xba10200e}, {0x2c,0xae10200e}, {0x30,0xa2102001}, {0x34,0xc20022fc}, {0x38,0x80a44001}, {0x3c,0x18800078},
    {0x40,0x03000044}, {0x44,0xac040001}, {0x48,0x9b2f2002}, {0x4c,0x992f6002}, {0x50,0x972de002}, {0x54,0x03000050}, {0x58,0xaa040001}, {0x5c,0xa8036004},
    {0x60,0xa6032008}, {0x64,0xa402e004}, {0x68,0xc2002308}, {0x6c,0x80a44001}, {0x70,0x3880002f}, {0x74,0xc2002304}, {0x78,0xc2002300}, {0x7c,0x80a40001},

    {0xf0,0x64},
    {0x00,0x38800041}, {0x04,0xc200237c}, {0x08,0x90100011}, {0x0c,0x92100010}, {0x10,0x7ffffeb7}, {0x14,0x94102001}, {0x18,0x80a22000}, {0x1c,0x02800057},
    {0x20,0x1b000040}, {0x24,0x1b000018}, {0x28,0x8213621c}, {0x2c,0x96136220}, {0x30,0xd8048001}, {0x34,0xd604800b}, {0x38,0x9a136224}, {0x3c,0x832c2002},
    {0x40,0x9803000b}, {0x44,0xda04800d}, {0x48,0x8200401b}, {0x4c,0x9803000d}, {0x50,0xc2007f9c}, {0x54,0x80a30001}, {0x58,0x06800048}, {0x5c,0x1b000040},
    {0x60,0x80a22000}, {0x64,0x3680000d}, {0x68,0xc2002514}, {0x6c,0x90100011}, {0x70,0x92100010}, {0x74,0x7ffffe9e}, {0x78,0x94102002}, {0x7c,0x80a22000},

    {0xf0,0x65},
    {0x00,0x0280003e}, {0x04,0x1b000040}, {0x08,0xc2002514}, {0x0c,0x9b286002}, {0x10,0x10800034}, {0x14,0xea236768}, {0x18,0x9b2c6010}, {0x1c,0x9a034010},
    {0x20,0x99286002}, {0x24,0x1080002f}, {0x28,0xda232768}, {0x2c,0x80a06000}, {0x30,0x02800007}, {0x34,0x19000018}, {0x38,0xc2002300}, {0x3c,0x80a40001},
    {0x40,0x0880002e}, {0x44,0x1b000040}, {0x48,0x19000018}, {0x4c,0x82132220}, {0x50,0xda04c001}, {0x54,0xc200251c}, {0x58,0x80a34001}, {0x5c,0x24800027},
    {0x60,0x1b000040}, {0x64,0x821321e8}, {0x68,0xc204c001}, {0x6c,0x80a0400d}, {0x70,0x36800022}, {0x74,0x1b000040}, {0x78,0x82132258}, {0x7c,0x10800013},

    {0xf0,0x66},
    {0x00,0xc204c001}, {0x04,0x80a06000}, {0x08,0x1280001c}, {0x0c,0x1b000040}, {0x10,0x19000018}, {0x14,0x82132220}, {0x18,0xda050001}, {0x1c,0xc200251c},
    {0x20,0x80a34001}, {0x24,0x24800015}, {0x28,0x1b000040}, {0x2c,0x8213221c}, {0x30,0xc2050001}, {0x34,0x80a0400d}, {0x38,0x36800010}, {0x3c,0x1b000040},
    {0x40,0x82132224}, {0x44,0xc2050001}, {0x48,0x80a34001}, {0x4c,0x0680000b}, {0x50,0x1b000040}, {0x54,0xc2002514}, {0x58,0x9b286002}, {0x5c,0xec236768},
    {0x60,0x82006001}, {0x64,0xc2202514}, {0x68,0xc2002514}, {0x6c,0x80a06009}, {0x70,0x18800012}, {0x74,0x1b000040}, {0x78,0xa2046001}, {0x7c,0xc20022fc},

    {0xf0,0x67},
    {0x00,0xac05800d}, {0x04,0x80a44001}, {0x08,0xa404a004}, {0x0c,0xa604e004}, {0x10,0xa8052004}, {0x14,0x08bfff95}, {0x18,0xaa05400d}, {0x1c,0xa0042001},
    {0x20,0xc20022f8}, {0x24,0x80a40001}, {0x28,0xae05e00e}, {0x2c,0xba07600e}, {0x30,0x08bfff80}, {0x34,0xb807200e}, {0x38,0x81c7e008}, {0x3c,0x81e80000},
    {0x40,0x80a22000}, {0x44,0x2280001d}, {0x48,0xc2002558}, {0x4c,0xd4002208}, {0x50,0x80a2a000}, {0x54,0x0280002f}, {0x58,0x01000000}, {0x5c,0xc2002514},
    {0x60,0x80a06000}, {0x64,0x12800007}, {0x68,0xc2002558}, {0x6c,0x80a06000}, {0x70,0x02800028}, {0x74,0x82007fff}, {0x78,0x10800026}, {0x7c,0xc2202558},

    {0xf0,0x68},
    {0x00,0x80a06000}, {0x04,0x32800023}, {0x08,0xd4202558}, {0x0c,0x17200040}, {0x10,0x193fc200}, {0x14,0x8212e001}, {0x18,0xc2230000}, {0x1c,0xc200233c},
    {0x20,0x83306002}, {0x24,0x9a132070}, {0x28,0xc2234000}, {0x2c,0xd6230000}, {0x30,0x10800018}, {0x34,0xd4202558}, {0x38,0x80a06000}, {0x3c,0x32800007},
    {0x40,0xc2002514}, {0x44,0xc2002208}, {0x48,0x80a06000}, {0x4c,0x1280000e}, {0x50,0x033fc200}, {0x54,0xc2002514}, {0x58,0x80a06001}, {0x5c,0x08800006},
    {0x60,0xd800233c}, {0x64,0x82007fff}, {0x68,0xda002204}, {0x6c,0x8258400d}, {0x70,0x98030001}, {0x74,0x033fc200}, {0x78,0x82106070}, {0x7c,0x10800005},

    {0xf0,0x69},
    {0x00,0xd8204000}, {0x04,0xda002234}, {0x08,0x82106070}, {0x0c,0xda204000}, {0x10,0x81c3e008}, {0x14,0x01000000}, {0x18,0x82220009}, {0x1c,0x9a58400a},
    {0x20,0x833b601f}, {0x24,0x80a20009}, {0x28,0x83306019}, {0x2c,0x04800004}, {0x30,0x90102000}, {0x34,0x82034001}, {0x38,0x91386007}, {0x3c,0x81c3e008},
    {0x40,0x01000000}, {0x44,0x9de3bf98}, {0x48,0xc2002308}, {0x4c,0x82006001}, {0x50,0xe60022fc}, {0x54,0x80a4c001}, {0x58,0x2a800019}, {0x5c,0xe80022f8},
    {0x60,0x15000018}, {0x64,0xa8102001}, {0x68,0xc20022f8}, {0x6c,0x80a50001}, {0x70,0x1880000c}, {0x74,0x832ce002}, {0x78,0x9a006038}, {0x7c,0x9612a224},

    {0xf0,0x6a},
    {0x00,0x9812a220}, {0x04,0xc203400c}, {0x08,0xc223400b}, {0x0c,0xa8052001}, {0x10,0xc20022f8}, {0x14,0x80a50001}, {0x18,0x08bffffb}, {0x1c,0x9a036038},
    {0x20,0xc2002308}, {0x24,0xa604ffff}, {0x28,0x82006001}, {0x2c,0x80a4c001}, {0x30,0x1abfffee}, {0x34,0xa8102001}, {0x38,0xe80022f8}, {0x3c,0x80a52000},
    {0x40,0x0280002a}, {0x44,0x832d2003}, {0x48,0xaa204014}, {0x4c,0x27000018}, {0x50,0xa52d6003}, {0x54,0x8214e228}, {0x58,0xa214e224}, {0x5c,0xd2048001},
    {0x60,0xd408228c}, {0x64,0x7fffffcd}, {0x68,0xd0048011}, {0x6c,0xac14e220}, {0x70,0xd0248016}, {0x74,0xc2002308}, {0x78,0xa0054015}, {0x7c,0xa0040001},

    {0xf0,0x6b},
    {0x00,0xa12c2002}, {0x04,0x8214e21c}, {0x08,0xd2040001}, {0x0c,0xd408228d}, {0x10,0x7fffffc2}, {0x14,0xd0040016}, {0x18,0xd0240011}, {0x1c,0xc2002300},
    {0x20,0x80a50001}, {0x24,0x2880000f}, {0x28,0xa8853fff}, {0x2c,0xa214e258}, {0x30,0x98100016}, {0x34,0x9a100012}, {0x38,0xa6102000}, {0x3c,0xc203400c},
    {0x40,0xc2234011}, {0x44,0xc2002308}, {0x48,0xa604e001}, {0x4c,0x82006001}, {0x50,0x80a4c001}, {0x54,0x08bffffa}, {0x58,0x9a036004}, {0x5c,0xa8853fff},
    {0x60,0x12bfffdb}, {0x64,0xaa057ff9}, {0x68,0xa6102001}, {0x6c,0xc2002308}, {0x70,0x80a4c001}, {0x74,0x18800019}, {0x78,0x23000018}, {0x7c,0xa12ce002},

    {0xf0,0x6c},
    {0x00,0x82146290}, {0x04,0xa4146258}, {0x08,0xd2040001}, {0x0c,0xd408228e}, {0x10,0x7fffffa2}, {0x14,0xd0040012}, {0x18,0x9a146220}, {0x1c,0xd024000d},
    {0x20,0xc2002300}, {0x24,0xa1286003}, {0x28,0xa0240001}, {0x2c,0xa0040010}, {0x30,0xa0040013}, {0x34,0xa12c2002}, {0x38,0xa21461e8}, {0x3c,0xd004000d},
    {0x40,0xd2040011}, {0x44,0x7fffff95}, {0x48,0xd408228f}, {0x4c,0xd0240012}, {0x50,0x10bfffe7}, {0x54,0xa604e001}, {0x58,0x17000018}, {0x5c,0x9012e224},
    {0x60,0x9212e258}, {0x64,0xda024000}, {0x68,0xc2020000}, {0x6c,0x8200400d}, {0x70,0x9412e220}, {0x74,0x83386001}, {0x78,0xc2228000}, {0x7c,0xd8002308},

    {0xf0,0x6d},
    {0x00,0x992b2002}, {0x04,0x9612e25c}, {0x08,0xda03000b}, {0x0c,0xc203000a}, {0x10,0x8200400d}, {0x14,0x83386001}, {0x18,0xc2230008}, {0x1c,0xc2002300},
    {0x20,0x9b286003}, {0x24,0x9a234001}, {0x28,0x9b2b6003}, {0x2c,0xd803400a}, {0x30,0xc203400b}, {0x34,0x8200400c}, {0x38,0x83386001}, {0x3c,0xc2234009},
    {0x40,0xda002300}, {0x44,0x832b6003}, {0x48,0x8220400d}, {0x4c,0xda002308}, {0x50,0x82004001}, {0x54,0x8200400d}, {0x58,0x83286002}, {0x5c,0xda004009},
    {0x60,0xd8004008}, {0x64,0x9a03400c}, {0x68,0x9b3b6001}, {0x6c,0xda20400b}, {0x70,0x81c7e008}, {0x74,0x81e80000}, {0x78,0x80a2200d}, {0x7c,0x82402000},

    {0xf0,0x6e},
    {0x00,0x80a26018}, {0x04,0x90402000}, {0x08,0x81c3e008}, {0x0c,0x90084008}, {0x10,0x9de3bf98}, {0x14,0xa026001b}, {0x18,0xae06001b}, {0x1c,0xf427a04c},
    {0x20,0x03000007}, {0x24,0xba1063fe}, {0x28,0x80a40017}, {0x2c,0xb8102000}, {0x30,0xaa102000}, {0x34,0xac102000}, {0x38,0x1480001f}, {0x3c,0xb4100010},
    {0x40,0x832c2003}, {0x44,0x82204010}, {0x48,0xa6004001}, {0x4c,0xa226401b}, {0x50,0xa806401b}, {0x54,0x80a44014}, {0x58,0x34800014}, {0x5c,0xa0042001},
    {0x60,0x82044013}, {0x64,0xa5286002}, {0x68,0x90100011}, {0x6c,0x7fffffe3}, {0x70,0x92100010}, {0x74,0x80a22000}, {0x78,0x02800008}, {0x7c,0xa2046001},

    {0xf0,0x6f},
    {0x00,0x03000018}, {0x04,0x82106220}, {0x08,0xc2048001}, {0x0c,0x80a0401d}, {0x10,0x26800002}, {0x14,0xba100001}, {0x18,0x80a44014}, {0x1c,0x04bffff3},
    {0x20,0xa404a004}, {0x24,0xa0042001}, {0x28,0x80a40017}, {0x2c,0x04bfffe8}, {0x30,0xa604e00e}, {0x34,0xc2002250}, {0x38,0x80a74001}, {0x3c,0x26800002},
    {0x40,0xba100001}, {0x44,0xb006001b}, {0x48,0x80a68018}, {0x4c,0x14800029}, {0x50,0xa010001a}, {0x54,0x832ea003}, {0x58,0x8220401a}, {0x5c,0xa6004001},
    {0x60,0xa226401b}, {0x64,0xa806401b}, {0x68,0x80a44014}, {0x6c,0x1480001a}, {0x70,0x82044013}, {0x74,0xa5286002}, {0x78,0x90100011}, {0x7c,0x7fffffbf},

    {0xf0,0x70},
    {0x00,0x92100010}, {0x04,0x80a22000}, {0x08,0x22800010}, {0x0c,0xa2046001}, {0x10,0x03000018}, {0x14,0x82106220}, {0x18,0xc2048001}, {0x1c,0x8220401d},
    {0x20,0x9a046001}, {0x24,0x98042001}, {0x28,0x9658400d}, {0x2c,0x80a06000}, {0x30,0x04800005}, {0x34,0x9a58400c}, {0x38,0xaa05400d}, {0x3c,0xac05800b},
    {0x40,0xb8070001}, {0x44,0xa2046001}, {0x48,0x80a44014}, {0x4c,0x04bfffeb}, {0x50,0xa404a004}, {0x54,0xa0042001}, {0x58,0x80a40018}, {0x5c,0x04bfffe1},
    {0x60,0xa604e00e}, {0x64,0x80a72000}, {0x68,0x14800006}, {0x6c,0x9b2d6006}, {0x70,0xd807a04c}, {0x74,0x832b2002}, {0x78,0x1080001d}, {0x7c,0xc0206768},

    {0xf0,0x71},
    {0x00,0x833b601f}, {0x04,0x81806000}, {0x08,0x01000000}, {0x0c,0x01000000}, {0x10,0x01000000}, {0x14,0x9a7b401c}, {0x18,0x832da006}, {0x1c,0x9938601f},
    {0x20,0x81832000}, {0x24,0x01000000}, {0x28,0x01000000}, {0x2c,0x01000000}, {0x30,0x8278401c}, {0x34,0xaa037fa0}, {0x38,0x80a56000}, {0x3c,0x14800003},
    {0x40,0xac007fa0}, {0x44,0xaa102001}, {0x48,0x80a5a000}, {0x4c,0x24800002}, {0x50,0xac102001}, {0x54,0x9a0dafff}, {0x58,0x832d6010}, {0x5c,0x8210400d},
    {0x60,0xd807a04c}, {0x64,0x9b2b2002}, {0x68,0xc2236768}, {0x6c,0x81c7e008}, {0x70,0x81e80000}, {0x74,0x9de3bf98}, {0x78,0x03000018}, {0x7c,0xb6106254},

    {0xf0,0x72},
    {0x00,0xb810625c}, {0x04,0x96106258}, {0x08,0xc2002274}, {0x0c,0x80a06000}, {0x10,0x832e2003}, {0x14,0x82204018}, {0x18,0x82004001}, {0x1c,0x82004019},
    {0x20,0xb12e2006}, {0x24,0xbb2e6006}, {0x28,0xb5286002}, {0x2c,0xb0063fe0}, {0x30,0x9a066001}, {0x34,0x98066002}, {0x38,0x9f2e2010}, {0x3c,0x02800020},
    {0x40,0x82077fe0}, {0x44,0xfa06801b}, {0x48,0xf806801c}, {0x4c,0xf406800b}, {0x50,0x8207401a}, {0x54,0xb610001d}, {0x58,0x80a7401c}, {0x5c,0x04800003},
    {0x60,0xb000401c}, {0x64,0xb610001c}, {0x68,0x8227401b}, {0x6c,0xba26801b}, {0x70,0xba5f400d}, {0x74,0x82584019}, {0x78,0x8200401d}, {0x7c,0xb827001b},

    {0xf0,0x73},
    {0x00,0xb85f000c}, {0x04,0xba06c01b}, {0x08,0x8200401c}, {0x0c,0xba07401b}, {0x10,0xba26001d}, {0x14,0x83286006}, {0x18,0x9b38601f}, {0x1c,0x81836000},
    {0x20,0x01000000}, {0x24,0x01000000}, {0x28,0x01000000}, {0x2c,0x8278401d}, {0x30,0x82807fa0}, {0x34,0x2c800002}, {0x38,0x82102000}, {0x3c,0xb003c001},
    {0x40,0xb0263000}, {0x44,0x81c7e008}, {0x48,0x81e80000}, {0x4c,0x9de3bf98}, {0x50,0xa2102000}, {0x54,0xc2002514}, {0x58,0x80a44001}, {0x5c,0x1a800029},
    {0x60,0xa12c6002}, {0x64,0xda042768}, {0x68,0x93336010}, {0x6c,0x8333600c}, {0x70,0x900b6fff}, {0x74,0x80886001}, {0x78,0x02800006}, {0x7c,0x920a6fff},

    {0xf0,0x74},
    {0x00,0x7fffffbd}, {0x04,0xa2046001}, {0x08,0x1080001a}, {0x0c,0xd0242768}, {0x10,0x80a36000}, {0x14,0x22800017}, {0x18,0xa2046001}, {0x1c,0x93336010},
    {0x20,0xc200246c}, {0x24,0x98100009}, {0x28,0x9f33600e}, {0x2c,0x80a06000}, {0x30,0x900b6fff}, {0x34,0x920a6fff}, {0x38,0x0280000c}, {0x3c,0x94100011},
    {0x40,0x808be001}, {0x44,0x12800005}, {0x48,0x96102002}, {0x4c,0x920b2fff}, {0x50,0x94100011}, {0x54,0x96102001}, {0x58,0x7fffff2e}, {0x5c,0xa2046001},
    {0x60,0x10800005}, {0x64,0xc2002514}, {0x68,0x7ffff99f}, {0x6c,0xa2046001}, {0x70,0xc2002514}, {0x74,0x80a44001}, {0x78,0x0abfffdb}, {0x7c,0xa12c6002},

    {0xf0,0x75},
    {0x00,0x81c7e008}, {0x04,0x81e80000}, {0x08,0x9de3bf98}, {0x0c,0x9e102000}, {0x10,0x832be002}, {0x14,0xfa006768}, {0x18,0x80a76000}, {0x1c,0x2280002e},
    {0x20,0x9e03e001}, {0x24,0x83376010}, {0x28,0xba0f6fff}, {0x2c,0x82086fff}, {0x30,0xb403e001}, {0x34,0x98076020}, {0x38,0x96006020}, {0x3c,0x80a6a009},
    {0x40,0x9a007fe0}, {0x44,0xba077fe0}, {0x48,0x18800022}, {0x4c,0x832ea002}, {0x50,0xf8006768}, {0x54,0x80a72000}, {0x58,0x2280001c}, {0x5c,0xb406a001},
    {0x60,0xb7372010}, {0x64,0xb60eefff}, {0x68,0xb20f2fff}, {0x6c,0x80a6c00d}, {0x70,0x14800003}, {0x74,0xb0102001}, {0x78,0xb0102000}, {0x7c,0x80a6c00b},

    {0xf0,0x76},
    {0x00,0x06800003}, {0x04,0xb8102001}, {0x08,0xb8102000}, {0x0c,0x808e001c}, {0x10,0x2280000e}, {0x14,0xb406a001}, {0x18,0x80a6401d}, {0x1c,0x14800003},
    {0x20,0xb6102001}, {0x24,0xb6102000}, {0x28,0x80a6400c}, {0x2c,0x06800003}, {0x30,0xb8102001}, {0x34,0xb8102000}, {0x38,0x808ec01c}, {0x3c,0x32800002},
    {0x40,0xc0206768}, {0x44,0xb406a001}, {0x48,0x10bfffe0}, {0x4c,0x80a6a009}, {0x50,0x9e03e001}, {0x54,0x80a3e009}, {0x58,0x08bfffcf}, {0x5c,0x832be002},
    {0x60,0x81c7e008}, {0x64,0x81e80000}, {0x68,0xc2002510}, {0x6c,0x82006001}, {0x70,0x80a06008}, {0x74,0x08800003}, {0x78,0xc2202510}, {0x7c,0xc0202510},

    {0xf0,0x77},
    {0x00,0xd8002510}, {0x04,0x96102000}, {0x08,0x832b2002}, {0x0c,0x8200400c}, {0x10,0x83286003}, {0x14,0x82006600}, {0x18,0x9b2ae002}, {0x1c,0x80a32000},
    {0x20,0xc2236790}, {0x24,0x12800003}, {0x28,0x98033fff}, {0x2c,0x98102008}, {0x30,0x9602e001}, {0x34,0x80a2e008}, {0x38,0x04bffff5}, {0x3c,0x832b2002},
    {0x40,0x0303ffc7}, {0x44,0x921063ff}, {0x48,0x98102000}, {0x4c,0x96102000}, {0x50,0x9b2ae002}, {0x54,0xc2036768}, {0x58,0x82084009}, {0x5c,0x9602e001},
    {0x60,0x952b2002}, {0x64,0x80a06000}, {0x68,0x02800004}, {0x6c,0xc2236768}, {0x70,0x98032001}, {0x74,0xc222a768}, {0x78,0x80a2e009}, {0x7c,0x24bffff6},

    {0xf0,0x78},
    {0x00,0x9b2ae002}, {0x04,0x9610000c}, {0x08,0x80a32009}, {0x0c,0x14800007}, {0x10,0xd8202514}, {0x14,0x832ae002}, {0x18,0x9602e001}, {0x1c,0x80a2e009},
    {0x20,0x04bffffd}, {0x24,0xc0206768}, {0x28,0x81c3e008}, {0x2c,0x01000000}, {0x30,0x9de3bf98}, {0x34,0xc20022f4}, {0x38,0x80a06000}, {0x3c,0x02800049},
    {0x40,0xb0102000}, {0x44,0xc2002514}, {0x48,0x80a60001}, {0x4c,0x1a800045}, {0x50,0x033c003f}, {0x54,0x9e1063ff}, {0x58,0xb52e2002}, {0x5c,0xfa06a768},
    {0x60,0x8337600c}, {0x64,0x80886001}, {0x68,0x3280003a}, {0x6c,0xb0062001}, {0x70,0xb9376010}, {0x74,0xb80f2fff}, {0x78,0x80a7201f}, {0x7c,0x2880001a},

    {0xf0,0x79},
    {0x00,0xfa06a768}, {0x04,0xc2002300}, {0x08,0x83286006}, {0x0c,0x82007fe0}, {0x10,0x80a70001}, {0x14,0x38800014}, {0x18,0xfa06a768}, {0x1c,0x808f2020},
    {0x20,0x02800008}, {0x24,0xb60f3fe0}, {0x28,0x8238001c}, {0x2c,0x8208601f}, {0x30,0xc20862d4}, {0x34,0x8226c001}, {0x38,0x10800005}, {0x3c,0x8200601f},
    {0x40,0x820f201f}, {0x44,0xc20862d4}, {0x48,0x8206c001}, {0x4c,0x82086fff}, {0x50,0x83286010}, {0x54,0xba0f400f}, {0x58,0xba174001}, {0x5c,0xfa26a768},
    {0x60,0xfa06a768}, {0x64,0xb80f6fff}, {0x68,0x80a7201f}, {0x6c,0x28800019}, {0x70,0xb0062001}, {0x74,0xc2002308}, {0x78,0x83286006}, {0x7c,0x82007fe0},

    {0xf0,0x7a},
    {0x00,0x80a70001}, {0x04,0x38800013}, {0x08,0xb0062001}, {0x0c,0x808f6020}, {0x10,0xb60f6fe0}, {0x14,0x02800008}, {0x18,0xb20f7000}, {0x1c,0x8238001c},
    {0x20,0x8208601f}, {0x24,0xc2086254}, {0x28,0x8226c001}, {0x2c,0x10800005}, {0x30,0x8200601f}, {0x34,0x820f601f}, {0x38,0xc2086254}, {0x3c,0x8206c001},
    {0x40,0x82086fff}, {0x44,0x82164001}, {0x48,0xc226a768}, {0x4c,0xb0062001}, {0x50,0xc2002514}, {0x54,0x80a60001}, {0x58,0x0abfffc1}, {0x5c,0xb52e2002},
    {0x60,0x81c7e008}, {0x64,0x81e80000}, {0x68,0x912a2002}, {0x6c,0xc2002794}, {0x70,0xda004008}, {0x74,0x033c003c}, {0x78,0x822b4001}, {0x7c,0x98102790},

    {0xf0,0x7b},
    {0x00,0xda030000}, {0x04,0xc2234008}, {0x08,0xd8030000}, {0x0c,0xda030008}, {0x10,0x03000020}, {0x14,0x822b4001}, {0x18,0x81c3e008}, {0x1c,0xc2230008},
    {0x20,0x912a2002}, {0x24,0xc2002790}, {0x28,0xc0204008}, {0x2c,0xc2002794}, {0x30,0xc2104008}, {0x34,0xda002798}, {0x38,0xda134008}, {0x3c,0x82086fff},
    {0x40,0x94004001}, {0x44,0x9a0b6fff}, {0x48,0x80a2800d}, {0x4c,0x18800003}, {0x50,0x9422800d}, {0x54,0x94102000}, {0x58,0xd6002790}, {0x5c,0x9a0aafff},
    {0x60,0xd802c008}, {0x64,0x0303ffc0}, {0x68,0x9b2b6010}, {0x6c,0x822b0001}, {0x70,0x8210400d}, {0x74,0xc222c008}, {0x78,0xc2002794}, {0x7c,0xc2004008},

    {0xf0,0x7c},
    {0x00,0xda002798}, {0x04,0xda034008}, {0x08,0x82086fff}, {0x0c,0x94004001}, {0x10,0x9a0b6fff}, {0x14,0x80a2800d}, {0x18,0x18800003}, {0x1c,0x9422800d},
    {0x20,0x94102000}, {0x24,0xd8002790}, {0x28,0xc2030008}, {0x2c,0x9a0aafff}, {0x30,0x82087000}, {0x34,0x8210400d}, {0x38,0xc2230008}, {0x3c,0xd8002790},
    {0x40,0xc2030008}, {0x44,0x1b000020}, {0x48,0x8210400d}, {0x4c,0x81c3e008}, {0x50,0xc2230008}, {0x54,0x912a2002}, {0x58,0xc2002790}, {0x5c,0xc0204008},
    {0x60,0xc2002794}, {0x64,0xda104008}, {0x68,0xc200279c}, {0x6c,0xd6104008}, {0x70,0xc2002798}, {0x74,0x9a0b6fff}, {0x78,0xd8104008}, {0x7c,0x832b6002},

    {0xf0,0x7d},
    {0x00,0x8200400d}, {0x04,0x960aefff}, {0x08,0x980b2fff}, {0x0c,0x8200400b}, {0x10,0x992b2002}, {0x14,0x80a0400c}, {0x18,0x18800003}, {0x1c,0x8220400c},
    {0x20,0x82102000}, {0x24,0xd6002790}, {0x28,0x9b306001}, {0x2c,0xd802c008}, {0x30,0x9a0b6fff}, {0x34,0x0303ffc0}, {0x38,0x822b0001}, {0x3c,0x9b2b6010},
    {0x40,0x8210400d}, {0x44,0xc222c008}, {0x48,0xc2002794}, {0x4c,0xda004008}, {0x50,0xc200279c}, {0x54,0xd6004008}, {0x58,0xc2002798}, {0x5c,0x9a0b6fff},
    {0x60,0xd8004008}, {0x64,0x832b6002}, {0x68,0x8200400d}, {0x6c,0x960aefff}, {0x70,0x980b2fff}, {0x74,0x8200400b}, {0x78,0x992b2002}, {0x7c,0x80a0400c},

    {0xf0,0x7e},
    {0x00,0x18800003}, {0x04,0x8220400c}, {0x08,0x82102000}, {0x0c,0xd8002790}, {0x10,0x9b306001}, {0x14,0xc2030008}, {0x18,0x9a0b6fff}, {0x1c,0x82087000},
    {0x20,0x8210400d}, {0x24,0xc2230008}, {0x28,0xd8002790}, {0x2c,0xc2030008}, {0x30,0x1b000020}, {0x34,0x8210400d}, {0x38,0x81c3e008}, {0x3c,0xc2230008},
    {0x40,0x9de3bf98}, {0x44,0xa2102000}, {0x48,0xa12c6002}, {0x4c,0xc2002794}, {0x50,0xc2004010}, {0x54,0x80a06000}, {0x58,0x0280001f}, {0x5c,0x0303ffc3},
    {0x60,0xc2002798}, {0x64,0xc2004010}, {0x68,0x80a06000}, {0x6c,0x0280000c}, {0x70,0x01000000}, {0x74,0x8330600d}, {0x78,0x80886001}, {0x7c,0x12800008},

    {0xf0,0x7f},
    {0x00,0x01000000}, {0x04,0xc200279c}, {0x08,0xda004010}, {0x0c,0x8333600d}, {0x10,0x80886001}, {0x14,0x02800006}, {0x18,0x80a36000}, {0x1c,0x7fffff73},
    {0x20,0x90100011}, {0x24,0x10800010}, {0x28,0xc2002794}, {0x2c,0x02800006}, {0x30,0x01000000}, {0x34,0x7fffffa8}, {0x38,0x90100011}, {0x3c,0x1080000a},
    {0x40,0xc2002794}, {0x44,0x7fffff77}, {0x48,0x90100011}, {0x4c,0x10800006}, {0x50,0xc2002794}, {0x54,0x821063ff}, {0x58,0xda002790}, {0x5c,0xc2234010},
    {0x60,0xc2002794}, {0x64,0xc2004010}, {0x68,0x8330600c}, {0x6c,0x80886001}, {0x70,0x02800007}, {0x74,0xa2046001}, {0x78,0xc2002790}, {0x7c,0xda004010},

    {0xf0,0x80},
    {0x00,0x19000004}, {0x04,0x9a13400c}, {0x08,0xda204010}, {0x0c,0x80a46009}, {0x10,0x04bfffcf}, {0x14,0xa12c6002}, {0x18,0x81c7e008}, {0x1c,0x81e80000},
    {0x20,0xd6020000}, {0x24,0xd8024000}, {0x28,0x9132e010}, {0x2c,0x95332010}, {0x30,0x900a2fff}, {0x34,0x940aafff}, {0x38,0x03000007}, {0x3c,0x9a22000a},
    {0x40,0x821063ff}, {0x44,0x940b0001}, {0x48,0x900ac001}, {0x4c,0x9022000a}, {0x50,0x9a5b400d}, {0x54,0x905a0008}, {0x58,0x81c3e008}, {0x5c,0x90034008},
    {0x60,0x031fffff}, {0x64,0x9002200c}, {0x68,0x821063ff}, {0x6c,0x9a102063}, {0x70,0xc2220000}, {0x74,0x9a837fff}, {0x78,0x1cbffffe}, {0x7c,0x90022004},

    {0xf0,0x81},
    {0x00,0x81c3e008}, {0x04,0x01000000}, {0x08,0x031fffff}, {0x0c,0x821063ff}, {0x10,0xc2222008}, {0x14,0x92102000}, {0x18,0x96100008}, {0x1c,0x94102000},
    {0x20,0x9a02e00c}, {0x24,0xd8034000}, {0x28,0xc2022008}, {0x2c,0x80a30001}, {0x30,0x16800005}, {0x34,0x9a036004}, {0x38,0xd8222008}, {0x3c,0xd4220000},
    {0x40,0xd2222004}, {0x44,0x9402a001}, {0x48,0x80a2a009}, {0x4c,0x24bffff7}, {0x50,0xd8034000}, {0x54,0x92026001}, {0x58,0x80a26009}, {0x5c,0x04bffff0},
    {0x60,0x9602e028}, {0x64,0xda022008}, {0x68,0x03200000}, {0x6c,0x8238400d}, {0x70,0x80a00001}, {0x74,0x81c3e008}, {0x78,0x90402000}, {0x7c,0xc2022004},

    {0xf0,0x82},
    {0x00,0x9b286002}, {0x04,0x9a034001}, {0x08,0x031fffff}, {0x0c,0x9b2b6003}, {0x10,0x9a034008}, {0x14,0x981063ff}, {0x18,0x9a03600c}, {0x1c,0x82102009},
    {0x20,0xd8234000}, {0x24,0x82807fff}, {0x28,0x1cbffffe}, {0x2c,0x9a036004}, {0x30,0xc2020000}, {0x34,0x83286002}, {0x38,0x82004008}, {0x3c,0x8200600c},
    {0x40,0x9a102009}, {0x44,0xd8204000}, {0x48,0x9a837fff}, {0x4c,0x1cbffffe}, {0x50,0x82006028}, {0x54,0x81c3e008}, {0x58,0x01000000}, {0x5c,0x98100008},
    {0x60,0x90102008}, {0x64,0x9a102100}, {0x68,0x832b4008}, {0x6c,0x80a30001}, {0x70,0x14800006}, {0x74,0x01000000}, {0x78,0x90023fff}, {0x7c,0x80a22000},

    {0xf0,0x83},
    {0x00,0x14bffffb}, {0x04,0x832b4008}, {0x08,0x81c3e008}, {0x0c,0x01000000}, {0x10,0x9de3bdd0}, {0x14,0xae07be58}, {0x18,0x7fffffb2}, {0x1c,0x90100017},
    {0x20,0xa6102000}, {0x24,0xa12ce002}, {0x28,0xd2002790}, {0x2c,0xc2024010}, {0x30,0x8330600f}, {0x34,0x80886001}, {0x38,0x2280000f}, {0x3c,0xd000245c},
    {0x40,0xc2002794}, {0x44,0x90004010}, {0x48,0xc2004010}, {0x4c,0x8330600d}, {0x50,0x80886001}, {0x54,0x02800004}, {0x58,0x92024010}, {0x5c,0x10800006},
    {0x60,0xd000245c}, {0x64,0x7fffff8f}, {0x68,0x01000000}, {0x6c,0x7fffffdc}, {0x70,0x01000000}, {0x74,0xc2002358}, {0x78,0x9807bff8}, {0x7c,0x825a0001},

    {0xf0,0x84},
    {0x00,0x9a04000c}, {0x04,0xa604e001}, {0x08,0x80a4e009}, {0x0c,0x04bfffe6}, {0x10,0xc2237e38}, {0x14,0xac10000c}, {0x18,0xa6102000}, {0x1c,0xa8102000},
    {0x20,0xea002790}, {0x24,0x0303ffc3}, {0x28,0xda054014}, {0x2c,0x821063ff}, {0x30,0x80a34001}, {0x34,0x22800014}, {0x38,0xa604e001}, {0x3c,0xa2102000},
    {0x40,0xc2002514}, {0x44,0x80a44001}, {0x48,0x3a80000f}, {0x4c,0xa604e001}, {0x50,0xa005be6c}, {0x54,0xa4102768}, {0x58,0x90100012}, {0x5c,0x7fffff71},
    {0x60,0x92054014}, {0x64,0xd0240000}, {0x68,0xa2046001}, {0x6c,0xc2002514}, {0x70,0x80a44001}, {0x74,0xa404a004}, {0x78,0x0abffff8}, {0x7c,0xa0042028},

    {0xf0,0x85},
    {0x00,0xa604e001}, {0x04,0xa8052004}, {0x08,0x80a4e009}, {0x0c,0x04bfffe5}, {0x10,0xac05a004}, {0x14,0xa2102000}, {0x18,0xc2002514}, {0x1c,0x80a44001},
    {0x20,0x1a80002d}, {0x24,0x01000000}, {0x28,0x7fffff78}, {0x2c,0x90100017}, {0x30,0x80a22000}, {0x34,0xa0046001}, {0x38,0x02800027}, {0x3c,0x90100017},
    {0x40,0xd807be58}, {0x44,0x832b2002}, {0x48,0x8200401e}, {0x4c,0xc2007e30}, {0x50,0xda002230}, {0x54,0x9a034001}, {0x58,0xc2002548}, {0x5c,0x9a5b4001},
    {0x60,0xc2002334}, {0x64,0x82006001}, {0x68,0x81800000}, {0x6c,0x01000000}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x9a734001}, {0x7c,0xc207be60},

    {0xf0,0x86},
    {0x00,0x80a0400d}, {0x04,0x98032001}, {0x08,0xc207be5c}, {0x0c,0x992b201c}, {0x10,0x0a800007}, {0x14,0x95286002}, {0x18,0xc202a768}, {0x1c,0x1b3c0000},
    {0x20,0x8210400d}, {0x24,0x10800008}, {0x28,0xc222a768}, {0x2c,0xda02a768}, {0x30,0x033c0000}, {0x34,0x822b4001}, {0x38,0x8210400c}, {0x3c,0x7fffff70},
    {0x40,0xc222a768}, {0x44,0xc2002514}, {0x48,0x80a40001}, {0x4c,0x0abfffd7}, {0x50,0xa2100010}, {0x54,0x81c7e008}, {0x58,0x81e80000}, {0x5c,0x92102000},
    {0x60,0xc2002514}, {0x64,0x80a24001}, {0x68,0x1a800037}, {0x6c,0x0303ffff}, {0x70,0x901063ff}, {0x74,0x952a6002}, {0x78,0xc202a768}, {0x7c,0x8330601c},

    {0xf0,0x87},
    {0x00,0x80a00001}, {0x04,0x9a603fff}, {0x08,0x8218600f}, {0x0c,0x80a00001}, {0x10,0x82603fff}, {0x14,0x80934001}, {0x18,0x22800027}, {0x1c,0x92026001},
    {0x20,0x9a102001}, {0x24,0x96102000}, {0x28,0x992ae002}, {0x2c,0xc2032768}, {0x30,0x8330601c}, {0x34,0x80a0400d}, {0x38,0x02800013}, {0x3c,0x80a2e00a},
    {0x40,0xc2002794}, {0x44,0xc200400c}, {0x48,0x8330601c}, {0x4c,0x80a0400d}, {0x50,0x0280000d}, {0x54,0x80a2e00a}, {0x58,0xc2002798}, {0x5c,0xc200400c},
    {0x60,0x8330601c}, {0x64,0x80a0400d}, {0x68,0x02800007}, {0x6c,0x80a2e00a}, {0x70,0x9602e001}, {0x74,0x80a2e009}, {0x78,0x08bfffed}, {0x7c,0x992ae002},

    {0xf0,0x88},
    {0x00,0x80a2e00a}, {0x04,0x22800007}, {0x08,0xc202a768}, {0x0c,0x9a036001}, {0x10,0x80a3600a}, {0x14,0x08bfffe5}, {0x18,0x96102000}, {0x1c,0xc202a768},
    {0x20,0x9b2b601c}, {0x24,0x82084008}, {0x28,0x8210400d}, {0x2c,0xc222a768}, {0x30,0x92026001}, {0x34,0xc2002514}, {0x38,0x80a24001}, {0x3c,0x0abfffcf},
    {0x40,0x952a6002}, {0x44,0x81c3e008}, {0x48,0x01000000}, {0x4c,0x98102000}, {0x50,0x9b2b2002}, {0x54,0x98032001}, {0x58,0xc2002790}, {0x5c,0x80a32009},
    {0x60,0x08bffffc}, {0x64,0xc020400d}, {0x68,0x98102000}, {0x6c,0xc2002514}, {0x70,0x80a30001}, {0x74,0x1a800012}, {0x78,0x033fffc7}, {0x7c,0x941063ff},

    {0xf0,0x89},
    {0x00,0x832b2002}, {0x04,0xda006768}, {0x08,0x8333601c}, {0x0c,0x82007fff}, {0x10,0x98032001}, {0x14,0x80a06009}, {0x18,0x97286002}, {0x1c,0x18800004},
    {0x20,0x9a0b400a}, {0x24,0xc2002790}, {0x28,0xda20400b}, {0x2c,0xc2002514}, {0x30,0x80a30001}, {0x34,0x0abffff4}, {0x38,0x832b2002}, {0x3c,0x81c3e008},
    {0x40,0x01000000}, {0x44,0x9de3bf98}, {0x48,0x92102000}, {0x4c,0x94026001}, {0x50,0x80a2a009}, {0x54,0x18800068}, {0x58,0x9610000a}, {0x5c,0x033c003f},
    {0x60,0x901063ff}, {0x64,0xf6002790}, {0x68,0xb32ae002}, {0x6c,0xfa06c019}, {0x70,0x80a76000}, {0x74,0x2280005c}, {0x78,0x9602e001}, {0x7c,0xb52a6002},

    {0xf0,0x8a},
    {0x00,0xc206c01a}, {0x04,0x80a06000}, {0x08,0x22800057}, {0x0c,0x9602e001}, {0x10,0xda002794}, {0x14,0xf0034019}, {0x18,0x80a62000}, {0x1c,0x22800052},
    {0x20,0x9602e001}, {0x24,0xf803401a}, {0x28,0x80a72000}, {0x2c,0x2280004e}, {0x30,0x9602e001}, {0x34,0x83306010}, {0x38,0xbb376010}, {0x3c,0x98086fff},
    {0x40,0x9e0f6fff}, {0x44,0x80a3000f}, {0x48,0x16800009}, {0x4c,0xbb372010}, {0x50,0x83362010}, {0x54,0xba0f6fff}, {0x58,0x82086fff}, {0x5c,0x80a74001},
    {0x60,0x3480000d}, {0x64,0xc206c01a}, {0x68,0x80a3000f}, {0x6c,0x2480003e}, {0x70,0x9602e001}, {0x74,0xbb372010}, {0x78,0x83362010}, {0x7c,0xba0f6fff},

    {0xf0,0x8b},
    {0x00,0x82086fff}, {0x04,0x80a74001}, {0x08,0x36800037}, {0x0c,0x9602e001}, {0x10,0xc206c01a}, {0x14,0xfa06c019}, {0x18,0xb0086fff}, {0x1c,0xb80f6fff},
    {0x20,0x80a6001c}, {0x24,0x1680000a}, {0x28,0x01000000}, {0x2c,0xfa034019}, {0x30,0xc203401a}, {0x34,0x82086fff}, {0x38,0xba0f6fff}, {0x3c,0x80a0401d},
    {0x40,0x3480000e}, {0x44,0xfa16c01a}, {0x48,0x80a6001c}, {0x4c,0x24800026}, {0x50,0x9602e001}, {0x54,0xc2002794}, {0x58,0xfa004019}, {0x5c,0xc200401a},
    {0x60,0x82086fff}, {0x64,0xba0f6fff}, {0x68,0x80a0401d}, {0x6c,0x3680001e}, {0x70,0x9602e001}, {0x74,0xfa16c01a}, {0x78,0xf806c019}, {0x7c,0xba0f6fff},

    {0xf0,0x8c},
    {0x00,0xbb2f6010}, {0x04,0x820f0008}, {0x08,0x8210401d}, {0x0c,0xc226c019}, {0x10,0xf6002790}, {0x14,0xc206c01a}, {0x18,0x3b03ffc0}, {0x1c,0xb80f001d},
    {0x20,0x82084008}, {0x24,0x8210401c}, {0x28,0xc226c01a}, {0x2c,0xf8002790}, {0x30,0xf6070019}, {0x34,0xfa07001a}, {0x38,0xba0f6fff}, {0x3c,0x820ef000},
    {0x40,0x8210401d}, {0x44,0xc2270019}, {0x48,0xfa002790}, {0x4c,0xc207401a}, {0x50,0x82087000}, {0x54,0xb60eefff}, {0x58,0x8210401b}, {0x5c,0xc227401a},
    {0x60,0x9602e001}, {0x64,0x80a2e009}, {0x68,0x28bfffa0}, {0x6c,0xf6002790}, {0x70,0x80a2a009}, {0x74,0x08bfff96}, {0x78,0x9210000a}, {0x7c,0x81c7e008},

    {0xf0,0x8d},
    {0x00,0x81e80000}, {0x04,0x9de3bf98}, {0x08,0xa6102000}, {0x0c,0xda002244}, {0x10,0x80a36000}, {0x14,0x02800033}, {0x18,0xa12ce002}, {0x1c,0xe4002790},
    {0x20,0xc2048010}, {0x24,0x80a06000}, {0x28,0x22800004}, {0x2c,0xc204282c}, {0x30,0x1080002c}, {0x34,0xc024282c}, {0x38,0x80a06000}, {0x3c,0x2280000b},
    {0x40,0xc2002518}, {0x44,0xc2002794}, {0x48,0xc2004010}, {0x4c,0x1b000008}, {0x50,0x8210400d}, {0x54,0xc2248010}, {0x58,0xc204282c}, {0x5c,0x82007fff},
    {0x60,0x10800020}, {0x64,0xc224282c}, {0x68,0x80a0400d}, {0x6c,0x2a80001e}, {0x70,0xa604e001}, {0x74,0xe2002794}, {0x78,0xc2044010}, {0x7c,0x80a06000},

    {0xf0,0x8e},
    {0x00,0x22800019}, {0x04,0xa604e001}, {0x08,0x8330600d}, {0x0c,0x80886001}, {0x10,0x32800015}, {0x14,0xa604e001}, {0x18,0xd2002798}, {0x1c,0xc2024010},
    {0x20,0x80a06000}, {0x24,0x22800010}, {0x28,0xa604e001}, {0x2c,0x92024010}, {0x30,0x7ffffe3c}, {0x34,0x90044010}, {0x38,0xc200224c}, {0x3c,0x80a20001},
    {0x40,0x38800009}, {0x44,0xa604e001}, {0x48,0xc2002248}, {0x4c,0xc224282c}, {0x50,0xc2044010}, {0x54,0x1b000008}, {0x58,0x8210400d}, {0x5c,0xc2248010},
    {0x60,0xa604e001}, {0x64,0x80a4e009}, {0x68,0x24bfffca}, {0x6c,0xda002244}, {0x70,0x81c7e008}, {0x74,0x81e80000}, {0x78,0x9de3bf98}, {0x7c,0xc2002514},

    {0xf0,0x8f},
    {0x00,0x80a06000}, {0x04,0x22800006}, {0x08,0xc2002200}, {0x0c,0xc2002314}, {0x10,0x82200001}, {0x14,0x10800062}, {0x18,0xc2202538}, {0x1c,0x80a06000},
    {0x20,0x1280005f}, {0x24,0x01000000}, {0x28,0xfa002314}, {0x2c,0x80a76000}, {0x30,0x0280005b}, {0x34,0x01000000}, {0x38,0xc2002538}, {0x3c,0x82006001},
    {0x40,0x80a0401d}, {0x44,0x06800056}, {0x48,0xc2202538}, {0x4c,0x9e102001}, {0x50,0xc20022fc}, {0x54,0x80a3c001}, {0x58,0x18800051}, {0x5c,0xc0202538},
    {0x60,0x13000017}, {0x64,0x9a102001}, {0x68,0xc20022f8}, {0x6c,0x80a34001}, {0x70,0x18800046}, {0x74,0xf20be37f}, {0x78,0x0300003f}, {0x7c,0x941063ff},

    {0xf0,0x90},
    {0x00,0x21000017}, {0x04,0x961263f8}, {0x08,0x901261d0}, {0x0c,0x98102001}, {0x10,0xf8002548}, {0x14,0x80a72008}, {0x18,0xf400234c}, {0x1c,0x08800005},
    {0x20,0x82064019}, {0x24,0xc210400b}, {0x28,0x10800003}, {0x2c,0xb6004001}, {0x30,0xf610400b}, {0x34,0xb0064019}, {0x38,0x81800000}, {0x3c,0x01000000},
    {0x40,0x01000000}, {0x44,0x01000000}, {0x48,0xba76c01c}, {0x4c,0xc2160008}, {0x50,0xb6a74001}, {0x54,0x22800027}, {0x58,0xc200247c}, {0x5c,0x80a6e000},
    {0x60,0x04800007}, {0x64,0x832b001a}, {0x68,0x80a6c001}, {0x6c,0x3480000c}, {0x70,0xb73ec01a}, {0x74,0x1080000a}, {0x78,0xb6102001}, {0x7c,0x36800009},

    {0xf0,0x91},
    {0x00,0xb41421d0}, {0x04,0x832b001a}, {0x08,0x82200001}, {0x0c,0x80a6c001}, {0x10,0x36800003}, {0x14,0xb6103fff}, {0x18,0xb73ec01a}, {0x1c,0xb41421d0},
    {0x20,0xc216001a}, {0x24,0xb606c001}, {0x28,0x808e6001}, {0x2c,0x0280000a}, {0x30,0x83366001}, {0x34,0xb9286002}, {0x38,0xc207001a}, {0x3c,0x3b3fffc0},
    {0x40,0x8208401d}, {0x44,0xba0ec00a}, {0x48,0x8200401d}, {0x4c,0x10800008}, {0x50,0xc227001a}, {0x54,0x83286002}, {0x58,0xfa00401a}, {0x5c,0xb92ee010},
    {0x60,0xba0f400a}, {0x64,0xb807001d}, {0x68,0xf820401a}, {0x6c,0xc200247c}, {0x70,0xb2064001}, {0x74,0x9a036001}, {0x78,0xc20022f8}, {0x7c,0x80a34001},

    {0xf0,0x92},
    {0x00,0x28bfffc5}, {0x04,0xf8002548}, {0x08,0x9e03e001}, {0x0c,0xc20022fc}, {0x10,0x80a3c001}, {0x14,0x08bfffb5}, {0x18,0x9a102001}, {0x1c,0x81c7e008},
    {0x20,0x81e80000}, {0x24,0xc0202514}, {0x28,0x9a102000}, {0x2c,0x832b6002}, {0x30,0xc2020001}, {0x34,0x80a06000}, {0x38,0x02800005}, {0x3c,0x9a036001},
    {0x40,0xc2002514}, {0x44,0x82006001}, {0x48,0xc2202514}, {0x4c,0x80a36009}, {0x50,0x04bffff8}, {0x54,0x832b6002}, {0x58,0x81c3e008}, {0x5c,0x01000000},
    {0x60,0x9de3bf98}, {0x64,0xa8102000}, {0x68,0xa0102000}, {0x6c,0xc200235c}, {0x70,0x80a06000}, {0x74,0x32800004}, {0x78,0xc0242768}, {0x7c,0x1080005d},

    {0xf0,0x93},
    {0x00,0xc2002790}, {0x04,0xc2002790}, {0x08,0xc2004010}, {0x0c,0x80a06000}, {0x10,0x02800019}, {0x14,0xda042854}, {0x18,0x03300000}, {0x1c,0x808b4001},
    {0x20,0x32800010}, {0x24,0xc2002790}, {0x28,0xda002514}, {0x2c,0x80a36000}, {0x30,0x22800053}, {0x34,0xa8052001}, {0x38,0x8203400d}, {0x3c,0x8200400d},
    {0x40,0x82007ffd}, {0x44,0xda00235c}, {0x48,0x9b334001}, {0x4c,0x9a0b6007}, {0x50,0x03200000}, {0x54,0x9a134001}, {0x58,0xda242854}, {0x5c,0xc2002790},
    {0x60,0xc2004010}, {0x64,0x80a06000}, {0x68,0x32800007}, {0x6c,0xc2042854}, {0x70,0xda042854}, {0x74,0x03200000}, {0x78,0x822b4001}, {0x7c,0xc2242854},

    {0xf0,0x94},
    {0x00,0xc2042854}, {0x04,0x1b300000}, {0x08,0x9a08400d}, {0x0c,0x19200000}, {0x10,0x80a3400c}, {0x14,0x12800019}, {0x18,0xa40860ff}, {0x1c,0x98102000},
    {0x20,0x832b2002}, {0x24,0xc2006790}, {0x28,0xc2004010}, {0x2c,0x80a06000}, {0x30,0x0280000b}, {0x34,0x9b30600d}, {0x38,0x808b6001}, {0x3c,0x12800009},
    {0x40,0x80a30012}, {0x44,0x98032001}, {0x48,0x80a30012}, {0x4c,0x24bffff6}, {0x50,0x832b2002}, {0x54,0x10800006}, {0x58,0xc2042854}, {0x5c,0x80a30012},
    {0x60,0x24800027}, {0x64,0xa8052001}, {0x68,0xc2042854}, {0x6c,0x1b100000}, {0x70,0x8210400d}, {0x74,0xc2242854}, {0x78,0xa32ca002}, {0x7c,0xd0046790},

    {0xf0,0x95},
    {0x00,0xc2020010}, {0x04,0x80a06000}, {0x08,0x12800006}, {0x0c,0x03100000}, {0x10,0xda042854}, {0x14,0x822b4001}, {0x18,0x10800018}, {0x1c,0xc2242854},
    {0x20,0xe6042854}, {0x24,0x8334e01e}, {0x28,0x80886001}, {0x2c,0x22800014}, {0x30,0xa8052001}, {0x34,0x80a4a000}, {0x38,0x2280000e}, {0x3c,0xc2046790},
    {0x40,0xd204678c}, {0x44,0x90020010}, {0x48,0x7ffffd56}, {0x4c,0x92024010}, {0x50,0x80a22008}, {0x54,0x34800007}, {0x58,0xc2046790}, {0x5c,0x820cfff0},
    {0x60,0x9a04bfff}, {0x64,0x8210400d}, {0x68,0xc2242854}, {0x6c,0xc2046790}, {0x70,0xc2004010}, {0x74,0xc2242768}, {0x78,0xa8052001}, {0x7c,0x80a52009},

    {0xf0,0x96},
    {0x00,0x04bfff9b}, {0x04,0xa0042004}, {0x08,0x81c7e008}, {0x0c,0x81e80000}, {0x10,0x8332a01f}, {0x14,0x8200400a}, {0x18,0x83386001}, {0x1c,0x80a24001},
    {0x20,0x26800015}, {0x24,0x90102000}, {0x28,0x9a024001}, {0x2c,0x80a36008}, {0x30,0x24800004}, {0x34,0x92224001}, {0x38,0x1080000f}, {0x3c,0x90102000},
    {0x40,0x80a2400d}, {0x44,0x1480000b}, {0x48,0x912a2002}, {0x4c,0x832a6002}, {0x50,0xc2006790}, {0x54,0xc2004008}, {0x58,0x80a06000}, {0x5c,0x02bffff7},
    {0x60,0x92026001}, {0x64,0x80a2400d}, {0x68,0x04bffffa}, {0x6c,0x832a6002}, {0x70,0x90102001}, {0x74,0x81c3e008}, {0x78,0x01000000}, {0x7c,0x9de3bf98},

    {0xf0,0x97},
    {0x00,0x92100019}, {0x04,0x90100018}, {0x08,0x7fffffe2}, {0x0c,0x9410001a}, {0x10,0xa4100018}, {0x14,0x80a22000}, {0x18,0x12800028}, {0x1c,0x92100019},
    {0x20,0xa33ea01f}, {0x24,0x8334601f}, {0x28,0x82068001}, {0x2c,0x83386001}, {0x30,0x80a64001}, {0x34,0x2680000e}, {0x38,0x8334601f}, {0x3c,0x82264001},
    {0x40,0x83286002}, {0x44,0xda006790}, {0x48,0x832e2002}, {0x4c,0xc2034001}, {0x50,0x80a06000}, {0x54,0x02800019}, {0x58,0x92103fff}, {0x5c,0x10800004},
    {0x60,0x8334601f}, {0x64,0x10800015}, {0x68,0x92100018}, {0x6c,0x82068001}, {0x70,0x83386001}, {0x74,0xa0102001}, {0x78,0x80a40001}, {0x7c,0x1480000e},

    {0xf0,0x98},
    {0x00,0x90100012}, {0x04,0xb0064010}, {0x08,0x92100018}, {0x0c,0x7fffffc1}, {0x10,0x9410001a}, {0x14,0x8334601f}, {0x18,0x82068001}, {0x1c,0xa0042001},
    {0x20,0x80a22000}, {0x24,0x12bffff0}, {0x28,0x83386001}, {0x2c,0x10bffff4}, {0x30,0x80a40001}, {0x34,0x92103fff}, {0x38,0x81c7e008}, {0x3c,0x91e80009},
    {0x40,0x9de3bf98}, {0x44,0xa32e2002}, {0x48,0xc20467b4}, {0x4c,0x80a06000}, {0x50,0x0280001c}, {0x54,0xb0102001}, {0x58,0x8336a01f}, {0x5c,0x82068001},
    {0x60,0xb5386001}, {0x64,0xa026401a}, {0x68,0xb2066001}, {0x6c,0xc20ea35f}, {0x70,0xb4584001}, {0x74,0x80a40019}, {0x78,0x14800011}, {0x7c,0xb0102000},

    {0xf0,0x99},
    {0x00,0x832c2002}, {0x04,0xd0006790}, {0x08,0x90020011}, {0x0c,0x7ffffce5}, {0x10,0x920467b4}, {0x14,0x80a2001a}, {0x18,0x04800003}, {0x1c,0xa0042001},
    {0x20,0xb0062001}, {0x24,0x80a40019}, {0x28,0x04bffff7}, {0x2c,0x832c2002}, {0x30,0x80a62001}, {0x34,0x14800003}, {0x38,0xb0102001}, {0x3c,0xb0102000},
    {0x40,0x81c7e008}, {0x44,0x81e80000}, {0x48,0x9de3bf48}, {0x4c,0xc2082360}, {0x50,0x80a06000}, {0x54,0x0280007c}, {0x58,0xba102000}, {0x5c,0xa6102000},
    {0x60,0xda04e854}, {0x64,0x8333601e}, {0x68,0x80886001}, {0x6c,0x22800073}, {0x70,0xba076001}, {0x74,0x83336008}, {0x78,0x820860ff}, {0x7c,0x80a06002},

    {0xf0,0x9a},
    {0x00,0x0480000c}, {0x04,0xa4102003}, {0x08,0x82006002}, {0x0c,0xa4106001}, {0x10,0x80a4a009}, {0x14,0x04800005}, {0x18,0x80a4a002}, {0x1c,0x10800005},
    {0x20,0xa4102009}, {0x24,0x80a4a002}, {0x28,0x0480005d}, {0x2c,0x1b3fffc0}, {0x30,0x94100012}, {0x34,0xd20ce857}, {0x38,0x7fffff91}, {0x3c,0x9010001d},
    {0x40,0xa2100008}, {0x44,0x94100012}, {0x48,0x92946000}, {0x4c,0x04800051}, {0x50,0x9010001d}, {0x54,0x7fffffbb}, {0x58,0x01000000}, {0x5c,0x80a22000},
    {0x60,0x32bffff1}, {0x64,0xa404bffe}, {0x68,0xad3ca01f}, {0x6c,0x8335a01f}, {0x70,0x82048001}, {0x74,0x83386001}, {0x78,0x9a044001}, {0x7c,0xa0244001},

    {0xf0,0x9b},
    {0x00,0x80a4000d}, {0x04,0x1480000f}, {0x08,0x9610000d}, {0x0c,0x9807bff8}, {0x10,0x832c2002}, {0x14,0xda006790}, {0x18,0xc2134013}, {0x1c,0x82086fff},
    {0x20,0xc2233fd8}, {0x24,0xc2034013}, {0x28,0x82086fff}, {0x2c,0xc2233fb0}, {0x30,0xa0042001}, {0x34,0x80a4000b}, {0x38,0x04bffff6}, {0x3c,0x98032004},
    {0x40,0x92100012}, {0x44,0x7ffff22a}, {0x48,0x9007bfd0}, {0x4c,0x9007bfa8}, {0x50,0x7ffff227}, {0x54,0x92100012}, {0x58,0x9935a01f}, {0x5c,0x9804800c},
    {0x60,0x993b2001}, {0x64,0x8207bff8}, {0x68,0x952b2002}, {0x6c,0x94028001}, {0x70,0xda02bfd8}, {0x74,0xd604e768}, {0x78,0x9a0b6fff}, {0x7c,0x0303ffc0},

    {0xf0,0x9c},
    {0x00,0x9b2b6010}, {0x04,0x822ac001}, {0x08,0x8210400d}, {0x0c,0xc224e768}, {0x10,0xda02bfb0}, {0x14,0x9a0b6fff}, {0x18,0x82087000}, {0x1c,0x8210400d},
    {0x20,0xc224e768}, {0x24,0x832c6002}, {0x28,0xda006790}, {0x2c,0x8204400c}, {0x30,0xa024400c}, {0x34,0x80a40001}, {0x38,0x031fffff}, {0x3c,0xea034013},
    {0x40,0xae1063ff}, {0x44,0x14800011}, {0x48,0x832c2002}, {0x4c,0xe8006790}, {0x50,0x90050013}, {0x54,0x7ffffc73}, {0x58,0x9204e768}, {0x5c,0x8335a01f},
    {0x60,0x82048001}, {0x64,0x83386001}, {0x68,0xa0042001}, {0x6c,0x80a20017}, {0x70,0x16800004}, {0x74,0x82044001}, {0x78,0xae100008}, {0x7c,0xea050013},

    {0xf0,0x9d},
    {0x00,0x10bffff1}, {0x04,0x80a40001}, {0x08,0x10800004}, {0x0c,0xea24e768}, {0x10,0x10bfffa5}, {0x14,0xa404bffe}, {0x18,0x1b3fffc0}, {0x1c,0xc204e854},
    {0x20,0x9a1360ff}, {0x24,0x8208400d}, {0x28,0x9b2ca008}, {0x2c,0x8210400d}, {0x30,0xc224e854}, {0x34,0xba076001}, {0x38,0x80a76009}, {0x3c,0x04bfff89},
    {0x40,0xa604e004}, {0x44,0x81c7e008}, {0x48,0x81e80000}, {0x4c,0x9de3bf98}, {0x50,0xa6102000}, {0x54,0xa12ce002}, {0x58,0xda042768}, {0x5c,0x80a36000},
    {0x60,0x12800008}, {0x64,0x82102001}, {0x68,0xc02427b4}, {0x6c,0xda002550}, {0x70,0x83284013}, {0x74,0x822b4001}, {0x78,0x1080001c}, {0x7c,0xc2202550},

    {0xf0,0x9e},
    {0x00,0xe80427b4}, {0x04,0x80a52000}, {0x08,0x12800004}, {0x0c,0xa5284013}, {0x10,0x10800016}, {0x14,0xda2427b4}, {0x18,0xe2002550}, {0x1c,0x808c4012},
    {0x20,0x32800011}, {0x24,0xc2042768}, {0x28,0x8333600c}, {0x2c,0x80886001}, {0x30,0x3280000d}, {0x34,0xc2042768}, {0x38,0x90042768}, {0x3c,0x7ffffc39},
    {0x40,0x920427b4}, {0x44,0xc2002354}, {0x48,0x80a20001}, {0x4c,0x1a800004}, {0x50,0x82144012}, {0x54,0x10800005}, {0x58,0xe8242768}, {0x5c,0xc2202550},
    {0x60,0xc2042768}, {0x64,0xc22427b4}, {0x68,0xa604e001}, {0x6c,0x80a4e009}, {0x70,0x08bfffda}, {0x74,0xa12ce002}, {0x78,0x81c7e008}, {0x7c,0x81e80000},

    {0xf0,0x9f},
    {0x00,0x9de3bf98}, {0x04,0xc2060000}, {0x08,0xbb30600c}, {0x0c,0xb9306010}, {0x10,0xb80f2fff}, {0x14,0xb08f6001}, {0x18,0xb6086fff}, {0x1c,0x12800014},
    {0x20,0x9f30601c}, {0x24,0xc250229e}, {0x28,0xfa5022a2}, {0x2c,0x8226c001}, {0x30,0xba27001d}, {0x34,0xf850229c}, {0x38,0xf65022a0}, {0x3c,0x8258401c},
    {0x40,0xba5f401b}, {0x44,0x82006800}, {0x48,0xba076800}, {0x4c,0xb938601f}, {0x50,0xb73f601f}, {0x54,0xb9372014}, {0x58,0xb736e014}, {0x5c,0x8200401c},
    {0x60,0xba07401b}, {0x64,0xb738600c}, {0x68,0xb93f600c}, {0x6c,0xf4002324}, {0x70,0xf2002328}, {0x74,0xfa002308}, {0x78,0xc2002300}, {0x7c,0xb65ec01a},

    {0xf0,0xa0},
    {0x00,0xbb2f6006}, {0x04,0xb85f0019}, {0x08,0x83286006}, {0x0c,0x9b3ee01f}, {0x10,0x81836000}, {0x14,0x01000000}, {0x18,0x01000000}, {0x1c,0x01000000},
    {0x20,0xb67ec01d}, {0x24,0x9b3f201f}, {0x28,0x81836000}, {0x2c,0x01000000}, {0x30,0x01000000}, {0x34,0x01000000}, {0x38,0xb87f0001}, {0x3c,0x80a62000},
    {0x40,0x32800031}, {0x44,0x3b03ffc0}, {0x48,0xc20022a4}, {0x4c,0x80a06000}, {0x50,0x0280000a}, {0x54,0x80a6e000}, {0x58,0xc25022a6}, {0x5c,0x80a6c001},
    {0x60,0x14800031}, {0x64,0xb0102000}, {0x68,0xc25022a4}, {0x6c,0x80a6c001}, {0x70,0x0680002d}, {0x74,0x80a6e000}, {0x78,0x24800002}, {0x7c,0xb6102001},

    {0xf0,0xa1},
    {0x00,0x80a6c01a}, {0x04,0x3a800002}, {0x08,0xb606bfff}, {0x0c,0xc20022a8}, {0x10,0x80a06000}, {0x14,0x0280000a}, {0x18,0x80a72000}, {0x1c,0xc25022aa},
    {0x20,0x80a70001}, {0x24,0x14800020}, {0x28,0xb0102000}, {0x2c,0xc25022a8}, {0x30,0x80a70001}, {0x34,0x0680001c}, {0x38,0x80a72000}, {0x3c,0x24800002},
    {0x40,0xb8102001}, {0x44,0x80a70019}, {0x48,0x3a800002}, {0x4c,0xb8067fff}, {0x50,0xc20023c8}, {0x54,0x80886002}, {0x58,0x32800002}, {0x5c,0xb626801b},
    {0x60,0x80886004}, {0x64,0x32800002}, {0x68,0xb826401c}, {0x6c,0x80886008}, {0x70,0x02800005}, {0x74,0x3b03ffc0}, {0x78,0xb61ec01c}, {0x7c,0xb81f001b},

    {0xf0,0xa2},
    {0x00,0xb61ec01c}, {0x04,0x832ee010}, {0x08,0x8208401d}, {0x0c,0xbb2be01c}, {0x10,0xba074001}, {0x14,0x0300003f}, {0x18,0x821063ff}, {0x1c,0x820f0001},
    {0x20,0xb0074001}, {0x24,0x81c7e008}, {0x28,0x81e80000}, {0x2c,0x9de3bf98}, {0x30,0xda002514}, {0x34,0xc2002284}, {0x38,0x80a34001}, {0x3c,0x0880000a},
    {0x40,0xa0102000}, {0x44,0xc20023c8}, {0x48,0x80886001}, {0x4c,0x02800007}, {0x50,0xa2102000}, {0x54,0x033fc180}, {0x58,0xc0204000}, {0x5c,0x1080001c},
    {0x60,0xc0202514}, {0x64,0xa2102000}, {0x68,0x912c6002}, {0x6c,0xc2022768}, {0x70,0x9b30601c}, {0x74,0x80a36000}, {0x78,0x0280000f}, {0x7c,0xa2046001},

    {0xf0,0xa3},
    {0x00,0xc2002284}, {0x04,0x80a34001}, {0x08,0x1880000b}, {0x0c,0x90022768}, {0x10,0x7fffff7c}, {0x14,0x01000000}, {0x18,0x80a22000}, {0x1c,0x02800007},
    {0x20,0x80a46009}, {0x24,0xa0042001}, {0x28,0x9b2c2002}, {0x2c,0x033fc180}, {0x30,0xd0234001}, {0x34,0x80a46009}, {0x38,0x28bfffed}, {0x3c,0x912c6002},
    {0x40,0x033fc180}, {0x44,0xe0204000}, {0x48,0xe0202514}, {0x4c,0x81c7e008}, {0x50,0x81e80000}, {0x54,0x9de3bf98}, {0x58,0xd0002320}, {0x5c,0x80a22000},
    {0x60,0x0280004b}, {0x64,0x01000000}, {0x68,0xc200231c}, {0x6c,0x80a06000}, {0x70,0x22800016}, {0x74,0xd800231c}, {0x78,0x82063fff}, {0x7c,0x80a06001},

    {0xf0,0xa4},
    {0x00,0x38800012}, {0x04,0xd800231c}, {0x08,0xc2002318}, {0x0c,0x80a06000}, {0x10,0x12800008}, {0x14,0x213fc000}, {0x18,0xa0142020}, {0x1c,0x82102001},
    {0x20,0x7ffff019}, {0x24,0xc2240000}, {0x28,0x10800007}, {0x2c,0xc0240000}, {0x30,0xa0142020}, {0x34,0x7ffff014}, {0x38,0xc0240000}, {0x3c,0x82102001},
    {0x40,0xc2240000}, {0x44,0xd800231c}, {0x48,0x80a0000c}, {0x4c,0x82603fff}, {0x50,0x9a1e2001}, {0x54,0x80a0000d}, {0x58,0x9a603fff}, {0x5c,0x8088400d},
    {0x60,0x0280000d}, {0x64,0x80a0000c}, {0x68,0xc2002318}, {0x6c,0x80a06000}, {0x70,0x12800006}, {0x74,0x033fc000}, {0x78,0x9a102001}, {0x7c,0x82106020},

    {0xf0,0xa5},
    {0x00,0x10800004}, {0x04,0xda204000}, {0x08,0x82106020}, {0x0c,0xc0204000}, {0x10,0x80a0000c}, {0x14,0x82603fff}, {0x18,0x9a1e2002}, {0x1c,0x80a0000d},
    {0x20,0x9a603fff}, {0x24,0x8088400d}, {0x28,0x0280000d}, {0x2c,0x80a62000}, {0x30,0xc2002318}, {0x34,0x80a06000}, {0x38,0x12800005}, {0x3c,0x033fc000},
    {0x40,0x82106020}, {0x44,0x10800005}, {0x48,0xc0204000}, {0x4c,0x9a102001}, {0x50,0x82106020}, {0x54,0xda204000}, {0x58,0x80a62000}, {0x5c,0x1280000c},
    {0x60,0x01000000}, {0x64,0xc2002318}, {0x68,0x80a06000}, {0x6c,0x12800005}, {0x70,0x033fc000}, {0x74,0x82106020}, {0x78,0x10800005}, {0x7c,0xc0204000},

    {0xf0,0xa6},
    {0x00,0x9a102001}, {0x04,0x82106020}, {0x08,0xda204000}, {0x0c,0x81c7e008}, {0x10,0x81e80000}, {0x14,0x9de3bf98}, {0x18,0xc2002514}, {0x1c,0x80a06000},
    {0x20,0x12800007}, {0x24,0x90102001}, {0x28,0xda002568}, {0x2c,0xc2002570}, {0x30,0x80a34001}, {0x34,0x22800006}, {0x38,0xc2002514}, {0x3c,0x82102001},
    {0x40,0x7fffffa5}, {0x44,0xc220250c}, {0x48,0xc2002514}, {0x4c,0x80a06000}, {0x50,0x1280000c}, {0x54,0x01000000}, {0x58,0xc200250c}, {0x5c,0x80a06000},
    {0x60,0x02800008}, {0x64,0x9a007fff}, {0x68,0xb0102002}, {0x6c,0x80a36000}, {0x70,0x12800004}, {0x74,0xda20250c}, {0x78,0x7fffff97}, {0x7c,0x81e80000},

    {0xf0,0xa7},
    {0x00,0x01000000}, {0x04,0x81c7e008}, {0x08,0x81e80000}, {0x0c,0x01000000}, {0x10,0x27001040}, {0x14,0xa614e00f}, {0x18,0xe6a00040}, {0x1c,0x01000000},
    {0x20,0x81c3e008}, {0x24,0x01000000}, {0x28,0x9de3bf98}, {0x2c,0xc2002508}, {0x30,0x80a06000}, {0x34,0x0280000e}, {0x38,0x1b3fc180}, {0x3c,0x82102001},
    {0x40,0x9a13603c}, {0x44,0xc2234000}, {0x48,0xc2002508}, {0x4c,0x80a06000}, {0x50,0x02800005}, {0x54,0x033fc180}, {0x58,0x7fffffed}, {0x5c,0x01000000},
    {0x60,0x30bffffa}, {0x64,0x8210603c}, {0x68,0xc0204000}, {0x6c,0x81c7e008}, {0x70,0x81e80000}, {0x74,0x9de3bf98}, {0x78,0xda002500}, {0x7c,0xc20022d0},

    {0xf0,0xa8},
    {0x00,0x80a34001}, {0x04,0x18800025}, {0x08,0xa4102000}, {0x0c,0xd2002790}, {0x10,0x832ca002}, {0x14,0xe2024001}, {0x18,0x80a46000}, {0x1c,0x12800004},
    {0x20,0xa12ca003}, {0x24,0x10800019}, {0x28,0xc02427dc}, {0x2c,0x92024001}, {0x30,0xc20427dc}, {0x34,0x80a06000}, {0x38,0x02800008}, {0x3c,0x900427dc},
    {0x40,0x7ffffaf8}, {0x44,0x01000000}, {0x48,0xc20022ac}, {0x4c,0x80a20001}, {0x50,0x28800005}, {0x54,0xc20427e0}, {0x58,0xe22427dc}, {0x5c,0x1080000b},
    {0x60,0xc02427e0}, {0x64,0x82006001}, {0x68,0xc22427e0}, {0x6c,0xda002378}, {0x70,0x80a0400d}, {0x74,0x28800006}, {0x78,0xa404a001}, {0x7c,0x7ffff069},

    {0xf0,0xa9},
    {0x00,0x01000000}, {0x04,0x30800005}, {0x08,0xa404a001}, {0x0c,0x80a4a009}, {0x10,0x24bfffe0}, {0x14,0xd2002790}, {0x18,0x81c7e008}, {0x1c,0x81e80000},
    {0x20,0x9de3bf98}, {0x24,0x7ffff54c}, {0x28,0x01000000}, {0x2c,0x7ffff390}, {0x30,0x01000000}, {0x34,0x7ffff3d0}, {0x38,0x01000000}, {0x3c,0x7ffff535},
    {0x40,0x01000000}, {0x44,0x7ffff800}, {0x48,0x01000000}, {0x4c,0x7ffff571}, {0x50,0x01000000}, {0x54,0x7ffff714}, {0x58,0x01000000}, {0x5c,0x7ffff7b9},
    {0x60,0x90102001}, {0x64,0x7ffff93a}, {0x68,0x01000000}, {0x6c,0x7ffffca3}, {0x70,0x01000000}, {0x74,0x7ffff9cf}, {0x78,0x01000000}, {0x7c,0x7ffff963},

    {0xf0,0xaa},
    {0x00,0x01000000}, {0x04,0x7ffffd08}, {0x08,0x90102768}, {0x0c,0x7ffff997}, {0x10,0x01000000}, {0x14,0x7ffffa8b}, {0x18,0x01000000}, {0x1c,0x7ffffb1d},
    {0x20,0x01000000}, {0x24,0x7ffffb8e}, {0x28,0x01000000}, {0x2c,0x7ffffbc8}, {0x30,0x01000000}, {0x34,0x7ffffbe4}, {0x38,0x01000000}, {0x3c,0x7ffffc52},
    {0x40,0x01000000}, {0x44,0x7ffffcf8}, {0x48,0xd0002790}, {0x4c,0xc2002514}, {0x50,0x7ffffd04}, {0x54,0xc2202518}, {0x58,0x7ffffddc}, {0x5c,0x01000000},
    {0x60,0x7ffffe5b}, {0x64,0x01000000}, {0x68,0x7fffffa3}, {0x6c,0x01000000}, {0x70,0x7ffffeef}, {0x74,0x01000000}, {0x78,0x7fffff67}, {0x7c,0x01000000},

    {0xf0,0xab},
    {0x00,0x7fffff8a}, {0x04,0x81e80000}, {0x08,0x01000000}, {0x0c,0x9de3bf98}, {0x10,0xc200253c}, {0x14,0x80a06000}, {0x18,0x12800048}, {0x1c,0xb0102000},
    {0x20,0xd6002460}, {0x24,0x82102080}, {0x28,0x80a2e000}, {0x2c,0x02800043}, {0x30,0xc220256c}, {0x34,0x10800005}, {0x38,0xb0102001}, {0x3c,0xc220256c},
    {0x40,0x1080003e}, {0x44,0xf00e2468}, {0x48,0xd80022fc}, {0x4c,0x80a6000c}, {0x50,0x1880002d}, {0x54,0x9a102000}, {0x58,0xd40022f8}, {0x5c,0x33000018},
    {0x60,0xb6102001}, {0x64,0x80a6c00a}, {0x68,0x18800020}, {0x6c,0xb4102000}, {0x70,0x832e2002}, {0x74,0xb8006038}, {0x78,0xa0166220}, {0x7c,0x901661e8},

    {0xf0,0xac},
    {0x00,0x92166258}, {0x04,0xde0022f8}, {0x08,0xfa070010}, {0x0c,0x80a7400b}, {0x10,0x26800013}, {0x14,0xb606e001}, {0x18,0x80a6e001}, {0x1c,0x22800007},
    {0x20,0xc20022f8}, {0x24,0xc2070008}, {0x28,0x80a74001}, {0x2c,0x2480000c}, {0x30,0xb606e001}, {0x34,0xc20022f8}, {0x38,0x80a6c001}, {0x3c,0x22800007},
    {0x40,0xb406a001}, {0x44,0xc2070009}, {0x48,0x80a74001}, {0x4c,0x26800004}, {0x50,0xb606e001}, {0x54,0xb406a001}, {0x58,0xb606e001}, {0x5c,0x80a6c00f},
    {0x60,0x08bfffea}, {0x64,0xb8072038}, {0x68,0x80a6800d}, {0x6c,0x34800002}, {0x70,0x9a10001a}, {0x74,0xb0062001}, {0x78,0x80a6000c}, {0x7c,0x28bfffda},

    {0xf0,0xad},
    {0x00,0xb6102001}, {0x04,0xb0102000}, {0x08,0xc20e2464}, {0x0c,0x80a06000}, {0x10,0x22800006}, {0x14,0xb0062001}, {0x18,0x80a34001}, {0x1c,0x34bfffc8},
    {0x20,0xc20e2278}, {0x24,0xb0062001}, {0x28,0x80a62003}, {0x2c,0x24bffff8}, {0x30,0xc20e2464}, {0x34,0xb0102000}, {0x38,0x81c7e008}, {0x3c,0x81e80000},
    {0x40,0x9de3bf98}, {0x44,0xc2002574}, {0x48,0x80a06000}, {0x4c,0x02800021}, {0x50,0x90100018}, {0x54,0x82007fff}, {0x58,0x7ffff164}, {0x5c,0xc2202574},
    {0x60,0xc2002574}, {0x64,0x80a06000}, {0x68,0x3280001b}, {0x6c,0xc2002578}, {0x70,0xc200253c}, {0x74,0xda002334}, {0x78,0x8200400d}, {0x7c,0x82006001},

    {0xf0,0xae},
    {0x00,0xc2202548}, {0x04,0xc2002564}, {0x08,0x80a06000}, {0x0c,0x1280000f}, {0x10,0x01000000}, {0x14,0x7ffff1bc}, {0x18,0x01000000}, {0x1c,0x033fc200},
    {0x20,0xda002334}, {0x24,0xd800232c}, {0x28,0x82106074}, {0x2c,0xd8204000}, {0x30,0x96102001}, {0x34,0x9a036001}, {0x38,0xda202574}, {0x3c,0xd6202540},
    {0x40,0x10800004}, {0x44,0xd6202564}, {0x48,0x7ffff16c}, {0x4c,0x01000000}, {0x50,0xc2002578}, {0x54,0x80a06000}, {0x58,0x12800014}, {0x5c,0x01000000},
    {0x60,0xc2002574}, {0x64,0x80a06000}, {0x68,0x12800010}, {0x6c,0x01000000}, {0x70,0x7fffff87}, {0x74,0x01000000}, {0x78,0x80a22000}, {0x7c,0x1280000a},

    {0xf0,0xaf},
    {0x00,0xd020253c}, {0x04,0xc2002334}, {0x08,0x9a102001}, {0x0c,0x82006001}, {0x10,0xc2202574}, {0x14,0xda202578}, {0x18,0xda202540}, {0x1c,0x7ffff709},
    {0x20,0x91e82000}, {0x24,0xd0202574}, {0x28,0x81c7e008}, {0x2c,0x81e80000}, {0x30,0x9de3bf98}, {0x34,0x033fc200}, {0x38,0x82106030}, {0x3c,0xda004000},
    {0x40,0xc200257c}, {0x44,0x80a34001}, {0x48,0x12800017}, {0x4c,0x01000000}, {0x50,0x7ffff01d}, {0x54,0x01000000}, {0x58,0x80a22000}, {0x5c,0x32800008},
    {0x60,0xc2002514}, {0x64,0x7ffff066}, {0x68,0xb0102000}, {0x6c,0x80a22000}, {0x70,0x0280000f}, {0x74,0x01000000}, {0x78,0xc2002514}, {0x7c,0x80a06000},

    {0xf0,0xb0},
    {0x00,0x12800006}, {0x04,0x90102002}, {0x08,0xc200250c}, {0x0c,0x80a06000}, {0x10,0x02800005}, {0x14,0x01000000}, {0x18,0x033fc180}, {0x1c,0x7ffffe6e},
    {0x20,0xc0204000}, {0x24,0x7fffef7f}, {0x28,0xb0102001}, {0x2c,0x81c7e008}, {0x30,0x81e80000}, {0x34,0x9de3bf98}, {0x38,0x7ffffed5}, {0x3c,0x01000000},
    {0x40,0xe0002500}, {0x44,0x80a42015}, {0x48,0x08800016}, {0x4c,0x80a42000}, {0x50,0x7ffff15a}, {0x54,0x01000000}, {0x58,0x033fc140}, {0x5c,0x82106048},
    {0x60,0xda004000}, {0x64,0x03000040}, {0x68,0x11000016}, {0x6c,0x808b4001}, {0x70,0x12800004}, {0x74,0x90122180}, {0x78,0x11000016}, {0x7c,0x901223a8},

    {0xf0,0xb1},
    {0x00,0x7fffff90}, {0x04,0x01000000}, {0x08,0x7fffffca}, {0x0c,0x01000000}, {0x10,0x80a22000}, {0x14,0x2280001d}, {0x18,0xc2002500}, {0x1c,0x3080002f},
    {0x20,0x1280000f}, {0x24,0x80a42014}, {0x28,0x7fffef21}, {0x2c,0x01000000}, {0x30,0x80a22000}, {0x34,0x32800003}, {0x38,0x90102002}, {0x3c,0x90102001},
    {0x40,0x7ffffe45}, {0x44,0x01000000}, {0x48,0x7fffef56}, {0x4c,0x01000000}, {0x50,0x7fffee94}, {0x54,0x01000000}, {0x58,0x30800009}, {0x5c,0x3880000b},
    {0x60,0xc2002500}, {0x64,0x808c2001}, {0x68,0x32800008}, {0x6c,0xc2002500}, {0x70,0x90043ff8}, {0x74,0x7ffff074}, {0x78,0x91322001}, {0x7c,0x7ffff0cf},

    {0xf0,0xb2},
    {0x00,0x01000000}, {0x04,0xc2002500}, {0x08,0x80a40001}, {0x0c,0x3280000d}, {0x10,0xc2002578}, {0x14,0x031fffff}, {0x18,0x821063f0}, {0x1c,0x80a40001},
    {0x20,0x38800003}, {0x24,0x21040000}, {0x28,0xa0042001}, {0x2c,0x033fc180}, {0x30,0x82106034}, {0x34,0xe0204000}, {0x38,0xe0202500}, {0x3c,0xc2002578},
    {0x40,0x80a06000}, {0x44,0x02800005}, {0x48,0x01000000}, {0x4c,0x7ffffed5}, {0x50,0x01000000}, {0x54,0xc0202578}, {0x58,0x81c7e008}, {0x5c,0x81e80000},
    {0x60,0x81c3e008}, {0x64,0x01000000}, {0x68,0x01000000}, {0x6c,0x01000000}, {0x70,0x01000000}, {0x74,0x01000000}, {0x78,0x01000000}, {0x7c,0x01000000},

    {0xf0,0xb3},
    {0x00,0x00001682}, {0x04,0x00000000}, {0x08,0x46656220}, {0x0c,0x20352032}, {0x10,0x30313300}, {0x14,0x00000000}, {0x18,0x31353a34}, {0x1c,0x383a3334},
    {0x20,0x00000000}, {0x24,0x00000000}, {0x28,0x00000000}, {0x2c,0x00000000}, {0x30,0x00000000}, {0x34,0x00000000}, {0x38,0x00000000}, {0x3c,0x00000000},
    {0x40,0x00000000}, {0x44,0x00000000}, {0x48,0x00000000}, {0x4c,0x00000000}, {0x50,0x00000000}, {0x54,0x00000000}, {0x58,0x00000000}, {0x5c,0x00000000},
    {0x60,0x00000000}, {0x64,0x00000000}, {0x68,0x00000000}, {0x6c,0x00000000}, {0x70,0x00000000}, {0x74,0x00000000}, {0x78,0x00000000}, {0x7c,0x00000000},
};

#endif // GSL1680_FW



#endif // RA8875_TOUCH_GSL1680_FIRMWARE_H
