#include "mbed.h"
#include "RawSerial.h"

#include "WiflyInterface.h"

// Used later in hw adaptation
#define MBED_APP_BOARD 1        /* http://mbed.org/components/mbed-Application-Board/ */
#define SMART_BOARD    2        /* http://mbed.org/users/WiredHome/notebook/SmartBoard-baseboard/ */

// ########################################################################
#define HW_ADAPTER SMART_BOARD  /* Which board are we compiling against? */
// ########################################################################


#define SSID "ssid"
#define PASS "pass"

const char PROGINFO[] = "\r\n\r\nWifly Demo - built " __DATE__ " " __TIME__ "\r\n\r\n";

char Message[] = "Hello World";
char BCASTADDR[] = "255.255.255.255";
int Port = 40000;

#if HW_ADAPTER == MBED_APP_BOARD
WiflyInterface eth( p9, p10, p30, p29);
#elif HW_ADAPTER == SMART_BOARD
WiflyInterface eth(p28, p27, p23, p24);
#endif

RawSerial pc(USBTX, USBRX);
DigitalOut myled(LED1);

extern "C" void mbed_reset();

void ShowIPAddress(bool show)
{
    pc.printf("Ethernet connected as %s\r\n", show ? eth.getIPAddress() : "---.---.---.---");
}

void ShowMACAddress(bool show)
{
    pc.printf("Ethernet MAC is %s\r\n", show ? eth.getMACAddress() : "--.--.--.--.--.--");
}


int main() {
    pc.baud(460800);
    pc.printf(PROGINFO);
    pc.printf("Configuration for %s\r\n\r\n", (HW_ADAPTER == SMART_BOARD) ? "SmartBoard" : "mbed Application Board");
    
    eth.SetSecurity(SSID, PASS, WPA);
    pc.printf("***\r\n");
    pc.printf("Initializing network interface...\r\n");
    if (0 == eth.init()) {  //Use DHCP
        eth.setName("TestWiFi_01");
        do {
            pc.printf("Connecting to network...\r\n");
            if (0 == eth.connect()) {
                ShowIPAddress(true);
                ShowMACAddress(true);
                int speed = eth.get_connection_speed();
                pc.printf("Connected at %d Mb/s\r\n", speed);
                pc.printf("Wifly info: %s\r\n", eth.getWiflyVersionString());
                
                pc.printf("\r\n");
                pc.printf("  ctrl-a    to download Wifly firmward v4.41\r\n");
                pc.printf("  ctrl-b    to download Wifly firmward v4.75\r\n");
                pc.printf("  ctrl-c    to send a broadcast message\r\n");
                while (eth.is_connected()) {
                    if (eth.readable())
                        pc.putc(eth.getc());
                    else if (pc.readable()) {
                        int c = pc.getc();
                        if (c == 0x01) {        // ctrl-a           // Install Software v4.41
                            pc.printf("Trying to download wifly7-441.mif from microchip...\r\n");
                            if (eth.SWUpdateWifly("wifly7-441.mif")) {
                                pc.printf("  Success - now rebooting to this Wifly image.\r\n");
                                wait_ms(100);
                                mbed_reset();
                            }
                        } else if (c == 0x02) { // ctrl-b           // Install Software v4.75
                            pc.printf("Trying to download wifly7-475.mif from microchip...\r\n");
                            if (eth.SWUpdateWifly("wifly7-475.mif")){
                                pc.printf("  Success - now rebooting to this Wifly image.\r\n");
                                wait_ms(100);
                                mbed_reset();
                            }
                        } else if (c == 0x03) { // ctrl-c
                            UDPSocket bcast;
                            Endpoint ep;
                            pc.printf("Broadcast sending '%s' on port %d\r\n", Message, Port);
                            int h = ep.set_address(BCASTADDR, Port);
                            int i = bcast.bind(40000);
                            int j = bcast.set_broadcasting(true);
                            int k = bcast.sendTo(ep, Message, strlen(Message));
                            bcast.close();
                            pc.printf("done.\r\n");
                        } else {
                            eth.putc(c);
                        }
                    }
                }
                
                pc.printf("lost connection.\r\n");
                ShowIPAddress(false);
                eth.disconnect();
            }
            else {
                pc.printf("  ... failed to connect.\r\n");
            }
        }
        while (1);
    }
    else {
        pc.printf("  ... failed to initialize, rebooting...\r\n");
        mbed_reset();
    }
}
