//
// Software Update Test Program
// 
// Mostly minimized to what is needed to demonstrate the SWUpdate library:
// * EthernetInterface
// * HTTPClient
// * SWUpdate
//
// Additional items that I do not do without (but are not absolutely necessary):
// * EthernetStatus - to know the state of the connection.
// * Watchdog - because they are always a good idea.
// * mbed-rtos - required by the EthernetInterface (I think)
//
// 
// 
//
#include "mbed.h"               // mbed: ver 95, rtos: 64
#include "EthernetInterface.h"  // ver 47
#include "RawSerial.h"          // for console i/o

// My custom libraries
#include "HTTPClient.h"         // ver 33
#include "SWUpdate.h"           // ver 20
#include "Watchdog.h"           // ver 4
#include "EthStatus.h"          // connection state and speed...

// ===================================================
// SWUpdate
// ===================================================
// Server url to where the Software will be found.
const char *url = "http://192.168.1.201/mbed";

// Base filename for the software (e.g. if you save it as "abcdefg.bin", use "abcdefg" here)
// Remember: some servers are case-sensitive!
const char *name = "SWUp";
// ===================================================



// Banner that is sent to the console on reboot - shows build date/time.
const char * PROG_INFO = "SW Update: " __DATE__ ", " __TIME__;

EthernetInterface eth;
Watchdog wd;

RawSerial pc(USBTX, USBRX);
LocalFileSystem local("local");

HTTPClient http;

extern "C" void mbed_reset();


void SoftwareUpdateCheck(void)
{
    pc.printf("SoftwareUpdateCheck (%s) (%s)\r\n", url, name);
    SWUpdate_T su = SoftwareUpdate(url, name, DEFER_REBOOT);
    if (SWUP_OK == su) {
        pc.printf("  Update installed, rebooting ...\r\n");
        Thread::wait(3000);
        mbed_reset();
    } else if (SWUP_SAME_VER == su) {
        pc.printf("  no update available.\r\n");
    } else {
        pc.printf("  update failed %04X - %s\r\n", (int)su, SoftwareUpdateGetHTTPErrorMsg(su));
    }
}


// I like to see what address it gets, and when it loses the connection.
void ShowIPAddress(bool show)
{
    char buf[16];
    
    if (show)
        sprintf(buf, "%15s", eth.getIPAddress());
    else
        sprintf(buf, "%15s", "---.---.---.---");
    pc.printf("Ethernet connected as %s\r\n", buf);
}


void ShowMenu() {
    pc.printf("Menu:\r\n");
    pc.printf("  r - reset\r\n");
    pc.printf("  s - software update check\r\n");
    pc.printf("\r\n");
}

// Only 2 items in this menu - check for software updates and reset!
void CheckConsoleInput(void)
{
    if (pc.readable()) {
        int c = pc.getc();
        switch (c) {
            case 'r':
                mbed_reset();
                break;
            case 's':
                SoftwareUpdateCheck();
                break;
            default:
                pc.printf("unknown command '%c'\r\n", c);
                ShowMenu();
                break;
        }
    }
}

// Friendlier interface to find out if it is currently connected.
bool NetworkIsConnected(void)
{
    return get_link_status();
}


int main()
{
    pc.baud(460800);        // fast, because I like a snappy terminal!
    pc.printf("\r\n%s\r\n", PROG_INFO);

    if (wd.WatchdogCausedReset()) {
        pc.printf("**** Watchdog Event caused reset ****\r\n");
    }
    wd.Configure(30.0);   // nothing should take more than 30 s (we hope).
    
    pc.printf("***\r\n");
    pc.printf("Initializing network interface...\r\n");
    if (0 == eth.init()) {  //Use DHCP
        do {
            pc.printf("Connecting to network...\r\n");
            if (0 == eth.connect()) {
                ShowIPAddress(true);
                int speed = get_connection_speed();
                pc.printf("Connected at %d Mb/s\r\n", speed);
                ShowMenu();
                while (NetworkIsConnected()) {
                    Thread::wait(500);
                    CheckConsoleInput();
                    wd.Service();
                }
                pc.printf("lost connection.\r\n");
                ShowIPAddress(false);
                eth.disconnect();
            }
            else {
                pc.printf("  ... failed to connect.\r\n");
            }
            CheckConsoleInput();
        }
        while (1);
    }
    else {
        pc.printf("  ... failed to initialize, rebooting...\r\n");
        mbed_reset();
    }
}
