
#include "mbed.h"           // tested with v112
#include "RA8875.h"         // tested with v102


const char * PROG_NAME = "RA8875 Touch Colors";
const char * BUILD_DATE = __DATE__ " " __TIME__;

// Define this for 800x480 panel, undefine for 480x272
#define BIG_SCREEN

// Define this for Cap touch panel, undefine for resistive
#define CAP_TOUCH

#ifdef CAP_TOUCH
RA8875 lcd(p5, p6, p7, p12, NC, p9,p10,p13, "tft"); // SPI:{MOSI,MISO,SCK,/ChipSelect,/reset}, I2C:{SDA,SCL,/IRQ}, name
#else
RA8875 lcd(p5, p6, p7, p12, NC, "tft");             // SPI:{MOSI,MISO,SCK,/ChipSelect,/reset}, name
LocalFileSystem local("local");                     // access to calibration file for resistive touch.
#endif

#ifdef BIG_SCREEN
    #define LCD_W 800
    #define LCD_H 480
    #define LCD_C 8         // color - bits per pixel
    #define DEF_RADIUS 50   // default radius of the fingerprint
    #define BL_NORM 25      // Backlight Normal setting (0 to 255)
#else
    #define LCD_W 480
    #define LCD_H 272
    #define LCD_C 8         // color - bits per pixel
    #define DEF_RADIUS 20   // default radius of the fingerprint
    #define BL_NORM 25      // Backlight Normal setting (0 to 255)
#endif

Serial pc(USBTX, USBRX);

#define min(a,b) ((a<b)?a:b)
#define max(a,b) ((a>b)?a:b)

int GetScreenCapture(void)
{
    char fqfn[50];
    int i = 0;
    
    pc.printf("Screen Capture... ");
    for (i=1; i< 100; i++) {
        snprintf(fqfn, sizeof(fqfn), "/local/Screen%02d.bmp", i);
        FILE * fh = fopen(fqfn, "rb");
        if (!fh) {
            lcd.PrintScreen(0,0,LCD_W,LCD_H,fqfn);
            pc.printf(" as /local/Screen%02d.bmp\r\n", i);
            return i;
        } else {
            fclose(fh);     // close this and try the next
        }
    }
    return 0;
}

const char * Bitstream(uint16_t value)
{
    static char buffer[17];
    char * p = buffer;
    uint16_t mask = 0x8000;
    
    do {
        *p++ = (value & mask) ? '1' : '0';
        mask >>= 1;
    } while (mask);
    *p = '\0';
    return buffer;
}


// Calibrate the resistive touch screen, and store the data on the
// local file system.
//
void CalibrateTS(void)
{
    FILE * fh;
    tpMatrix_t matrix;
    RetCode_t r;
    Timer testperiod;
 
    r = lcd.TouchPanelCalibrate("Calibrate the touch panel", &matrix);
    if (r == noerror) {
        fh = fopen("/local/tpcal.cfg", "wb");
        if (fh) {
            fwrite(&matrix, sizeof(tpMatrix_t), 1, fh);
            fclose(fh);
            printf("  tp cal written.\r\n");
            lcd.cls();
        } else {
            printf("  couldn't open tpcal file.\r\n");
        }
    } else {
        printf("error return: %d\r\n", r);
    }
    lcd.cls();
}

// Try to load a previous resistive touch screen calibration from storage. If it
// doesn't exist, activate the touch screen calibration process.
//
void InitTS(void)
{
    FILE * fh;
    tpMatrix_t matrix;

    fh = fopen("/local/tpcal.cfg", "rb");
    if (fh) {
        fread(&matrix, sizeof(tpMatrix_t), 1, fh);
        fclose(fh);
        lcd.TouchPanelSetMatrix(&matrix);
        printf("  tp cal loaded.\r\n");
    } else {
        CalibrateTS();
    }
}


int main() {
    pc.baud(460800);    // I like a snappy terminal, so crank it up!
    pc.printf("\r\n%s Build %s\r\n", PROG_NAME, BUILD_DATE);
 
    pc.printf("Turning on display\r\n");
    lcd.init(LCD_W,LCD_H,LCD_C);
    lcd.TouchPanelInit();
    lcd.Backlight_u8(BL_NORM);
    
    #ifndef CAP_TOUCH
    InitTS();               // resistive touch calibration
    #endif
    lcd.cls();

    lcd.printf("%s\r\nBuild %s", PROG_NAME, BUILD_DATE);
    
    // +----------------------------------------------------+
    // | Prog Name            [Capture]     (x,y) (xxx,yyy) |
    // | Build Date           [Capture]      rgb (RR,GG,BB) |
    // | +------------------------------------------------+ | y =  50
    // | |                Sample Shown Here               | |
    // | +------------------------------------------------+ | y =  89
    // | Red 0 to 255 ------------------------------------+ | y = 100
    // | |                                                | |
    // | +------------------------------------------------+ | y = 139
    // | Grn 0 to 255 ------------------------------------+ | y = 150
    // | |                                                | |
    // | +------------------------------------------------+ | y = 189
    // | Blu 0 to 255 ------------------------------------+ | y = 200
    // | |                                                | |
    // | +------------------------------------------------+ | y = 239
    // +----------------------------------------------------+ y = 271
    //   10                                             w-10
    rect_t RGBList[] = {
        { 10,100, LCD_W-10,139 },    // R
        { 10,150, LCD_W-10,189 },    // G
        { 10,200, LCD_W-10,239 },    // B
        { 10, 50, LCD_W-10, 89 }     // This for the Sample
    };
    rect_t PrintScreenRect = { LCD_W/2-30, 5, LCD_W/2+30, 40 };
    lcd.fillrect(RGBList[0], Red);
    lcd.fillrect(RGBList[1], Green);
    lcd.fillrect(RGBList[2], Blue);
    lcd.fillrect(PrintScreenRect, Gray);
    lcd.foreground(Blue);
    lcd.background(Gray);
    lcd.puts(LCD_W/2-28, 15, "Capture");
    lcd.background(Black);
    color_t rgb = Black;
    uint8_t rgbVal[3] = { 0, 0, 0 };
    
    for (;;) {
        point_t p;
        
        if (lcd.TouchPanelReadable(&p)) {
            lcd.foreground(Blue);
            lcd.SetTextCursor(LCD_W-72, 0);
            lcd.printf("(%3d,%3d)", p.x, p.y);
            
            if (lcd.Intersect(PrintScreenRect, p)) {
                GetScreenCapture();
            }
            for (int i=0; i<3; i++) {
                if (lcd.Intersect(RGBList[i], p)) {
                    uint8_t mag = (255 * (float)(p.x - RGBList[i].p1.x)) / (float)(RGBList[i].p2.x - RGBList[i].p1.x);
                    rgbVal[i] = mag;
                    lcd.SetTextCursor(LCD_W-80, 16);
                    lcd.foreground(Blue);
                    lcd.printf("(%02X,%02X,%02X)", rgbVal[0], rgbVal[1], rgbVal[2]);
                    rgb = RGB(rgbVal[0], rgbVal[1], rgbVal[2]);
                    pc.printf("RGB: %04X %s\r\n", rgb, Bitstream(rgb));
                    lcd.fillrect(RGBList[3], rgb);
                    break;
                }
            }
        }
    }
}
