/// PUB_RA8875_Touch Example.
///
/// This touch screen example shows how easy it is to use the RA8875 library with
/// either the resistive touch panel, _OR_ the capacitive touch panel 
/// (using the FT5206 controller). When used with the capacitive touch
/// it tracks 5 fingers simultaneously, and only 1 for the resistive panel.
///
/// @note Copyright &copy; 2016 - 2019 by Smartware Computing, all rights reserved.
///     Individuals may use this application for evaluation or non-commercial
///     purposes. Within this restriction, changes may be made to this application
///     as long as this copyright notice is retained. The user shall make
///     clear that their work is a derived work, and not the original.
///     Users of this application and sources accept this application "as is" and
///     shall hold harmless Smartware Computing, for any undesired results while
///     using this application - whether real or imagined.
///
/// @author David Smart, Smartware Computing
//
#include "mbed.h"       // Last Working: v128 (OS2)
#include "RA8875.h"     // Last Working: v190


#ifndef MBED_MAJOR_VERSION
#define MBED_MAJOR_VERSION 2
#define MBED_MINOR_VERSION 0    // These are incorrect
#define MBED_PATCH_VERSION 0    // These are incorrect
#endif

// // // // // // // // // // // // // // // // // // // // // // // // 
// Configuration section
// adjust the following information for the screen size, touch panel technology,
// and port pin assignments.
// // // // // // // // // // // // // // // // // // // // // // // // 

// ############# CRITICAL - Set the display resolution ###############
// Define BIG_SCREEN for 800x480 panel, undefine for 480x272
//#define BIG_SCREEN

// Define this for Cap touch panel, undefine for resistive
//#define CAP_TOUCH

#ifdef CAP_TOUCH
//
// Constructor when using the display that has a FT5206 Touch Controller
//
// LCD[ SPI:{MOSI,MISO,SCK}, /ChipSelect, /reset], 
// Touch[ I2C:{SDA,SCL}, /IRQ], 
// name
//
RA8875 lcd(p5,p6,p7,p12,NC, p9,p10,p13, "tft");

//
// Constructor when using the display that has a GSL1680 Touch Controller
//
// LCD[ SPI:{MOSI,MISO,SCK}, /ChipSelect, /reset], 
// Touch[ I2C:{SDA,SCL}, Wake, /IRQ], 
// name
//
//RA8875 lcd(p5,p6,p7,p12,NC, p9,p10,p14,p13, "tft");
//
#else
//
// Constructor when using the basic display with integrated Resistive Touch Controller
//
// LCD[ SPI:{MOSI,MISO,SCK}, /ChipSelect, /reset], 
// name
//
RA8875 lcd(p5,p6,p7,p12,NC, "tft");

void InitTS(void);                                  // Needed for Resistive Touch
void CalibrateTS(void);                             // Needed for Resistive Touch
const char * TS_Config = "/local/tpcal.cfg";        // Path and file for storing touch config
LocalFileSystem local("local");                     // Needed for resistive touch config storage
#endif

#define PC_BAUD 460800  // I like the serial communications to be very fast

// // // // // // // // // // // // // // // // // // // // // // // // 
// End of Configuration Section
// // // // // // // // // // // // // // // // // // // // // // // // 

Serial pc(USBTX, USBRX);    // Not required for display

#ifdef BIG_SCREEN
    #define LCD_W 800
    #define LCD_H 480
    #define LCD_C 8         // color - bits per pixel
    #define DEF_RADIUS 50   // default radius of the fingerprint
    #define BL_NORM 25      // Backlight Normal setting (0 to 255)
#else
    #define LCD_W 480
    #define LCD_H 272
    #define LCD_C 8         // color - bits per pixel
    #define DEF_RADIUS 20   // default radius of the fingerprint
    #define BL_NORM 25      // Backlight Normal setting (0 to 255)
#endif




// When drawing a "fingerprint" under the touch point - the RA8875 
// cannot clip the drawing at the edge of the screen. Since it cannot
// draw an object partially off-screen, this function shrinks the 
// fingerprint to fit as the touch approaches the edge of the screen.
//
int ComputeRadius(point_t p)
{
    int radius = DEF_RADIUS;
    
    if (p.x < radius)
        radius = p.x;
    else if (LCD_W - p.x < radius)
        radius = LCD_W - p.x;
    if (p.y < radius)
        radius = p.y;
    else if (LCD_H - p.y < radius)
        radius = LCD_H - p.y;
    return radius;
}


// And here is where the fun begins.
int main()
{
    color_t fingerColor[5] = {Blue, Red, Green, Yellow, Magenta};
    
    pc.baud(PC_BAUD);    //I like a snappy terminal, so crank it up!
    pc.printf("\r\nRA8875 Touch Screen Example - Build " __DATE__ " " __TIME__ "\r\n");

    lcd.init(LCD_W,LCD_H,LCD_C,40); // 40 is rather dim, but doesn't overload USB ports so easily
    lcd.TouchPanelInit();
    lcd.foreground(White);          // Change to white since it is starting kinda dim.
    lcd.printf("RA8875 Touch Screen Example - Build " __DATE__ " " __TIME__ "\r\n");
    lcd.printf("MBED v%d.%d.%d\r\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
    
    point_t last[5];        // space for tracking 5 touches
    #ifndef CAP_TOUCH
    InitTS();               // resistive touch calibration
    #endif
    
    // draw on one layer and erase the other for smoother transition while
    // is shows both layers.
    lcd.SetLayerMode(RA8875::BooleanOR);
    int layer = 0;
    
    while (1) {
        TouchCode_t touch;
        
        touch = lcd.TouchPanelReadable();                           // any touch to report?
        if (touch) {
            layer++;
            printf("%d: %2X: ", lcd.TouchCount(), lcd.TouchGesture());  // all printf can be removed
            
            // TouchChannels reports 1 for resistive panel and 5 for capacitive sense
            for (int i = 0; i < lcd.TouchChannels(); i++) {
                uint8_t id = lcd.TouchID(i);                        // 'id' tracks the individual touches
                TouchCode_t ev = lcd.TouchCode(i);                  // 'ev'ent indicates no_touch, touch, held, release, ...
                point_t xy = lcd.TouchCoordinates(i);               // and of course the (x,y) coordinates
                int count = lcd.TouchCount();                       // how many simultaneous touches
                printf("%2d,%d:(%4d,%4d) ", id, ev, xy.x, xy.y);
                if ((id < 5) || (i < count)) {
                    int lastRadius, newRadius;
                    
                    lastRadius = ComputeRadius(last[id]);           // To erase the last fingerprint
                    newRadius = ComputeRadius(xy);                  // Shrink near edge of screen
                    lcd.SelectDrawingLayer(layer & 1);
                    lcd.fillcircle(xy, newRadius, fingerColor[id]); // draw new fingerprint
                    lcd.SelectDrawingLayer((layer+1) & 1);
                    lcd.fillcircle(last[id], lastRadius, Black);    // erase old fingerprint
                    last[id] = xy;
                }
            }
            printf("\r\n");
        }
    }
}

//
//
// For the Reistive Touch Screen, the following are essential.
// For the Capacitive Touch Screen, none of the following is required.
//
//
#ifndef CAP_TOUCH
// Calibrate the resistive touch screen, and store the data on the
// file system.
//
void CalibrateTS(void)
{
    FILE * fh;
    tpMatrix_t matrix;
    RetCode_t r;
    Timer testperiod;
 
    r = lcd.TouchPanelCalibrate("Calibrate the touch panel", &matrix);
    if (r == noerror) {
        fh = fopen(TS_Config, "wb");
        if (fh) {
            fwrite(&matrix, sizeof(tpMatrix_t), 1, fh);
            fclose(fh);
            printf("  tp cal written.\r\n");
            lcd.cls();
        } else {
            printf("  couldn't open tpcal file.\r\n");
        }
    } else {
        printf("error return: %d\r\n", r);
    }
    lcd.cls();
}

// Try to load a previous resistive touch screen calibration from storage. If it
// doesn't exist, activate the touch screen calibration process.
//
void InitTS(void)
{
    FILE * fh;
    tpMatrix_t matrix;

    fh = fopen(TS_Config, "rb");
    if (fh) {
        fread(&matrix, sizeof(tpMatrix_t), 1, fh);
        fclose(fh);
        lcd.TouchPanelSetMatrix(&matrix);
        printf("  tp cal loaded.\r\n");
    } else {
        CalibrateTS();
    }
}
#endif // CAP_TOUCH