/* Tetris game for touchscreen MicroTFT 320x240 and microcontorller LPC1768
 * Uses SeeedStudioTFTv2 library
 * Copyright (c) 2017 Sergejs Popovs    sergun2311
 * 
 * Derivation for RA8875 and to clean up a lot of magic numbers by D.Smart
 *
 */

#include "mbed.h"
#include <ctime>
#include "playGround.h"

LocalFileSystem local("local");
RawSerial pc(USBTX, USBRX);    // Not required for display
Timer tSinceTouch;

// Calibrate the resistive touch screen, and store the data on the
// local file system.
//
void CalibrateTS(void)
{
    FILE * fh;
    tpMatrix_t matrix;
    RetCode_t r;
    Timer testperiod;
 
    r = TFT.TouchPanelCalibrate("Calibrate the touch panel", &matrix);
    if (r == noerror) {
        fh = fopen("/local/tpcal.cfg", "wb");
        if (fh) {
            fwrite(&matrix, sizeof(tpMatrix_t), 1, fh);
            fclose(fh);
            printf("  tp cal written.\r\n");
            TFT.cls();
        } else {
            printf("  couldn't open tpcal file.\r\n");
        }
    } else {
        printf("error return: %d\r\n", r);
    }
    TFT.cls();
}

// Try to load a previous resistive touch screen calibration from storage. If it
// doesn't exist, activate the touch screen calibration process.
//
void InitTS(void)
{
    FILE * fh;
    tpMatrix_t matrix;

    fh = fopen("/local/tpcal.cfg", "rb");
    if (fh) {
        fread(&matrix, sizeof(tpMatrix_t), 1, fh);
        fclose(fh);
        TFT.TouchPanelSetMatrix(&matrix);
        printf("  tp cal loaded.\r\n");
    } else {
        CalibrateTS();
    }
}


int main()
{
    int score = 0;
    int period = SPEED;
    bool bottomedOut;
    clock_t start_s;
    pc.baud(460800);    //I like a snappy terminal, so crank it up!
    pc.printf("\r\nTetris - Build " __DATE__ " " __TIME__ "\r\n");
    int playState = 0;  // 0=init, 1=newBlock, 2=game over
    TFTInit();
    #ifndef CAP_TOUCH
    InitTS();               // resistive touch calibration
    #endif

    while (1) {
        switch (playState) {
            case 0: {    // init
                score = 0;
                period = SPEED;
                ReInitGame();
                playState = 10;
                // break;   // fall thru
            }
            case 10: {
                Block NewBlock;
                bottomedOut = false;
                drawScore(score);
                drawPeriod(period);
                drawNextBlock(NewBlock);
                while( !bottomedOut ) {
                    drawMap();
                    drawBlock(NewBlock);
                    start_s = clock();
                    while( (clock() - start_s) < period ) {
                        point_t p;
                        TouchCode_t panel = TFT.TouchPanelReadable(&p);
                        switch (panel) {
                            case touch:
                                //printf("Touch %4d (%d,%d)\r\n", panel, p.x, p.y);
                                tSinceTouch.start();
                                clrBlock(NewBlock);
                                NewBlock = doGest(NewBlock, p);
                                drawBlock(NewBlock);
                                break;
                            case held:
                                if (tSinceTouch.read_ms() > 300) {
                                    tSinceTouch.reset();
                                    tSinceTouch.start();
                                    clrBlock(NewBlock);
                                    NewBlock = doGest(NewBlock, p);
                                    drawBlock(NewBlock);
                                    //printf("Held  %4d (%d,%d)\r\n", tSinceTouch.read_ms(), p.x,p.y);
                                } else {
                                    p.x = 0; p.y = 0;
                                    //printf("held  %4d (%d,%d)\r", tSinceTouch.read_ms(), p.x,p.y);
                                }
                                break;
                            case release:
                                //printf("\r\nstop - release\r\n");
                                tSinceTouch.stop();
                                break;
                            case no_touch:
                                //printf("stop - no_touch\r\n");
                                tSinceTouch.stop();
                                break;
                            default:
                                //printf("\r\nstop - dunno\r\n");
                                tSinceTouch.stop();
                                break;
                        }
                    }
                    if ( NewBlock.CheckBottom() ) {
                        saveToField(NewBlock);
                        bottomedOut = true;
                    } else {
                        clrBlock(NewBlock);
                        NewBlock.y += 1;
                        drawBlock(NewBlock);
                    }
                }
                score += checkLine();
                if ( score < 3200 )
                    period = SPEED - score / 50;
                clrNextBlock(NewBlock);
                if ( checkGameOver() ) {
                    playState = 20;
                    gameOver(score);
                }
                break;
            }
            case 20: {
                if (ReplayTouched()) {
                    playState = 0;     // restart
                    break;
                }
            }
        }
    }
}