/// Block Move demo
///
/// This demo will simulate a scope. It will place a grid on one layer, and data-points
/// on the other layer. Rather than redrawing the entire grid space to simulate time
/// elapsing, it will BlockMove the data and then add a single new data point.
///
///
/// Place a jpg on the local file system named 'fullscrn.jpg'. The WxH of this jpg
/// should match the display size you have set below.
///

#include "mbed.h"               // Last known good: v148, Most work was with v129
#include "RA8875.h"             // Last known good: v148
#include "ResistiveTouchCal.h"

// These two defines can be enabled, or commented out
#define BIG_SCREEN
#define CAP_TOUCH

#ifdef CAP_TOUCH
RA8875 lcd(p5,p6,p7,p12,NC, p9,p10,p13, "tft"); // MOSI,MISO,SCK,/ChipSelect,/reset, SDA,SCL,/IRQ, name
#else
RA8875 lcd(p5,p6,p7,p12,NC, "tft");             //MOSI, MISO, SCK, /ChipSelect, /reset, name
#endif
LocalFileSystem local("local");                     // access to calibration file for resistive touch.

#define LCD_C 8         // color - bits per pixel in order to have 2 layers
#define BL_NORM 25      // Backlight Normal setting (0 to 255)

#ifdef BIG_SCREEN
    #define LCD_W 800
    #define LCD_H 480
#else
    #define LCD_W 480
    #define LCD_H 272
#endif

Serial pc(USBTX, USBRX);

void DrawGrid(rect_t r, int hDiv, int vDiv, color_t color)
{
    point_t p1, p2;
    int i;
    
    lcd.rect(r, color);
    p1.y = r.p1.y; p2.y = r.p2.y;
    for (i=0; i<hDiv; i++) {
        p1.x = p2.x = i * (r.p2.x - r.p1.x) / hDiv + r.p1.x;
        lcd.line(p1, p2, color);
    }
    p1.x = r.p1.x; p2.x = r.p2.x;
    for (i=0; i<vDiv; i++) {
        p1.y = p2.y = i * (r.p2.y - r.p1.y) / vDiv + r.p1.y;
        lcd.line(p1, p2, color);
    }
}

int main()
{
    point_t src;
    point_t dst;
    dim_t w, h;
        
    pc.baud(460800);    //I like a snappy terminal, so crank it up!
    pc.printf("\r\nRA8875 BTE Move Test - Build " __DATE__ " " __TIME__ "\r\n");
    lcd.init(LCD_W,LCD_H,LCD_C, BL_NORM);
    lcd.TouchPanelInit();
    #ifndef CAP_TOUCH
    InitTS(lcd);   // Calibration for resistive touch panel
    #endif
    rect_t grid = {50,50,LCD_W-50,LCD_H-50};
    lcd.SelectDrawingLayer(0);
    DrawGrid(grid,10,10,Green);

    w = grid.p2.x - grid.p1.x;
    h = grid.p2.y - grid.p1.y;

    lcd.SetLayerMode(RA8875::BooleanOR);
    lcd.SelectDrawingLayer(1);
    RetCode_t r = lcd.RenderImageFile(0,0,"/local/fullscrn.jpg");
    if (r) pc.printf("  Error: %d; %s\r\n", r, lcd.GetErrorMessage(r));
    float y;
    float t;
    while (1) {
        //wait_us(100);
        dst = grid.p1;
        src.x = dst.x + 1;
        src.y = dst.y;
        lcd.BlockMove(1,0,dst, 1,0,src, w+2,h, 0x2, 0xC);

        y = sin(t) * (grid.p2.y - grid.p1.y)/2;
        t += 0.1f;
        lcd.pixel(grid.p2.x, y + grid.p1.y + (grid.p2.y - grid.p1.y)/2, White);

        volatile TouchCode_t tc = lcd.TouchCode();
        while (tc == touch || tc == held) {
        #if 1       // This fails: on the first touch, it never exits the while
            tc = lcd.TouchCode();   // returns 0:no_touch, 1:touch, 2:held, 3:release
            wait_us(0);   // Adding this makes it work, even if the wait is 0 or 1
        #elif 0     // This works: but includes the useless committment of a digital output
            tc = lcd.TouchCode();
            DigitalOut led(LED1);
            led = !led;
        #else       // This works: but includes the undesired printf
            tc = lcd.TouchCode();   // returns 0:no_touch, 1:touch, 2:held, 3:release
            pc.printf("tc: %d\r", tc);
        #endif
        }
    }
}
