// game of life implementation inspired by this forum thread
// http://mbed.org/forum/helloworld/topic/4822/
// 
#include "mbed.h"       // v122
#include "RA8875.h"     // v126
#include "LifeRules.h"  // v3

// These two defines can be enabled, or commented out
#define BIG_SCREEN
#define CAP_TOUCH
#define LCD_C 16         // color - bits per pixel

#ifdef CAP_TOUCH
RA8875 lcd(p5,p6,p7,p12,NC, p9,p10,p13, "tft"); // MOSI,MISO,SCK,/ChipSelect,/reset, SDA,SCL,/IRQ, name
#else
RA8875 lcd(p5,p6,p7,p12,NC, "tft");             //MOSI, MISO, SCK, /ChipSelect, /reset, name
LocalFileSystem local("local");                     // access to calibration file for resistive touch.
#endif

#ifdef BIG_SCREEN
    #define LCD_W 800
    #define LCD_H 480
#else
    #define LCD_W 480
    #define LCD_H 272
#endif


// Define the life-map size
#define LIFE_W 150
#define LIFE_H 100
#define LIFE_C 2          /* 1 = monochrome, 2 = color */
#define LIFE_Z 2          /* Zoom factor */

// Try to check if there is enough memory (for LPC1768)
#if LIFE_W * LIFE_H * LIFE_C * 2 / 8 > 0x8000
#error "Sorry, but there isn't this much memory on an LPC1768"
#endif

extern "C" void mbed_reset();

//#define DEBUG "main"
// ...
// INFO("Stuff to show %d", var); // new-line is automatically appended
//
#if (defined(DEBUG) && !defined(TARGET_LPC11U24))
#define INFO(x, ...) std::printf("[INF %s %3d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define WARN(x, ...) std::printf("[WRN %s %3d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define ERR(x, ...)  std::printf("[ERR %s %3d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#else
#define INFO(x, ...)
#define WARN(x, ...)
#define ERR(x, ...)
#endif


#if LIFE_C == 2
#define LIFE_CLR Life::color
#else
#define LIFE_CLR Life::monochrome
#endif

Life life(LIFE_W, LIFE_H, LIFE_CLR);

// Where on screen do we locate it?
#if LIFE_W * LIFE_Z < LCD_W
#define H_OFFSET -1
#else
#define H_OFFSET 0
#endif

#if LIFE_H * LIFE_Z < LCD_H
#define V_OFFSET -1
#else
#define V_OFFSET 0
#endif

#define LIFE_OFFSET_X (LCD_W - (LIFE_W * LIFE_Z) + H_OFFSET)
#define LIFE_OFFSET_Y (LCD_H - (LIFE_H * LIFE_Z) + V_OFFSET)

unsigned char imgbuffer[3*LIFE_W]; // from forum thread...
FILE *img;

Serial pc(USBTX,USBRX); // for debugging

int msDelay = 1000;     //delay between frames

void CheckForUserInteraction(void);
void genrand(); //random start
void genglidergun(); //glider gun
void genglider(); //glider
void genship1(); //light weight ship
void genBMP(); //from image
void gentest();
void genBlinker();

void ScreenUpdate()
{
    lcd.SetWindow(LIFE_OFFSET_X, LIFE_OFFSET_Y, LIFE_W * LIFE_Z, LIFE_H * LIFE_Z);
    lcd.SetGraphicsCursor(LIFE_OFFSET_X+1, LIFE_OFFSET_Y+1);
    pc.printf("window(%d,%d, %d,%d)\r\n", LIFE_OFFSET_X, LIFE_OFFSET_Y, LIFE_W * LIFE_Z, LIFE_H * LIFE_Z);
    lcd._StartGraphicsStream();
    for (int j = 0; j < LIFE_H; j++) {
        for (int Zx = 0; Zx < LIFE_Z; Zx++) {
            for (int i = 0; i < LIFE_W; i++) {
                Life::ValueOfLife lifeState = life.getbit(i,j);
                color_t p;
                switch (lifeState) {
                    case Life::dead:
                        p = RGB(0,0,0);
                        break;
                    case Life::dying:
                        p = RGB(192,0,0);
                        break;
                    case Life::living:
                        p = RGB(64,255,64);
                        break;
                    case Life::birthing:
                        p = RGB(0,0,192);
                        break;
                    default:
                        p = RGB(192,192,0); // Should never see this one...
                        ERR(" lifeState = %d\r\n", lifeState);
                        break;
                }
                for (int Zy = 0; Zy < LIFE_Z; Zy++) {
                    lcd._putp(p);
                }
            }
        }
    }
    lcd._EndGraphicsStream();
    lcd.SetWindow();
}

int GetScreenCapture(void)
{
    char fqfn[50];
    int i = 0;
    
    pc.printf("Screen Capture... ");
    for (i=1; i< 100; i++) {
        snprintf(fqfn, sizeof(fqfn), "/local/Screen%02d.bmp", i);
        FILE * fh = fopen(fqfn, "rb");
        if (!fh) {
            lcd.PrintScreen(0,0,LCD_W,LCD_H,fqfn);
            pc.printf(" as /local/Screen%02d.bmp\r\n", i);
            return i;
        } else {
            fclose(fh);     // close this and try the next
        }
    }
    return 0;
}


int main()
{
    pc.baud(460800);    // I like a snappy terminal, so crank it up!
    pc.printf("\r\nConway's Game of Life - Build " __DATE__ " " __TIME__ "\r\n");

    lcd.init(LCD_W, LCD_H, LCD_C);
    lcd.Backlight(0.5f);

    //lcd.frequency(5000000);
    lcd.puts("Welcome to Conway's Game of Life\r\n\r\n");
    INFO("Destroy all life");
    life.DestroyAllLife();
    INFO("Life destroyed");

    lcd.foreground(RGB(255,255,255));
    
    //pc.printf("choice value = %d\n\r",choice); //for debugging
    lcd.puts("Please Select Starting Configuration:\r\n");
    lcd.puts(" 0 = Glider     1 = Random\r\n"
             " 2 = Ship       3 = Glider Gun\r\n"
             " 4 = BMP image  5 = Blinker\r\n"
             " t = test       \r\n"
             " r = reset\r\n"
             );
    
    // Frame the life map
    lcd.rect(LIFE_OFFSET_X-1,LIFE_OFFSET_Y-1,LIFE_OFFSET_X+LIFE_W*LIFE_Z,LIFE_OFFSET_Y+LIFE_H*LIFE_Z,Red);
    lcd.foreground(Blue);
    int choice = pc.getc();

    switch ( choice ) {
        case '0':
            lcd.puts(">Glider!\r\n");
            genglider();
            break;
        case '1':
            lcd.puts(">Random!\r\n");
            genrand();
            break;
        case '2':
            lcd.puts(">Ship!\r\n");
            genship1();
            break;
        case '3':
            lcd.puts(">Glider Gun!\r\n");
            genglidergun();
            break;
        case '4':
            lcd.puts(">BMP image!\r\n");
            genBMP();
            break;
        case '5':
            lcd.puts(">Blinker!\r\n");
            genBlinker();
            break;
        case 't':
            lcd.puts(">test!\r\n");
            gentest();
            break;
    }
    lcd.puts("\r\nRuntime Menu:\r\n"
             " p   = PrintScreen\r\n"
             " +   = faster\r\n"
             " -   = slower\r\n"
             " 0-9 = updates/sec\r\n"
             );

    ScreenUpdate();
    wait(1);
    
    while(1) {
        CheckForUserInteraction();
        if (msDelay >= 0) {
            static uint16_t toggle = 0;
            
            if ((++toggle & 1) == 0) {
                life.GenerationStep();
            } else {
                life.UpdateLifeCycle();
            }
            ScreenUpdate();
            wait_ms(msDelay);
        }
    }
}

void CheckForUserInteraction(void)
{
    while (pc.readable()) {
        int c = pc.getc();
        if (c == '+' && msDelay >= 0)
            msDelay -= 10;
        else if (c == '-' && msDelay < 1000)
            msDelay += 10;
        else if (c >= '0' && c <= '9') {
            if (c == '0')
                msDelay = -1;
            else
                msDelay = 10 * ((1000 / (c - '0'))/10);
        } else if (c == 'r')
            mbed_reset();
        else if (c == 'p')
            GetScreenCapture();
        lcd.locate(49, 0);
        if (msDelay < 0) {
            msDelay = -10;
            lcd.printf("Paused      ");
        } else {
            lcd.printf("Delay %4d", msDelay);
        }
    }
}


void genBlinker()
{
    life.setbit(1,1, Life::living);
    life.setbit(1,2, Life::living);
    life.setbit(1,3, Life::living);
}

void gentest()
{
    // Point
    life.setbit(6,3, Life::living);
    // Block
    life.setbit(1,1, Life::living);
    life.setbit(1,2, Life::living);
    life.setbit(2,1, Life::living);
    life.setbit(2,2, Life::living);
    // Beehive
    life.setbit(6,1, Life::living);
    life.setbit(7,1, Life::living);
    life.setbit(5,2, Life::living);
    life.setbit(8,2, Life::living);
    life.setbit(6,3, Life::living);
    life.setbit(7,3, Life::living);
    // Blinker
    life.setbit(11,2, Life::living);
    life.setbit(12,2, Life::living);
    life.setbit(13,2, Life::living);
    // Glider
    //  x
    //   x
    // xxx
    life.setbit(2,10, Life::living);
    life.setbit(3,11, Life::living);
    life.setbit(1,12, Life::living);
    life.setbit(2,12, Life::living);
    life.setbit(3,12, Life::living);    
}


void genrand()
{
    for (int i = 0; i < LIFE_W; i++) { //loop through each cell
        for (int j = 0; j < LIFE_H; j++) {
            if (rand() & 1) { //50% chance
                life.setbit(i,j, Life::living);
            }
        }
    }
}

void genBMP()
{
    for (int i = 0; i < LIFE_W; i++) {
        img = fopen("/local/TESTIM~1.BMP", "rb");
        fseek (img , 54 + (LIFE_W*i*3) , SEEK_SET);
        fread(imgbuffer, (LIFE_W*3), 1, img);
        fclose(img);
        for (int j = 0; j < LIFE_H; j++) {
            int red = imgbuffer[j*3];
            if (red == 0) {
                life.setbit(j, LIFE_H - 1 - i, Life::living);
            }
        }
    }
}


void genglider()   //set certain pixels
{
    life.setbit(0,0, Life::living);
    life.setbit(1,1, Life::living);
    life.setbit(1,2, Life::living);
    life.setbit(2,0, Life::living);
    life.setbit(2,1, Life::living);    
}

void genglidergun()   //set certain pixels
{
    life.setbit(0,7, Life::living); //gun
    life.setbit(0,8, Life::living);
    life.setbit(1,7, Life::living);
    life.setbit(1,8, Life::living);
    life.setbit(8,8, Life::living);
    life.setbit(8,9, Life::living);
    life.setbit(9,7, Life::living);
    life.setbit(9,9, Life::living);
    life.setbit(10,7, Life::living);
    life.setbit(10,8, Life::living);
    life.setbit(16,9, Life::living);
    life.setbit(16,10, Life::living);
    life.setbit(16,11, Life::living);
    life.setbit(17,9, Life::living);
    life.setbit(18,10, Life::living);
    life.setbit(22,6, Life::living);
    life.setbit(22,7, Life::living);
    life.setbit(23,5, Life::living);
    life.setbit(23,7, Life::living);
    life.setbit(24,5, Life::living);
    life.setbit(24,6, Life::living);
    life.setbit(24,17, Life::living);
    life.setbit(24,18, Life::living);
    life.setbit(25,17, Life::living);
    life.setbit(25,19, Life::living);
    life.setbit(26,17, Life::living);
    life.setbit(34,5, Life::living);
    life.setbit(34,6, Life::living);
    life.setbit(35,5, Life::living);
    life.setbit(35,6, Life::living);
    life.setbit(35,12, Life::living);
    life.setbit(35,13, Life::living);
    life.setbit(35,14, Life::living);
    life.setbit(36,12, Life::living);
    life.setbit(37,13, Life::living);

    life.setbit(50,38, Life::living); //eater
    life.setbit(51,38, Life::living);
    life.setbit(50,39, Life::living);
    life.setbit(52,39, Life::living);
    life.setbit(52,40, Life::living);
    life.setbit(52,41, Life::living);
    life.setbit(53,41, Life::living);
}

void genship1()   //set certain pixels
{
    life.setbit(10,10, Life::living);
    life.setbit(13,10, Life::living);
    life.setbit(14,11, Life::living);
    life.setbit(10,12, Life::living);
    life.setbit(14,12, Life::living);
    life.setbit(11,13, Life::living);
    life.setbit(12,13, Life::living);
    life.setbit(13,13, Life::living);
    life.setbit(14,13, Life::living);
}



