/* mbed Microcontroller Library
 * Copyright (c) 2006-2012 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#include "mbed.h"

#include "ffconf.h"
#include "mbed_debug.h"

#include "FATFileSystem.h"
#include "FATFileHandle.h"
#include "FATDirHandle.h"

// Define the maximum length for a filename. 
//
// Can accept the _MAX_LFN from ffconf, but this could be big (e.g. 255).
//     Generally this only inflates the stack frame for these functions.
//     Can also set a smaller number if you know the maximum is less.
//     example: "0:/12345678.abc\0" would be 16.
//     keep in mind that something 
//        like "/usb/12345678.abc" is translated to "0:/12345678.abc"
//
//     Various APIs will return the failure code if you try to pass too long
//     of a path/name in.
//
#define _MAX_FN_LEN _MAX_LFN

//#define DEBUG "FtFS"
// ...
// INFO("Stuff to show %d", var); // new-line is automatically appended
//
#if (defined(DEBUG) && !defined(TARGET_LPC11U24))
#include "mbed.h"
#define INFO(x, ...) std::printf("[INF %s %4d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define WARN(x, ...) std::printf("[WRN %s %4d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define ERR(x, ...)  std::printf("[ERR %s %4d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
static void HexDump(const char * title, const uint8_t * p, int count)
{
    int i;
    char buf[100] = "0000: ";

    if (*title)
        INFO("%s", title);
    for (i=0; i<count; ) {
        sprintf(buf + strlen(buf), "%02X ", *(p+i));
        if ((++i & 0x0F) == 0x00) {
            INFO("%s", buf);
            if (i < count)
                sprintf(buf, "%04X: ", i);
            else
                buf[0] = '\0';
        }
    }
    if (strlen(buf))
        INFO("%s", buf);
}
#else
#define INFO(x, ...)
#define WARN(x, ...)
#define ERR(x, ...)
#define HexDump(a, b, c)
#endif

DWORD get_fattime(void) {
    time_t rawtime;
    time(&rawtime);
    struct tm *ptm = localtime(&rawtime);
    return (DWORD)(ptm->tm_year - 80) << 25
         | (DWORD)(ptm->tm_mon + 1  ) << 21
         | (DWORD)(ptm->tm_mday     ) << 16
         | (DWORD)(ptm->tm_hour     ) << 11
         | (DWORD)(ptm->tm_min      ) << 5
         | (DWORD)(ptm->tm_sec/2    );
}

FATFileSystem *FATFileSystem::_ffs[_VOLUMES] = {0};

FATFileSystem::FATFileSystem(const char* n) : FileSystemLike(n) {
    INFO("FATFileSystem(%s)", n);
    for(int i=0; i<_VOLUMES; i++) {
        if(_ffs[i] == this)     // 2nd attempt to mount the same drive
            return;
        if(_ffs[i] == 0) {
            _ffs[i] = this;
            _fsid[0] = '0' + i;
            _fsid[1] = ':';
            _fsid[2] = '\0';
            INFO("Mounting [%s] on ffs drive [%s]", _name, _fsid);
            f_mount(&_fs, _fsid, 0);
            for (int d=0; d<_VOLUMES; d++) {
                INFO("  _ffs[%d] = %p", d, _ffs[d]);
            }
            return;
        }
    }
    error("Couldn't create %s in FATFileSystem::FATFileSystem\r\n", n);
}

FATFileSystem::~FATFileSystem() {
    for (int i=0; i<_VOLUMES; i++) {
        if (_ffs[i] == this) {
            _ffs[i] = 0;
            f_mount(NULL, _fsid, 0);
        }
    }
}

FileHandle *FATFileSystem::open(const char* name, int flags) {
    INFO("open(%s) on filesystem [%s], drv [%s]", name, _name, _fsid);
    char n[_MAX_FN_LEN];
    int sn = snprintf(n, _MAX_FN_LEN, "%s/%s", _fsid, name);
    if (sn < 0)
        return NULL;
    INFO("  :: open(%s)", n);
    
    /* POSIX flags -> FatFS open mode */
    BYTE openmode;
    if (flags & O_RDWR) {
        openmode = FA_READ|FA_WRITE;
    } else if(flags & O_WRONLY) {
        openmode = FA_WRITE;
    } else {
        openmode = FA_READ;
    }
    if(flags & O_CREAT) {
        if(flags & O_TRUNC) {
            openmode |= FA_CREATE_ALWAYS;
        } else {
            openmode |= FA_OPEN_ALWAYS;
        }
    }

    FIL fh;
    FRESULT res = f_open(&fh, n, openmode);
    if (res) {
        INFO("f_open('w') failed: %d", res);
        return NULL;
    }
    if (flags & O_APPEND) {
        f_lseek(&fh, fh.fsize);
    }
    return new FATFileHandle(fh);
}

int FATFileSystem::remove(const char *name) {
    INFO("remove(%s) on filesystem [%s], drv [%s]", name, _name, _fsid);
    char n[_MAX_FN_LEN];
    int sn = snprintf(n, _MAX_FN_LEN, "%s/%s", _fsid, name);
    if (sn < 0)
        return -1;

    INFO("  :: remove(%s)", n);
    FRESULT res = f_unlink(n);
    if (res) {
        INFO("f_unlink() failed: %d", res);
        return -1;
    }
    return 0;
}

int FATFileSystem::rename(const char *oldname, const char *newname) {
    char o[_MAX_FN_LEN], n[_MAX_FN_LEN];
    int sn = snprintf(o, _MAX_FN_LEN, "%s/%s", _fsid, oldname);
    if (sn < 0)
        return -1;
    sn = snprintf(n, _MAX_FN_LEN, "%s/%s", _fsid, newname);
    if (sn < 0)
        return -1;
    FRESULT res = f_rename(o, n);
    if (res) {
        INFO("f_rename() failed: %d", res);
        return -res;
    }
    return 0;
}

int FATFileSystem::format() {
    FRESULT res = f_mkfs(_fsid, 0, 512); // Logical drive number, Partitioning rule, Allocation unit size (bytes per cluster)
    if (res) {
        INFO("f_mkfs() failed: %d", res);
        return -1;
    }
    return 0;
}

DirHandle *FATFileSystem::opendir(const char *name) {
    FATFS_DIR dir;
    char n[_MAX_FN_LEN];
    int sn = snprintf(n, _MAX_FN_LEN, "%s/%s", _fsid, name);
    if (sn < 0)
        return NULL;
    FRESULT res = f_opendir(&dir, n);
    if (res != 0) {
        return NULL;
    }
    return new FATDirHandle(dir);
}

int FATFileSystem::mkdir(const char *name, mode_t mode) {
    char n[_MAX_FN_LEN];
    int sn = snprintf(n, _MAX_FN_LEN, "%s/%s", _fsid, name);
    if (sn < 0)
        return -1;
    FRESULT res = f_mkdir(n);
    return res == 0 ? 0 : -1;
}

int FATFileSystem::mount() {
    FRESULT res = f_mount(&_fs, _fsid, 1);
    return res == 0 ? 0 : -1;
}

int FATFileSystem::unmount() {
    if (disk_sync())
        return -1;
    FRESULT res = f_mount(NULL, _fsid, 0);
    return res == 0 ? 0 : -1;
}

#if 0
// I think this cannot work unless the FileSystemLike and other interfaces
// are updated.
int FATFileSystem::fstat(const char *path, FILINFO *fileinfo) {
    char n[_MAX_FN_LEN];
    int sn = snprintf(n, _MAX_FN_LEN, "%s/%s", _fsid, path);
    if (sn < 0)
        return -1;
    FRESULT res = f_stat(n, fileinfo);
    printf("f_stat(%s) returned: %d\r\n", n, res);
    return res == 0 ? 0 : -res;
}
#endif