/* mbed Microcontroller Library
 * Copyright (c) 2006-2012 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#include "ff.h"
#include "ffconf.h"
#include "mbed_debug.h"

#include "FATFileHandle.h"

//#define DEBUG "FtFH"
// ...
// INFO("Stuff to show %d", var); // new-line is automatically appended
//
#if (defined(DEBUG) && !defined(TARGET_LPC11U24))
#include "mbed.h"
#define INFO(x, ...) std::printf("[INF %s %4d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define WARN(x, ...) std::printf("[WRN %s %4d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
#define ERR(x, ...)  std::printf("[ERR %s %4d] "x"\r\n", DEBUG, __LINE__, ##__VA_ARGS__);
static void HexDump(const char * title, const uint8_t * p, int count)
{
    int i;
    char buf[100] = "0000: ";

    if (*title)
        INFO("%s", title);
    for (i=0; i<count; ) {
        sprintf(buf + strlen(buf), "%02X ", *(p+i));
        if ((++i & 0x0F) == 0x00) {
            INFO("%s", buf);
            if (i < count)
                sprintf(buf, "%04X: ", i);
            else
                buf[0] = '\0';
        }
    }
    if (strlen(buf))
        INFO("%s", buf);
}
#else
#define INFO(x, ...)
#define WARN(x, ...)
#define ERR(x, ...)
#define HexDump(a, b, c)
#endif

FATFileHandle::FATFileHandle(FIL fh) {
    _fh = fh;
}

int FATFileHandle::close() {
    int retval = f_close(&_fh);
    delete this;
    return retval;
}

ssize_t FATFileHandle::write(const void* buffer, size_t length) {
    UINT n;
    FRESULT res = f_write(&_fh, buffer, length, &n);
    if (res) {
        INFO("f_write() failed: %d", res);
        return -1;
    }
    return n;
}

ssize_t FATFileHandle::read(void* buffer, size_t length) {
    INFO("read(%d)", length);
    UINT n;
    FRESULT res = f_read(&_fh, buffer, length, &n);
    if (res) {
        INFO("f_read() failed: %d", res);
        return -1;
    }
    return n;
}

int FATFileHandle::isatty() {
    return 0;
}

off_t FATFileHandle::lseek(off_t position, int whence) {
    if (whence == SEEK_END) {
        position += _fh.fsize;
    } else if(whence==SEEK_CUR) {
        position += _fh.fptr;
    }
    FRESULT res = f_lseek(&_fh, position);
    if (res) {
        INFO("lseek failed: %d", res);
        return -1;
    } else {
        INFO("lseek OK, returning %i", _fh.fptr);
        return _fh.fptr;
    }
}

int FATFileHandle::fsync() {
    FRESULT res = f_sync(&_fh);
    if (res) {
        INFO("f_sync() failed: %d", res);
        return -1;
    }
    return 0;
}

off_t FATFileHandle::flen() {
    return _fh.fsize;
}

