/* mbed Microcontroller Library - MemFileSystem
 * Copyright (c) 2008, sford
 */


#ifndef MBED_MEMFILESYSTEM_H
#define MBED_MEMFILESYSTEM_H

#include "FATFileSystem.h"

// 4 Bytes / Sector, each 512 Bytes - malloc'd as required
// Value was 2000 (8K)
#define NUM_SECTORS 100
#define SECTOR_SIZE 512

// moved the zeroed RAM buffer to flash, 
// so it doesn't waste precious RAM
static const char zero[SECTOR_SIZE] = {0};

namespace mbed
{
    class MemFileSystem : public FATFileSystem
    {
    public:
        // NUM_SECTORS sectors, each 512 bytes (malloced as required)
        char *sectors[NUM_SECTORS];
    
        MemFileSystem(const char* name) : FATFileSystem(name) {
            memset(sectors, 0, sizeof(sectors));
        }

        virtual ~MemFileSystem() {
            for(int i = 0; i < NUM_SECTORS; i++) {
                if(sectors[i]) {
                    free(sectors[i]);
                }
            }
        }

        // read a sector in to the buffer, return 0 if ok
        virtual int disk_read(char *buffer, int sector) {
            if(sectors[sector] == 0) {
                // nothing allocated means sector is empty
                memset(buffer, 0, SECTOR_SIZE);
            } else {
                memcpy(buffer, sectors[sector], SECTOR_SIZE);
            }
            return 0;
        }

        // write a sector from the buffer, return 0 if ok
        virtual int disk_write(const char *buffer, int sector) {
            // if buffer is zero deallocate sector
            //char zero[SECTOR_SIZE] = {0};     // DS this used to be a ram buffer
            //memset(zero, 0, SECTOR_SIZE);
            if(memcmp(zero, buffer, SECTOR_SIZE)==0) {
                if(sectors[sector] != 0) {
                    free(sectors[sector]);
                    sectors[sector] = 0;
                }
                return 0;
            }
            // else allocate a sector if needed, and write
            if(sectors[sector] == 0) {
                char *sec = (char*)malloc(SECTOR_SIZE);
                if(sec==0) {
                    return 1; // out of memory
                }
                sectors[sector] = sec;
            }
            memcpy(sectors[sector], buffer, SECTOR_SIZE);
            return 0;
        }

        // return the number of sectors
        virtual int disk_sectors() {
            return sizeof(sectors)/sizeof(sectors[0]);
        }
    };
}

#endif