/// @file CANMessage.h 
/// CANMessage is the primitive CAN message object. It supports creation,
/// parsing, formatting, of messages.
///
/// @version 1.0
///
/// @note Copyright &copr; 2011 by Smartware Computing, all rights reserved.
///     Individuals may use this application for evaluation or non-commercial
///     purposes. Within this restriction, changes may be made to this application
///     as long as this copyright notice is retained. The user shall make
///     clear that their work is a derived work, and not the original.
///     Users of this application and sources accept this application "as is" and
///     shall hold harmless Smartware Computing, for any undesired results while
///     using this application - whether real or imagined.
///
/// @author David Smart, Smartware Computing
///

#ifndef CANMESSAGE_H
#define CANMESSAGE_H
#include "mbed.h"

/// CAN direction, in the set {rcv, xmt}
typedef enum {
    rcv,
    xmt
} CANDIR_T;


/// CAN Channel in the set {CH1, CH2} of CANCHANNELS
typedef enum {
    CH1,
    CH2,
    CANCHANNELS
} CANCHANNEL_T;


// CAN mode - active(acknowledge) or passive(monitor)
//typedef enum {
//    ACTIVE,
//    PASSIVE
//} CANMODE_T;


/// A super CAN message object, which has additional capabilities
///
/// This object is derived from CANMessage, however it adds
/// a timestamp and other useful methods.
///
/// the CANMessage format element by default is:
///     - CANStandard = 0,
///     - CANExtended = 1,
///     - CANAny = 2
/// But for certain purposes it will be convenient to use the CANmsg object,
/// which inherits CANMessage, and overload this item with additional values.
///     - Error = 128
///
class CANmsg : public CANMessage
{
public:
    /// Constructor for a CANmsg object, which is a superclass of CANMessage
    ///
    /// The CANmsg object included additional information, including
    /// direction, channel number, timestamp, and the ability to format
    /// into ASCII
    ///
    CANmsg();
    
    /// Constructor for a CANmsg object, which is a superclass of CANMessage
    ///
    /// The CANmsg object included additional information, including
    /// direction, channel number, timestamp, and the ability to format
    /// into ASCII.
    /// @param ch is the channel number (CH1, CH2)
    /// @param dir is the direction (xmt, rcv)
    /// @param msg is a CANMessage object from which to construct this object
    ///
    CANmsg(CANCHANNEL_T ch, CANDIR_T dir, CANMessage msg);
    
    /// Constructor for a CANmsg object, which is a superclass of CANMessage
    /// 
    /// The CANmsg object included additional information, including
    /// direction, channel number, timestamp, and the ability to format
    /// into ASCII.
    /// This constructor creates a message from an ascii buffer with
    /// the standard message format in it.
    ///
    /// @todo Improve the parser, it may not handle malformed input well
    ///
    /// @verbatim
    /// t xtd 02 1CF00400 08 11 22 33 44 55 66 77 88 0   0 1234.567890
    /// If the first letter is not 't' or 'r', transmit is assumed
    ///   xtd 02 1CF00400 08 11 22 33 44 55 66 77 88 0   0 1234.567890
    /// @endverbatim
    ///
    /// @param p is a pointer to a text string of the message contents,
    ///        in the same format as what would be output with the 
    ///        FormatCANMessage method.
    ///
    CANmsg(char *p);
        
    /// Destructor for the CANmsg object
    ~CANmsg();
    
    /// Parse a text string into the CANmsg object, which is a superclass of CANMessage
    ///
    /// The CANmsg object included additional information, including
    /// direction, channel number, timestamp, and the ability to format
    /// into ASCII.
    /// This constructor creates a message from an ascii buffer with
    /// the standard message format in it.
    ///
    /// @todo Improve the parser, it may not handle malformed input well
    ///
    /// @verbatim
    /// t xtd 02 1CF00400 08 11 22 33 44 55 66 77 88 0   0 1234.567890
    /// If the first letter is not 't' or 'r', transmit is assumed
    ///   xtd 02 1CF00400 08 11 22 33 44 55 66 77 88 0   0 1234.567890
    /// @endverbatim
    ///
    /// @param p is a pointer to a text string of the message contents,
    ///        in the same format as what would be output with the 
    ///        FormatCANMessage method.
    /// @returns true
    ///
    bool ParseCANMessage(char *p);

    /// Formats this CAN message into a text buffer - which should be at least 70
    ///
    /// @todo buflen is currently ignored, which is unsafe
    ///
    /// @verbatim
    /// t xtd 02 1CF00400 08 11 22 33 44 55 66 77 88 0   0 1234.567890
    /// @endverbatim
    ///
    /// @param buffer is a pointer to the buffer to fill
    /// @param buflen is the size of the buffer (minimum 70 recommended)
    /// 
    /// @returns nothing
    void FormatCANMessage(char *buffer, int buflen);
    
    /// Overrides the timestamp in this message with the current time
    void SetTimestamp();
    
    /// Gets the 64-bit timestamp of this message
    /// @returns time in microseconds
    ///
    uint64_t GetTimestamp() { return timestamp; };
    
    /// direction of this CAN message - rcv or xmt
    CANDIR_T dir;

    /// channel number of this CAN message
    CANCHANNEL_T ch;

private:
    /// internally held timestamp
    uint64_t timestamp;
    
    // internally held mode
    //CANMODE_T mode;
};

#endif // CANMESSAGE_H