/*
     PulseCounter 
     interrrupt driven counter of edges (UP or DOWN)
     with min en max width
         
     Copyright (c) 2010 Wim De Roeve
 
     Permission is hereby granted, free of charge, to any person obtaining a copy
     of this software and associated documentation files (the "Software"), to deal
     in the Software without restriction, including without limitation the rights
     to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
     copies of the Software, and to permit persons to whom the Software is
     furnished to do so, subject to the following conditions:
 
     The above copyright notice and this permission notice shall be included in
     all copies or substantial portions of the Software.
 
     THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
     IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
     FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
     AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
     LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
     OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
     THE SOFTWARE.
*/

#ifndef PULSECOUNTER_H
#define PULSECOUNTER_H

#include "mbed.h"

extern "C" DigitalOut led3;
extern "C" DigitalOut led4;

enum PULSMODE { PULSE_DOWN=false, PULSE_UP = true};

class PulseCounter {
public:
    PulseCounter(PinName pin, PinMode pull, int step,  uint32_t minwidth, uint32_t maxwidth,
                 bool up, char c):
            _interrupt(pin)
    { 

        if (up) {   // pulse up
            _interrupt.rise(this, &PulseCounter::startpulse); // first rise then fall
            _interrupt.fall(this, &PulseCounter::endpulse);

        } else {   // pulse down
            _interrupt.fall(this, &PulseCounter::startpulse); // first fall then rise
            _interrupt.rise(this, &PulseCounter::endpulse);
        }

        _interrupt.mode(pull);
      
        _step=step;
        _count=0;
        _countall=0;
        _minwidth =minwidth;
        _maxwidth =maxwidth;
      
        _useled3= (c=='E');
        _f=false;
        _t.stop();
        _t.reset();
        _up=up;

    }

    void startpulse() {

        _diff_start=_t.read_ms();

        _t.reset();
        _t.start();
        
        if (_useled3)
            led3=1;
        else
            led4=1;

        _f=true;  // pulse is on
    }

    void endpulse() {
        _diff_stop=_t.read_ms();
        _t.stop();
        
        if (_useled3)
            led3=0;
        else
            led4=0;

       
        if ((_diff_stop >= _minwidth) and (_diff_stop <= _maxwidth) and (_f)) {

            _count=_count+_step;
        }
        _countall=_countall+_step;
        _f=false;  // pulse is off
    }


    int read() {
        return _count;
    }

    int readall() {
        return _countall;
    }


    void set(int value) {
        _count=value;
    }

    void setall(int value) {
        _countall=value;
    }

private:
    InterruptIn _interrupt;

    volatile int _count;
    volatile int _countall;
    volatile int _step;

    Timer _t;

    float _minwidth;
    float _maxwidth;
    bool _f;
    bool _up;
    bool _useled3;
    uint32_t _diff_start;
    uint32_t _diff_stop;
    
};
#endif
