/* mbed PCD8544MS - Graphic Library for driving multi monochrome displays based on PCD8544
 *  used in Nokia 3310, 3315, 3330, 3350, 3410, 3210,  5110, 5120, 5130, 5160, 6110, 6150
 *
 * Copyright (c) 2011, Wim De Roeve
 * partial port of the code found on http://serdisplib.sourceforge.net/ser/pcd8544.html#links
 * and by Petras Saduikis <petras@petras.co.uk>
 *
 * it uses a PCF8574 / PCF8575 for control line handling (/CS RESET and DC)
 * and SPI for data handling
 * up to 6 screens
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef PCD8544MS_H
#define PCD8544MS_H

//  the Nokia 3310 has a resolution of 84 x 48
//  the Nokia 3410 has a resolution of 102 x 72 officialy 84x64

#include "mbed.h"

#define LCD_X_RES  102   //84  
#define LCD_Y_RES   72    //48
#define LCD_BANKS (LCD_Y_RES / 8)

#define LCD_CACHE_SIZE ((LCD_X_RES * LCD_Y_RES) / 8)

#define MAX_ADR_X  (LCD_X_RES)-1
#define MAX_ADR_Y  LCD_BANKS-1
//#define MAXSCREENS 1

typedef uint8_t BYTE;

class PCD8544MS {

    /* PCD8544 from Philips Semiconductors is
        48 x 84 pixels monochrome matrix LCD controller/driver

        The PCD8544 has a 504 byte memory with NO read function.
        Each bit is a pixel
        You can only write 1 byte at a time in vertical or horizontal mode.
        There is no read functionality with the controller.
        Caching a copy of the LCD-memory is the only solution to set one pixel at a time.


     * wiring SPI
     *  mosi // yellow
     *  miso NC
     *  sclk //green

     * PCF8575/PCF8574  to  PCD8544 LCD
     * ----------------------
     * P0              -  CS1    //gray
     * P1              -  DC     //blue
     * P2              -  RESET  //white
     * P3              -  CS2
     * P4              -  CS3
     * P5              -  CS4
     * P6              -  CS5
     * P7              -  CS6
     *
     */
    // yellow    green      //gray //blue //white
    // PCD8544MS lcd(p11,NC,p13,p15,p16,p17);  //mosi miso sclk cs dc reset


public:

    /** LCD panel format */
    typedef enum {
        PIXEL_OFF = 0,
        PIXEL_ON = 1,
        PIXEL_XOR = 2
    }ePixelMode;

    typedef enum {
        FILL_OFF = 0,
        FILL_ON = 1
    }eFillMode;

    typedef enum {
        RASTER_OFF = 0,
        RASTER_ON = 1
    }eRasterMode;

    typedef enum {
        DRAW_OVERWRITE = 0,
        DRAW_MERGE = 1
    }eDrawMode;


    typedef enum {
        VERYSMALLFONT = 0, //3x5
        TINYFONT =      1, //5x7
        SMALLFONT =     2, //6x8
        NORMALFONT =    3, //8x8
        BIGFONT    =    4, //8x12&#65533;
        TIMENUMBERFONT= 5, //16x20
        BIGNUMBERFONT=  6
    }eFonts;

    typedef enum {
        C_POINT          = 0, //point
        C_LINE    =      1, //line
        C_VLINE       =     2, //Vertical Line
        C_HLINE       =    3, //Horizontal Line
    }eChartMode;

    typedef enum {
        SPACE_NONE = 0,
        SPACE_NORMAL = 1
    }eSpaceMode;

    enum eDisplayMode {DISPLAY_NORMAL, DISPLAY_HIGHLIGHT};

    enum LCDType {
        LCD3310,
        LCD3410,
        LCD6100,   /**< Nokia 6100, as found on sparkfun board (default) */
        LCD6610,   /**< Nokia 6610, as found on olimex board */
        PCF8833
    };


    PCD8544MS(PinName mosi, PinName miso, PinName sclk, PinName sda, PinName scl,int i2cAddress,bool TextLCD, bool backlight);

    /** init()
     *
     * Initialise the device.
     * @param PinName SPI mosi
     * @param PinName SPI miso
     * @param PinName SPI sclk
     * @param PinName sda
     * @param PinName sck
     * @param PinName i2cAddress
     * @param int maxscreens      // number of screens used
    */

#if 0 // Inhereted from Stream, for documentation only
    /* Function: putc
     *  Write a character
     *
     * Variables:
     *  c - The character to write to the serial port
     */
    int putc(int c);

    /* Function: printf
     *  Write a formated string
     *
     * Variables:
     *  format - A printf-style format string, followed by the
     *      variables to use in formating the string.
     */
    int printf(const char* format, ...);
#endif


  virtual void TLCD_locate(int column, int row);

    /* Function: cls
     * Clear the screen, and locate to 0,0
     */
    virtual void TLCD_cls();

      /* Function: backlight
    * Sets the backlight on or off
    *
    * Variables:
    *  on (true or false)
    */
   
   
     virtual void TLCD_backlight(bool on);


    /** cls()
     *  clears the cached copy of the screen
     *  and the screen itself
    */
    void GLCD_cls(int screen,bool fupdate=true);

    /** update()
     *  copies the cached memory to the screen
     *  use this to update the screen after
     *  - drawBitmap
    */
    void GLCD_update(int screen);

    /** close()
     *  screen display OFF
    */
    void GLCD_close();

    /** locate(x,y)
     *  sets the cursor on position x,y
    */
    void GLCD_locate         (BYTE x0, BYTE y0);

    void chooseFont(eFonts font);
    void GLCD_writeString    (BYTE x0, BYTE y0, char* string,  eFonts font,ePixelMode pmode,eDisplayMode dmode,eSpaceMode smode, BYTE fupdate,int screen );
    void GLCD_writeChar      (BYTE x0, BYTE y0, BYTE ch,  eFonts font,ePixelMode pmode,eDisplayMode mode, BYTE fupdate,int screen);

    /** drawBitmap(x,y,bitmap,xsize,ysize)
     *  draw a monochrome bitmap on position x,y
     *  with size xsize,ysize
    */
    void GLCD_drawBitmap     (BYTE x0, BYTE y0, const unsigned char* bitmap, BYTE bmpXSize, BYTE bmpYSize, BYTE fupdate,int screen );

    void GLCD_drawpixel      (BYTE x0, BYTE y0, ePixelMode pmode, BYTE fupdate,int screen );
    void GLCD_drawline       (BYTE x0, BYTE y0, BYTE x1,BYTE y1, ePixelMode pmode, BYTE fupdate,int screen );
    void GLCD_drawcircle     (BYTE x0, BYTE y0, BYTE radius, eFillMode fill,ePixelMode pmode, BYTE fupdate,int screen );
    void GLCD_drawrectangle  (BYTE x0, BYTE y0, BYTE x1,BYTE y1, eFillMode fill, ePixelMode pmode, BYTE fupdate,int screen );
    void GLCD_drawprogressbar(BYTE x0, BYTE y0, BYTE w, BYTE h, BYTE percentage, BYTE fupdate,int screen );
    void GLCD_drawchart      (BYTE  x0, BYTE y0, BYTE w, BYTE h, BYTE unitx, BYTE unity,
                         eRasterMode rMode, eChartMode cMode, eDrawMode dMode,int16_t * val,  int size, int t,int screen );

private:

    SPI _spi;
    I2C _i2c;
    int _i2cAddress;
    //  LCDType _type;

    // int _lcd_x_res,_lcd_y_res,_lcd_cache_size;


    void TLCD_reset();
    void TLCD_writeData(int data);
    void TLCD_writeCommand(int command);
    void TLCD_writeByte(int value, bool rs);
    void TLCD_writeNibble(int value, bool rs);

    void GLCD_reset();
    void GLCD_writeCmd(BYTE data, BYTE CF,int screen);
    void GLCD_writeData(BYTE data, BYTE CF,int screen);
    void GLCD_writeI2C(bool reset, bool dc, bool cs,int screen );
    
    void writeI2CByte(int data);
    int readI2C();
    
    virtual int _putc(int c);
    virtual int _getc();
    virtual void TLCD_newline();

    int _row;
    int _column;
    int _columns;
    int _rows;

    bool _backlight;

    //BYTE LcdCache[][_lcd_cache_size]; // __attribute__((section("AHBSRAM0")));
    BYTE LcdCache[LCD_CACHE_SIZE]; // __attribute__((section("AHBSRAM0")));

    int LcdCacheIdx;
    int _LoMark;
    int _HiMark;
    int Scale;

    BYTE _font_width;
    BYTE _font_height;
    BYTE _font_start;
    BYTE _font_end;
    BYTE _font_bytes;
    unsigned char* _pFont;



};

#endif