/*
* OneWireEEPROM. Library for Maxim One-Wire EEPROM.
*
* see http://www.maxim-ic.com
*
* DS2433
* DS28EC20
*
* Copyright (C) <2011> Wim De Roeve <wim312@gmail.com>
*
* OneWireEEPROM is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* OneWireEEPROM is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* If not, see <http://www.gnu.org/licenses/>.
*/

#include "mbed.h"
#include "OneWireEEPROM.h"

Serial output(USBTX, USBRX);

OneWireEEPROM::OneWireEEPROM(PinName pin, bool crcOn, bool useAddr, bool parasitic, DSTYPE ds, unsigned char *ROMaddress)
        :oneWire(pin, STANDARD) {
    _useCRC=crcOn;
    _useAddress=useAddr;
    _useParasiticPower=parasitic;
    switch (ds) {
        case DS2433: {
            _pages = DS2433PAGES;
            _eeprom_id =DS2433EEPROM_ID;
            break;
        }
        case DS28EC20: {
            _pages =DS28EC20PAGES;
            _eeprom_id =DS28EC20EEPROM_ID;
            break;
        }

    }
    _memsize = _pages * PAGESIZE;

    if (_useAddress) {
        for (int i = 0; i < ADDRESS_SIZE; i++)
            _ROMCode[i]=ROMaddress[i];
    } else {
        for (int i = 0; i < ADDRESS_SIZE; i++)
            _ROMCode[i]=0;
    }

    active=0;
}


bool OneWireEEPROM::Initialize() {

    int OneWireFound;
    int OneWireSameAddress;
    int i;
    BYTE _dummyaddress[8];


    if (_useAddress) {
        output.printf("\r\nScan for device with address ");
        for (i = 0; i < ADDRESS_SIZE; i++) {
            output.printf("%x ", (int)_ROMCode[i]);
        }
        output.printf("\r\n");
    }

    OneWireSameAddress=0;

    oneWire.resetSearch();
    do {
        OneWireFound=(oneWire.search(_dummyaddress));
        if (OneWireFound) {

            if (!_useAddress) {
                output.printf("Device found with Address = ");
                for (i = 0; i < ADDRESS_SIZE; i++) {
                    output.printf("%x ", (int)_dummyaddress[i]);
                }
            }

            OneWireSameAddress=1;

            if (_useAddress) {
                for (i = 0; i < ADDRESS_SIZE; i++) {
                    if (!((OneWireSameAddress) && (_ROMCode[i] ==_dummyaddress[i])))
                        OneWireSameAddress=0;
                }
            } else {
                for (i = 0; i < ADDRESS_SIZE; i++) {
                    _ROMCode[i] =_dummyaddress[i];
                }
            }


            /*if (OneWireSameAddress) {
                output.printf("-> Address valid!\r\n");

            } else {

            }*/

        } else {

            output.printf("No more addresses.\r\n");
            oneWire.resetSearch();
            wait_ms(250);  //500
        }
    } while (OneWireFound && !OneWireSameAddress);

    if (!OneWireSameAddress) {
        output.printf("-> No Valid ROM Code found.\r\n");
        return false;
    }
    if (OneWireCRC::crc8(_ROMCode, ADDRESS_CRC_BYTE) != _ROMCode[ADDRESS_CRC_BYTE]) { // check address CRC is valid
        output.printf("CRC is not valid!\r\n");
        wait_ms(100);

        return false;
    }

    if (_ROMCode[0] !=  _eeprom_id) {
        output.printf("Device is not a OneWireEEPROM_ID device.\r\n");
        wait_ms(100);
        return false;

    } else {
        output.printf("OneWireEEPROM present and correct.\r\n");
    }
    active=1;
    return true;
}

void OneWireEEPROM::ResetAndAddress() {
    oneWire.reset();                // reset device
    if (_useAddress) {
        oneWire.matchROM(_ROMCode);  // select which device to talk to
    } else {
        oneWire.skipROM();          // broadcast
    }
}

bool OneWireEEPROM::WriteMemory(uint8_t* Source, uint16_t Address, uint8_t Size) {
    uint8_t _ES; //store Endif Address , datastatus
    uint8_t B;

    if (Address<_memsize) {
        uint8_t _TA1=(uint8_t)(Address & 0x00FF);
        uint8_t _TA2=(uint8_t)((Address & 0xFF00)>>8);

        if ((Size<=PAGESIZE) and ((Size+Address)<=_memsize)) {

            //  output.printf ("\r\nWriting to OneWireEEPROM %i Bytes",Size);

            ResetAndAddress();
            oneWire.writeByte(WRITESCRATCHPAD);
            oneWire.writeByte(_TA1); //begin address T7..T0
            oneWire.writeByte(_TA2); //begin address T15..T8

            //write _memPage to scratchpad

            for (int i = 0; i < Size; i++) {
                oneWire.writeByte(Source[i]);
                //     output.printf ("%X ",Source[i]);
            }
            // output.printf ("\r\nTA1=%X",_TA1);
            //   output.printf ("\r\nTA2=%X\r\n",_TA2);

            //read and check data in scratchpad
            ResetAndAddress();
            oneWire.writeByte(READSCRATCHPAD);
            B=oneWire.readByte();
            if (B != _TA1) { //check TA1, return if bad
                output.printf("\r\nWrite error in TA1 %X - %X\r\n",B,_TA1);
                return false;
            }
            B=oneWire.readByte();
            if (B != _TA2) { //check TA2, return if bad
                output.printf("\r\nWrite error in TA2 %X - %X\r\n",B,_TA2);
                return false;
            }
            _ES = oneWire.readByte(); // ES Register

            //check data written
            for (int i = 0; i < Size; i++) {
                B=oneWire.readByte();
                if (B != Source[i]) { //return if bad
                    output.printf("\r\nWrite error in scratchpad on %i %X<->%X\r\n",i,B ,Source[i]);
                    return false;
                }
            }

            //    output.printf("\r\nES=%X\r\n",_ES);

            //issue copy with auth data
            wait_ms(10);
            ResetAndAddress();
            oneWire.writeByte(COPYSCRATCHPAD);
            oneWire.writeByte(_TA1);
            oneWire.writeByte(_TA2);
            oneWire.writeByte(_ES); //pull-up!
            wait_ms(10); //10ms min strong pullup delay -> time needed to copy from scratchpad to memory

            oneWire.reset();

            //     output.printf ("\r\nData written\r\n");
            return true;
        } else {
            output.printf ("\r\nTrying to write more then %i bytes-> %i\r\n",PAGESIZE,Size);
            return false;
        }
    } else {
        output.printf ("\r\nAddress %X not available, EEPROM isn't that bigg..-> %i\r\n",Address);
        return false;
    }
}


int OneWireEEPROM::ReadMemory(uint8_t* Destination, uint16_t Address, uint16_t Size) {
    uint8_t tmpReader;
    //  bool readFF = 0;
    int memPtr;

    if (Address<_memsize) {

        uint8_t _TA1=(uint8_t)(Address & 0x00FF);
        uint8_t _TA2=(uint8_t)((Address & 0xFF00)>>8);

        if ((Size+Address)<=_memsize) {

            ResetAndAddress();
            oneWire.writeByte(READMEMORY);
            oneWire.writeByte(_TA1);
            oneWire.writeByte(_TA2);

            //       output.printf ("\r\nReading... TA1=%X TA2=%X",_TA1, _TA2);

            for (memPtr = 0; memPtr < Size; memPtr++) {
                tmpReader = oneWire.readByte();
                //   if (tmpReader == 0xff & !readFF)
                //       readFF = 1;
                //   else if (tmpReader == 0xff & readFF)
                // 0xff read twice, hopefully EoF as we break here :)
                //       break;

                Destination[memPtr] = tmpReader;
            }
            //    output.printf ("-> %i byte(s)\r\n",memPtr);
            return memPtr;

        }  else {
            output.printf ("\r\nTrying to read outside MEMORY -> %i\r\n",Address+Size);
            return 0;
        }
    } else {
        output.printf ("\r\nAddress %X not available, EEPROM isn't that bigg..-> %i\r\n",Address);
        return 0;
    }
}

void OneWireEEPROM::ShowMemory(int PageFrom, int PageTo) {
    int Size;
    int MemSize;

    if (PageFrom<0)
        PageFrom=0;
    if (PageTo>=_pages)
        PageTo=_pages-1;

    if ((PageFrom>=0) and (PageFrom<_pages) and
            (PageTo>=0) and (PageTo<_pages)) {

        MemSize=(PageTo-PageFrom+1)*PAGESIZE;

        uint8_t *MemAll = (uint8_t*) malloc(MemSize);

        if (MemAll!=NULL) {
            output.printf ("\r\nRead Page(s) from EEPROM %i to %i -> ", PageFrom,PageTo);

            Size=ReadMemory(MemAll,PageFrom*PAGESIZE,MemSize);

            output.printf("%i bytes\r\n ",Size);

            for (int j=PageFrom;j<=PageTo;j++) {
                output.printf ("\r\nPage %2i ->",j);
                for (int i=0;i<PAGESIZE;i++) {
                    if ((j*32+i)<= Size)
                        output.printf("%2X ",MemAll[j*32+i]);
                }
                if ((j*32)>Size)
                    break;
            }
            output.printf ("\r\n");

            free(MemAll);
        } else
            output.printf ("\r\nNOT enough memory to display EEPROM content !!\r\n");
    }
}

bool OneWireEEPROM::WriteWord(uint16_t v, uint16_t Address) {
    bool ok=false;
    int it=0;
    uint8_t Mem[2];
    Mem[0]=(uint8_t)(v & 0x00FF);
    Mem[1]=(uint8_t)((v & 0xFF00)>>8);

    do {
        ok=WriteMemory(Mem,Address,2);
        it++;
    } while ((ok==false) && (it<5));
    if (it>1) {
        output.printf (" EEPROM WriteWord %i times ->",it);
        if (ok)
            output.printf("ok\r\n");
        else
            output.printf("not ok\r\n");
    }
    return ok;
}

bool OneWireEEPROM::WriteByte(uint8_t b, uint16_t Address) {
    bool ok=false;
    int it=0;
    uint8_t Mem[1];
    Mem[0]=(uint8_t)(b & 0x00FF);
    do {
        ok=WriteMemory(Mem,Address,1);
        it++;
    } while ((ok==false) && (it<5));
    if (it>1) {
        output.printf (" EEPROM WriteByte %i times ->",it);
        if (ok)
            output.printf("ok\r\n");
        else
            output.printf("not ok\r\n");
    }
    return ok;
}

uint16_t OneWireEEPROM::ReadWord(uint16_t Address) {
    int size;
    int it=0;

    uint8_t Mem[2];
    do {
        size=ReadMemory(Mem,Address,2);
        it++;
    } while ((size==0) && (it<5));
//   output.printf ("%i %i \r\n",Mem[0],Mem[1]);
    if (it>1) {
        output.printf (" EEPROM ReadWord %i times ->",it);
        if (size>0)
            output.printf("ok\r\n");
        else
            output.printf("not ok\r\n");
    }
    return (uint16_t)Mem[0] + (uint16_t) (Mem[1]<<8);
}

uint8_t OneWireEEPROM::ReadByte(uint8_t Address) {
    int size;
    int it=0;
    uint8_t Mem[1];
    do {
        size=ReadMemory(Mem,Address,1);
        it++;
    } while ((size==0) && (it<5));
    //  output.printf ("%i \r\n",Mem[0]);
    if (it>1) {
        output.printf (" EEPROM ReadByte %i times ->",it);
        if (size>0)
            output.printf("ok\r\n");
        else
            output.printf("not ok\r\n");
    }
    return (uint8_t) Mem[0];
}