/* mbed LS020 Library, for driving the LCD display LS020 from SHARP used in
 * GSM S65 Siemens
 *
 * Copyright (c) 2010, Wim De Roeve, thanks to Christian Kranz research
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#ifndef MBED_LS020LCD_H
#define MBED_LS020LCD_H

#include "mbed.h"

namespace mbed {
/* Class: LS020LCD
     *  An abstraction of the LS020 Mobile phone screen, used in GSm Siemens S65
     *
     * Example:
     * >
     * > #include "mbed.h"
     * > #include "LS020LCD.h"
     * >
     * >LSO20LCD S65lcd(p5,p6,p7,p8,p9);
     * >
     * > int main() {
     * >     S65lcd.fillcircle(10,10,10,0xE0);
     * > }
*/

class LS020LCD {
public:
    /* Constructor: LSO20LCD
           *  Create and object for the LS020 LCD, using SPI and three DigitalOuts
           *
           * Variables:
           *  mosi - SPI data out
           *  miso - SPI data in, not used
           *  clk  - SPI clock
           *  cs   - Chip Select
           *  rst  - reset
           *  rs   - register select
    */
    LS020LCD(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName rst, PinName rs);

    virtual void orientation(bool rotate, bool mirror);
    virtual void reset();
    virtual void set_8bit_mode(char BGR);
    virtual void set_16bit_mode(void);
    virtual void set_8_color_mode(void);
    virtual void set_65k_color_mode(void);

    void fillrectangle(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1, unsigned int color);
    void drawpixel(unsigned int x, unsigned int y, unsigned int color);
    void drawline(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1, unsigned int color);
    void drawrectangle(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1, unsigned int color);
    void fillcircle(unsigned int x0, unsigned int y0, unsigned int radius, unsigned int color);
    void drawcircle(unsigned int x0, unsigned int y0, unsigned int radius, unsigned int color);
    unsigned int putc(unsigned int x, unsigned int y, unsigned int c, unsigned int size, unsigned int font, unsigned int color, unsigned int bgcolor);
    void drawtext(unsigned int x, unsigned int y, char* text, unsigned int size,unsigned int font, unsigned int color, unsigned int bgcolor);
    void scroll(char offset);
    void cls();
    void rectangle8(char x1, char y1, char x2, char y2, char color);
    void putpixel(unsigned char r,unsigned char g,unsigned char b, unsigned char x, unsigned char y);
    void put_char8(char x, char y, char symbol, char color, char bkcolor);
    void put_string8(char x, char y, char* text, char color, char bkcolor);
    void draw_table(void);
    
private:
    SPI _spi;
    DigitalOut _rst;
    DigitalOut _cs;
    DigitalOut _rs;

    int _row,_column,_rows,_columns,_width, _height;
    bool _rotate,_mirror ;
    int _font,_foreground, _background;

    void write_cmdRG(uint8_t reg, uint8_t param);
    void write_cmd8(uint8_t cmd8);
    void write_cmd16(uint16_t cmd16);
    void write_data8(char data);
    void write_data16(uint16_t cmd16);
    void draw(uint16_t cmd16) ;
    void drawstop(void);
    void drawstart(void);

    void foreground(unsigned int color);
    void background(unsigned int color);

    void locate(int column, int row);
    void newline();
    int columns();
    int rows();
    void set_cursor(unsigned int x, unsigned int y);
    void set_window(char x0, char y0, char x1,char y1);

};
}

#endif
